import ggml
import ggml.utils

import numpy as np


def test_utils():
    params = ggml.ggml_init_params(mem_size=16 * 1024 * 1024)
    ctx = ggml.ggml_init(params=params)
    assert ctx is not None
    x = np.ones((3,), dtype=np.float32)
    assert x.shape == (3,)
    t = ggml.utils.from_numpy(x, ctx)
    assert t.contents.ne[:1] == [3]
    assert t.contents.type == ggml.GGML_TYPE_F32
    assert np.allclose(ggml.utils.to_numpy(t), x)
    ggml.ggml_free(ctx)


def test_numpy_arrays():
    params = ggml.ggml_init_params(mem_size=16 * 1024 * 1024)
    ctx = ggml.ggml_init(params=params)
    assert ctx is not None
    x = np.array([[1, 2, 3], [4, 5, 6]], dtype=np.float32, order="F")
    assert x.shape == (2, 3)
    t = ggml.utils.from_numpy(x, ctx)
    assert t.contents.ne[:2] == [3, 2]
    y = ggml.utils.to_numpy(t)
    assert y.shape == (2, 3)
    ggml.ggml_free(ctx)


def test_numpy_arrays_transposed():
    params = ggml.ggml_init_params(mem_size=16 * 1024 * 1024)
    ctx = ggml.ggml_init(params=params)
    assert ctx is not None
    # 2D
    x = np.array([[1, 2, 3], [4, 5, 6]], dtype=np.int32)
    t = ggml.utils.from_numpy(x, ctx)
    t_t = ggml.ggml_transpose(ctx, t)
    x_t = ggml.utils.to_numpy(t_t)
    assert np.array_equal(x_t, x.T)

    t = ggml.utils.from_numpy(x.T, ctx)
    x_t = ggml.utils.to_numpy(t)
    assert np.array_equal(x.T, x_t)

    # 3D
    x = np.array(
        [[[1, 2], [3, 4], [5, 6]], [[7, 8], [9, 10], [11, 12]]], dtype=np.int32
    )
    t = ggml.utils.from_numpy(x, ctx)
    t_t = ggml.ggml_permute(ctx, t, 2, 1, 0, 3)
    x_t = ggml.utils.to_numpy(t_t)
    assert np.array_equal(x_t, x.T)

    t = ggml.utils.from_numpy(x.T, ctx)
    x_t = ggml.utils.to_numpy(t)
    assert np.array_equal(x.T, x_t)
    ggml.ggml_free(ctx)


def test_slice_tensor():
    params = ggml.ggml_init_params(mem_size=16 * 1024 * 1024)
    ctx = ggml.ggml_init(params=params)
    assert ctx is not None
    x = np.array([[1, 2, 3], [4, 5, 6]], dtype=np.int32)
    t = ggml.utils.from_numpy(x, ctx)
    t_slice = ggml.utils.slice_tensor(ctx, t, [
        slice(0, 2),
        slice(0, 1)
    ])
    x_slice = ggml.utils.to_numpy(t_slice)
    assert np.array_equal(x_slice, x[:1, :2].squeeze())
    ggml.ggml_free(ctx)
