"""OpenAI provider implementation.

This module contains only OpenAI-specific API calls and token counting.
Provider-agnostic orchestration/prompt logic lives in `_llm.py`.
"""

from __future__ import annotations

from openai import OpenAI
from openai.types.responses import Response
from os import environ
from tiktoken import Encoding, encoding_for_model, get_encoding
from ._llm import LLMTextResult, LLMUsage


def _encoding_for_model(
    model: str,
    /,
) -> Encoding:
    try:
        return encoding_for_model(model)
    except Exception:
        return get_encoding("cl100k_base")


class OpenAIResponsesProvider:
    name = "openai"

    def __init__(
        self,
        /,
        *,
        api_key: str | None = None,
    ) -> None:
        key = api_key or environ.get("OPENAI_API_KEY")
        if not key:
            raise RuntimeError("The OPENAI_API_KEY environment variable is required.")
        self._client = OpenAI(api_key=key)

    def count_tokens(
        self,
        /,
        *,
        model: str,
        text: str,
    ) -> int:
        encoding = _encoding_for_model(model)
        return len(encoding.encode(text))

    def generate_text(
        self,
        /,
        *,
        model: str,
        instructions: str,
        user_text: str,
    ) -> LLMTextResult:
        resp: Response = self._client.responses.create(
            model=model,
            instructions=instructions,
            input=[
                {
                    "role": "user",
                    "content": [
                        {
                            "type": "input_text",
                            "text": user_text,
                        }
                    ],
                }
            ],
        )

        text: str = (resp.output_text or "").strip()
        if not text:
            raise RuntimeError("An empty response text was generated by the provider.")

        usage: LLMUsage | None = None
        if resp.usage is not None:
            usage = LLMUsage(
                prompt_tokens=resp.usage.input_tokens,
                completion_tokens=resp.usage.output_tokens,
                total_tokens=resp.usage.total_tokens,
            )

        return LLMTextResult(
            text=text,
            response_id=resp.id,
            usage=usage,
        )
