"""Google (Gemini) provider implementation.

This module contains only Google GenAI-specific API calls and token counting.
Provider-agnostic orchestration/prompt logic lives in `_llm.py`.
"""

from __future__ import annotations

from os import environ
from typing import ClassVar

from google import genai
from google.genai import types

from ._llm import LLMTextResult, LLMUsage


class GoogleGenAIProvider:
    __slots__ = (
        "_client",
    )

    name: ClassVar[str] = "google"

    def __init__(
        self,
        /,
        *,
        api_key: str | None = None,
    ) -> None:
        key = api_key or environ.get("GOOGLE_API_KEY")
        if not key:
            raise RuntimeError("The GOOGLE_API_KEY environment variable is required.")
        self._client = genai.Client(api_key=key)

    def count_tokens(
        self,
        /,
        *,
        model: str,
        text: str,
    ) -> int:
        try:
            resp = self._client.models.count_tokens(
                model=model,
                contents=text,
            )
        except Exception as exc:
            raise RuntimeError(
                "Token counting failed for the Google provider. "
                "Try `--chunk-tokens 0` (default) or `--chunk-tokens -1` to disable summarisation."
            ) from exc

        total = getattr(resp, "total_tokens", None)
        if not isinstance(total, int):
            raise RuntimeError(
                "Token counting returned an unexpected response from the Google provider. "
                "Try `--chunk-tokens 0` (default) or `--chunk-tokens -1` to disable summarisation."
            )

        return total

    def generate_text(
        self,
        /,
        *,
        model: str,
        instructions: str,
        user_text: str,
    ) -> LLMTextResult:
        config = types.GenerateContentConfig(
            system_instruction=instructions,
        )

        resp = self._client.models.generate_content(
            model=model,
            contents=user_text,
            config=config,
        )

        text = self._extract_text(resp)
        if not text:
            raise RuntimeError("An empty response text was generated by the provider.")

        usage = self._extract_usage(resp)

        return LLMTextResult(
            text=text,
            response_id=getattr(resp, "response_id", None),
            usage=usage,
        )

    @staticmethod
    def _extract_text(
        resp: types.GenerateContentResponse,
        /,
    ) -> str:
        candidates = getattr(resp, "candidates", None)
        if not candidates:
            return ""

        parts = getattr(candidates[0].content, "parts", None) if candidates[0].content else None
        if not parts:
            return ""

        texts: list[str] = []
        for part in parts:
            t = getattr(part, "text", None)
            if isinstance(t, str) and t.strip():
                texts.append(t)

        return "\n".join(texts).strip()

    @staticmethod
    def _extract_usage(
        resp: types.GenerateContentResponse,
        /,
    ) -> LLMUsage | None:
        metadata = getattr(resp, "usage_metadata", None)
        if metadata is None:
            return None

        return LLMUsage(
            prompt_tokens=getattr(metadata, "prompt_token_count", None),
            completion_tokens=getattr(metadata, "candidates_token_count", None),
            total_tokens=getattr(metadata, "total_token_count", None),
        )
