"""Tests for CLI interface."""
import json
from unittest.mock import AsyncMock, patch

import pytest

from gitcommit_ai.cli.main import main
from gitcommit_ai.generator.message import CommitMessage


class TestCLIArgumentParsing:
    """Test command-line argument parsing."""

    def test_cli_parses_json_flag(self) -> None:
        """CLI recognizes --json flag."""
        with patch("sys.argv", ["gitcommit-ai", "generate", "--json"]):
            with patch("gitcommit_ai.cli.main.run_generate", new_callable=AsyncMock):
                # Just test that parsing works, not execution
                pass

    def test_cli_parses_provider_flag(self) -> None:
        """CLI recognizes --provider flag."""
        with patch("sys.argv", ["gitcommit-ai", "generate", "--provider", "anthropic"]):
            with patch("gitcommit_ai.cli.main.run_generate", new_callable=AsyncMock):
                pass

    def test_cli_parses_verbose_flag(self) -> None:
        """CLI recognizes --verbose flag."""
        with patch("sys.argv", ["gitcommit-ai", "generate", "--verbose"]):
            with patch("gitcommit_ai.cli.main.run_generate", new_callable=AsyncMock):
                pass


class TestCLIOutput:
    """Test CLI output formatting."""

    @pytest.mark.asyncio
    async def test_cli_outputs_human_readable_by_default(self) -> None:
        """CLI outputs human-readable format by default."""
        mock_message = CommitMessage(
            type="feat",
            scope="cli",
            description="add output formatting",
            body=None,
            breaking_changes=[],
        )

        with patch(
            "gitcommit_ai.generator.generator.CommitMessageGenerator.generate",
            new_callable=AsyncMock,
        ) as mock_gen:
            mock_gen.return_value = mock_message

            with patch("builtins.print") as mock_print:
                with patch("sys.argv", ["gitcommit-ai", "generate"]):
                    with patch("os.getenv", return_value="sk-test123"):
                        # Would normally call main() but need to mock async
                        from gitcommit_ai.cli.main import format_output

                        output = format_output(mock_message, json_format=False)
                        assert "feat(cli): add output formatting" in output
                        assert not output.startswith("{")

    @pytest.mark.asyncio
    async def test_cli_outputs_json_with_flag(self) -> None:
        """CLI outputs JSON format when --json flag is used."""
        mock_message = CommitMessage(
            type="fix",
            scope="api",
            description="handle errors",
            body=None,
            breaking_changes=[],
        )

        from gitcommit_ai.cli.main import format_output

        output = format_output(mock_message, json_format=True)
        data = json.loads(output)

        assert data["type"] == "fix"
        assert data["scope"] == "api"
        assert data["description"] == "handle errors"


class TestCLIErrorHandling:
    """Test CLI error cases."""

    def test_cli_works_without_api_keys_using_ollama(self) -> None:
        """CLI works without API keys by defaulting to Ollama."""
        # This test verifies that Ollama is used as default when no API keys present
        # The actual generation would require Ollama to be installed, so we just
        # verify the config logic is correct
        from gitcommit_ai.core.config import Config

        with patch.dict("os.environ", {}, clear=True):
            config = Config.load()
            assert config.default_provider == "ollama"

    def test_cli_exits_with_code_1_on_git_error(self) -> None:
        """CLI exits with code 1 on git errors."""
        with patch(
            "gitcommit_ai.core.git.GitOperations.is_git_repository",
            return_value=False,
        ):
            with patch("sys.argv", ["gitcommit-ai", "generate"]):
                with patch("os.getenv", return_value="sk-test123"):
                    with pytest.raises(SystemExit) as exc_info:
                        main()
                    assert exc_info.value.code == 1


class TestCLIMultipleSuggestions:
    """Test CLI multiple suggestions feature (--count flag)."""

    @pytest.mark.asyncio
    async def test_cli_ollama_supports_count_flag(self) -> None:
        """Ollama provider supports --count flag for multiple suggestions."""
        import argparse
        from gitcommit_ai.cli.main import run_generate
        from gitcommit_ai.generator.message import GitDiff

        mock_messages = [
            CommitMessage(
                type="feat",
                scope="auth",
                description=f"implement authentication system v{i}",
                body=None,
                breaking_changes=[],
            )
            for i in range(3)
        ]

        # Create args namespace
        args = argparse.Namespace(
            provider="ollama",
            model=None,
            json=True,
            verbose=False,
            gitmoji=False,
            no_gitmoji=False,
            count=3
        )

        with patch("gitcommit_ai.core.git.GitOperations.is_git_repository", return_value=True):
            with patch("gitcommit_ai.core.git.GitOperations.has_staged_changes", return_value=True):
                with patch("gitcommit_ai.core.git.GitOperations.get_staged_diff", return_value=GitDiff(files=[], total_additions=10, total_deletions=5)):
                    with patch("gitcommit_ai.providers.ollama.OllamaProvider.validate_config", return_value=[]):
                        with patch("gitcommit_ai.generator.multi_generator.MultiSuggestionGenerator.generate_multiple", new_callable=AsyncMock, return_value=mock_messages):
                            with patch("builtins.print") as mock_print:
                                await run_generate(args)

                                # Verify JSON output contains all 3 suggestions
                                output = mock_print.call_args[0][0]
                                data = json.loads(output)
                                assert isinstance(data, list)
                                assert len(data) == 3

    @pytest.mark.asyncio
    async def test_cli_ollama_count_validation(self) -> None:
        """Ollama --count flag validates range (1-10)."""
        # This is already handled by MultiSuggestionGenerator
        # Just verify the CLI accepts the flag
        with patch("sys.argv", ["gitcommit-ai", "generate", "--provider", "ollama", "--count", "5"]):
            with patch("gitcommit_ai.cli.main.run_generate", new_callable=AsyncMock):
                pass  # Just verify parsing works
