import json

from .models import ParsedDiff, Hunk, DiffLine, AllParsedDiffs

LINE_TYPE_ADDED = "+"
LINE_TYPE_REMOVED = "-"
LINE_TYPE_UNCHANGED = " "

def parse_diff(diff_str) -> AllParsedDiffs:
    """
    Parses a git diff string into a structured format.

    Args:
        diff_str (str): The git diff string to parse.

    Returns:
        dict: A dictionary with the parsed diff information.
        The structure is as follows:
        {
            "file": "filename",
            "hunks": [
                {
                    "old_start_line": int,
                    "old_end_line": int,
                    "new_start_line": int,
                    "new_end_line": int,
                    "lines": [
                        {
                            "type": "+", "-", or " ",
                            "content": str
                        }
                    ]
                }
            ]
        }
    """

    def parse_file_header(parse_line):
        """Extract file information from a diff --git header.

        Args:
            parse_line (str): The header line from the diff.

        Returns:
            str: The extracted file name.
        """
        parts = parse_line.split()
        return parts[3].replace("'", "")  # Extract file name

    def parse_hunk_header(parse_line) -> Hunk:
        """Parse a hunk header and return a hunk dictionary.

        Args:
            parse_line (str): The hunk header line from the diff.

        Returns:
            dict: A dictionary with hunk information.
        """
        parts = parse_line.split("@@")
        hunk_parts = parts[1].strip().split()
        old_start_line, old_count = map(int, hunk_parts[0].lstrip("-").split(","))
        new_start_line, new_count = map(int, hunk_parts[1].lstrip("+").split(","))
        hunk = Hunk(
            old_start_line=old_start_line,
            old_end_line=old_start_line + old_count - 1,
            new_start_line=new_start_line,
            new_end_line=new_start_line + new_count - 1,
            lines=[]
        )

        if len(parts) > 2 and parts[2].strip():  # Include trailing context if present
            hunk.lines.append(DiffLine(diff_type="context", content=parts[2].strip()))
        return hunk

    def parse_diff_line(diff_line: str) -> DiffLine:
        """
        Parse a diff line to determine its type and content.

        Args:
            diff_line (str): A single line from the diff.

        Returns:
            DiffLine: An object representing the type and content of the diff line.
        """
        mapped_type = DiffLine.map_type(diff_line[0])  # Map the line type using DiffLine's method
        content = diff_line[1:].strip()  # Extract content, excluding the type symbol
        return DiffLine(diff_type=mapped_type, content=content)

    all_parsed_diffs = AllParsedDiffs(diffs=[])
    diff_lines = diff_str.split("\n")
    parsed_diff= ParsedDiff(file="", hunks=[])
    current_hunk_data = None

    for line in diff_lines:
        if not line.strip():
            continue
        if line.startswith("diff --git "):
            # if we already have a parsed diff append it before creating a new one
            if parsed_diff.file:
                all_parsed_diffs.diffs.append(parsed_diff)
            parsed_diff = ParsedDiff(file=parse_file_header(line), hunks=[])
        elif line.startswith("@@"):
            current_hunk_data = parse_hunk_header(line)
            # parsed_diff["hunks"].append(current_hunk_data)
            parsed_diff.hunks.append(current_hunk_data)
        elif (
                line[0] in (LINE_TYPE_ADDED, LINE_TYPE_REMOVED, LINE_TYPE_UNCHANGED)
                and current_hunk_data is not None
        ):
            # current_hunk_data["lines"].append(parse_diff_line(line))

            current_hunk_data.lines.append(parse_diff_line(line))

    all_parsed_diffs.diffs.append(parsed_diff)
    return all_parsed_diffs

def app(diff_str) -> str:
    """
    Main function to parse a git diff string and return the JSON representation.

    Args:
        diff_str (str): The git diff string to parse.

    Returns:
        str: The JSON representation of the parsed diff.
    """
    parsed_diff = parse_diff(diff_str)
    json_output = (
        json.dumps(parsed_diff, ensure_ascii=False)
        .replace("\u2028", "\\u2028")
        .replace("\u2029", "\\u2029")
    )

    return json_output