#!/usr/bin/env python3
"""
Simplified Commit Classification System Test

This script tests the commit classification system using the current gitflow-analytics
repository as test data. This allows us to validate the classification pipeline
without needing access to external repositories.
"""

import sys
import json
import logging
from pathlib import Path
from datetime import datetime, timezone, timedelta
from collections import Counter

# Add the src directory to Python path for local development
sys.path.insert(0, str(Path(__file__).parent / "src"))

from gitflow_analytics.classification.classifier import CommitClassifier
from gitflow_analytics.core.analyzer import GitAnalyzer
from gitflow_analytics.core.cache import GitAnalysisCache
from gitflow_analytics.core.identity import DeveloperIdentityResolver
from gitflow_analytics.reports.classification_writer import ClassificationReportGenerator

# Configure logging
logging.basicConfig(
    level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
)
logger = logging.getLogger(__name__)


def test_classification_system():
    """Test the classification system with the current repository."""

    print("🚀 GitFlow Analytics - Commit Classification System Test")
    print("=" * 60)

    # Setup paths and dates
    repo_path = Path.cwd()  # Current gitflow-analytics repository
    cache_dir = Path.cwd() / ".test_cache"
    reports_dir = Path.cwd() / "test_reports"

    # Test with recent commits (last 30 days)
    end_date = datetime.now(timezone.utc)
    start_date = end_date - timedelta(days=30)

    print(f"📂 Repository: {repo_path}")
    print(f"📅 Date range: {start_date.date()} to {end_date.date()}")
    print(f"💾 Cache directory: {cache_dir}")
    print(f"📊 Reports directory: {reports_dir}")

    try:
        # Step 1: Initialize components
        print("\n🔧 Initializing components...")

        # Setup cache
        cache = GitAnalysisCache(cache_dir)
        print("✅ Cache initialized")

        # Setup identity resolver
        identity_db_path = cache_dir / "identities.db"
        identity_resolver = DeveloperIdentityResolver(
            db_path=identity_db_path, similarity_threshold=0.85
        )
        print("✅ Identity resolver initialized")

        # Setup classifier
        classification_config = {"enabled": True, "confidence_threshold": 0.6, "batch_size": 50}

        classifier = CommitClassifier(
            config=classification_config, cache_dir=cache_dir / "classification"
        )
        print("✅ Classification system initialized")

        # Setup analyzer
        analyzer = GitAnalyzer(
            cache=cache, batch_size=100, classification_config=classification_config
        )
        print("✅ Git analyzer initialized")

        # Step 2: Analyze commits
        print("\n📊 Analyzing commits...")

        commits = analyzer.analyze_repository(repo_path=repo_path, since=start_date)

        # Filter to date range - handle timezone-aware vs naive datetimes
        filtered_commits = []
        for commit in commits:
            commit_time = commit["timestamp"]
            # Ensure both datetimes are timezone-aware
            if commit_time.tzinfo is None:
                commit_time = commit_time.replace(tzinfo=timezone.utc)

            if start_date <= commit_time <= end_date:
                filtered_commits.append(commit)

        print(f"✅ Found {len(filtered_commits)} commits in date range")

        if not filtered_commits:
            print("⚠️ No commits found in the specified date range")
            return False

        # Step 3: Apply identity resolution
        print("\n👥 Applying identity resolution...")

        normalized_commits = []
        unique_developers = set()

        for commit in filtered_commits:
            canonical_id = identity_resolver.resolve_developer(
                commit["author_name"], commit["author_email"]
            )

            commit["canonical_author_name"] = canonical_id
            commit["canonical_author_id"] = canonical_id
            commit["repository"] = "gitflow-analytics"
            commit["project_key"] = "GFA"

            normalized_commits.append(commit)
            unique_developers.add(canonical_id)

        print(f"✅ Normalized identities for {len(normalized_commits)} commits")
        print(f"👥 Found {len(unique_developers)} unique developers")

        # Step 4: Apply classification
        print("\n🏷️ Applying commit classification...")

        # Skip training for now to test basic classification
        print("ℹ️ Using fallback rule-based classification (no training)")

        # Classify commits
        try:
            classification_results = classifier.classify_commits(normalized_commits)
            print(f"✅ Got classification results: {type(classification_results)}")

            # Handle case where classification_results might be a single result or list
            if not isinstance(classification_results, list):
                print(f"⚠️ Unexpected classification result type: {type(classification_results)}")
                classification_results = [classification_results] * len(normalized_commits)

            if len(classification_results) != len(normalized_commits):
                print(
                    f"⚠️ Classification results length mismatch: {len(classification_results)} vs {len(normalized_commits)}"
                )
                # Pad or truncate as needed
                while len(classification_results) < len(normalized_commits):
                    classification_results.append(
                        {
                            "predicted_class": "unknown",
                            "confidence": 0.0,
                            "is_reliable_prediction": False,
                        }
                    )

            # Merge results
            classified_commits = []
            for commit, classification in zip(normalized_commits, classification_results):
                if isinstance(classification, dict):
                    commit.update(
                        {
                            "predicted_class": classification.get("predicted_class", "unknown"),
                            "classification_confidence": classification.get("confidence", 0.0),
                            "is_reliable_prediction": classification.get(
                                "is_reliable_prediction", False
                            ),
                            "class_probabilities": classification.get("class_probabilities", {}),
                            "file_analysis_summary": classification.get("file_analysis", {}),
                        }
                    )
                else:
                    print(f"⚠️ Unexpected classification format: {type(classification)}")
                    commit.update(
                        {
                            "predicted_class": "unknown",
                            "classification_confidence": 0.0,
                            "is_reliable_prediction": False,
                            "class_probabilities": {},
                            "file_analysis_summary": {},
                        }
                    )
                classified_commits.append(commit)

        except Exception as e:
            print(f"⚠️ Classification failed: {e}")
            # Fall back to unclassified commits
            classified_commits = normalized_commits.copy()
            for commit in classified_commits:
                commit.update(
                    {
                        "predicted_class": "unknown",
                        "classification_confidence": 0.0,
                        "is_reliable_prediction": False,
                        "class_probabilities": {},
                        "file_analysis_summary": {},
                    }
                )

        print(f"✅ Classified {len(classified_commits)} commits")

        # Step 5: Generate analysis summary
        print("\n📈 Generating analysis summary...")

        # Classification distribution
        classification_dist = Counter(
            commit["predicted_class"]
            for commit in classified_commits
            if "predicted_class" in commit
        )

        # Confidence analysis
        confidence_scores = [
            commit["classification_confidence"]
            for commit in classified_commits
            if "classification_confidence" in commit
        ]

        avg_confidence = sum(confidence_scores) / len(confidence_scores) if confidence_scores else 0
        high_confidence_count = sum(1 for score in confidence_scores if score >= 0.8)

        # Developer breakdown
        dev_stats = {}
        for commit in classified_commits:
            dev = commit.get("canonical_author_name", "unknown")
            if dev not in dev_stats:
                dev_stats[dev] = {"commits": 0, "classifications": Counter()}

            dev_stats[dev]["commits"] += 1
            if "predicted_class" in commit:
                dev_stats[dev]["classifications"][commit["predicted_class"]] += 1

        # Print summary
        print(f"\n📊 ANALYSIS RESULTS")
        print(f"=" * 50)
        print(f"Total Commits: {len(classified_commits)}")
        print(f"Unique Developers: {len(unique_developers)}")
        print(f"Average Confidence: {avg_confidence:.3f}")
        print(
            f"High Confidence (≥0.8): {high_confidence_count}/{len(confidence_scores)} ({high_confidence_count/len(confidence_scores)*100:.1f}%)"
        )

        print(f"\n🏷️ CLASSIFICATION DISTRIBUTION:")
        for class_type, count in classification_dist.most_common():
            percentage = (count / len(classified_commits)) * 100
            print(f"  {class_type}: {count} ({percentage:.1f}%)")

        print(f"\n👥 TOP DEVELOPERS:")
        for dev, stats in sorted(dev_stats.items(), key=lambda x: x[1]["commits"], reverse=True)[
            :5
        ]:
            primary_class = (
                stats["classifications"].most_common(1)[0][0]
                if stats["classifications"]
                else "none"
            )
            print(f"  {dev}: {stats['commits']} commits (primary: {primary_class})")

        # Step 6: Generate professional reports
        print(f"\n📋 Generating professional reports...")

        reports_dir.mkdir(exist_ok=True)

        report_generator = ClassificationReportGenerator(
            output_directory=reports_dir, config={"confidence_threshold": 0.6}
        )

        metadata = {
            "start_date": start_date.date().isoformat(),
            "end_date": end_date.date().isoformat(),
            "repository": "gitflow-analytics",
            "analysis_type": "Classification System Test",
        }

        report_paths = report_generator.generate_comprehensive_report(
            classified_commits=classified_commits, metadata=metadata
        )

        print(f"✅ Generated {len(report_paths)} professional reports:")
        for report_type, path in report_paths.items():
            if Path(path).exists():
                print(f"  📄 {report_type}: {Path(path).name}")

        # Step 7: Save test results
        results_file = reports_dir / "test_results.json"
        test_results = {
            "metadata": {
                "test_date": datetime.now().isoformat(),
                "repository": "gitflow-analytics",
                "date_range": {"start": start_date.isoformat(), "end": end_date.isoformat()},
            },
            "summary": {
                "total_commits": len(classified_commits),
                "unique_developers": len(unique_developers),
                "average_confidence": avg_confidence,
                "high_confidence_rate": (
                    high_confidence_count / len(confidence_scores) if confidence_scores else 0
                ),
                "classification_distribution": dict(classification_dist),
                "developer_stats": {
                    dev: {
                        "commits": stats["commits"],
                        "primary_classification": (
                            stats["classifications"].most_common(1)[0][0]
                            if stats["classifications"]
                            else "none"
                        ),
                    }
                    for dev, stats in dev_stats.items()
                },
            },
        }

        with open(results_file, "w") as f:
            json.dump(test_results, f, indent=2, default=str)

        print(f"💾 Test results saved to: {results_file}")

        print(f"\n🎉 Classification system test completed successfully!")
        print(f"📁 All outputs available in: {reports_dir}")

        return True

    except Exception as e:
        logger.error(f"Test failed: {e}")
        print(f"\n❌ Test failed: {e}")
        return False


if __name__ == "__main__":
    success = test_classification_system()
    sys.exit(0 if success else 1)
