# Managing Developer Aliases

This guide explains how to manage developer identity aliases in GitFlow Analytics, including creating, sharing, and maintaining alias configurations across multiple projects.

## Overview

Developer aliases allow you to consolidate multiple email addresses and identities for the same developer into a single unified identity. This is particularly useful when:

- Developers use different email addresses across projects (personal, work, no-reply addresses)
- GitHub automatically generates no-reply email addresses for commits
- Team members change email addresses over time
- You want consistent identity resolution across multiple analysis configurations

## Shared Aliases File

GitFlow Analytics supports a shared `aliases.yaml` file that can be referenced by multiple configuration files. This approach:

- **Centralizes identity mappings** - Maintain one source of truth for all projects
- **Reduces duplication** - No need to repeat alias configurations in every config file
- **Simplifies updates** - Change an alias once, apply everywhere
- **Supports team collaboration** - Share alias files across team members

## Aliases File Format

Create an `aliases.yaml` file with the following structure:

```yaml
# Developer Identity Aliases
# Generated by GitFlow Analytics
# Share this file across multiple config files
# Each alias maps multiple email addresses to a single developer

developer_aliases:
  # Manual alias - high confidence
  - name: "John Smith"
    primary_email: "john.smith@company.com"
    aliases:
      - "jsmith@gmail.com"
      - "150280367+jsmith@users.noreply.github.com"
      - "john.smith@oldcompany.com"

  # LLM-generated alias - includes confidence and reasoning
  - name: "Jane Doe"
    primary_email: "jane.doe@company.com"
    aliases:
      - "jane@personal.com"
      - "jdoe@company.com"
    confidence: 0.95
    reasoning: "Same name pattern and commit style across all addresses"

  # Backward compatibility - canonical_email also supported
  - canonical_email: "bob@company.com"
    aliases:
      - "robert@gmail.com"
```

### Field Descriptions

- **name** (optional): Display name for the developer in reports
- **primary_email** (required): Primary email address for this developer
- **canonical_email**: Deprecated, use `primary_email` instead (still supported for backward compatibility)
- **aliases** (required): List of alternate email addresses for the same developer
- **confidence** (optional): Confidence score for LLM-generated aliases (0.0-1.0, default: 1.0)
- **reasoning** (optional): Explanation for why emails were grouped together (for LLM-generated aliases)

## Using Aliases in Configuration

### Method 1: Reference External Aliases File

Reference a shared aliases file in your config:

```yaml
# config.yaml
analysis:
  identity:
    aliases_file: ../shared/aliases.yaml  # Relative to config file
    similarity_threshold: 0.85
```

or with absolute path:

```yaml
analysis:
  identity:
    aliases_file: /Users/team/shared-configs/aliases.yaml
```

### Method 2: Inline Manual Mappings

For project-specific aliases, use inline mappings:

```yaml
analysis:
  identity:
    manual_mappings:
      - name: "John Smith"
        primary_email: "john.smith@company.com"
        aliases:
          - "jsmith@gmail.com"
```

### Method 3: Combine Both

Use both approaches for maximum flexibility:

```yaml
analysis:
  identity:
    aliases_file: ../shared/team-aliases.yaml
    manual_mappings:
      # Project-specific aliases
      - name: "Contract Developer"
        primary_email: "contractor@external.com"
        aliases:
          - "temp-dev@project.com"
```

The system will merge aliases from both sources, with inline `manual_mappings` taking precedence in case of conflicts.

## Generating Aliases Automatically

GitFlow Analytics can analyze your commit history and automatically generate alias suggestions using LLM analysis.

### Basic Usage

```bash
# Analyze commits and generate aliases
gitflow-analytics aliases -c config.yaml

# Save to specific location
gitflow-analytics aliases -c config.yaml -o ~/shared/aliases.yaml

# Generate and apply automatically
gitflow-analytics aliases -c config.yaml --apply
```

### Confidence Threshold

Control the minimum confidence level for automatic alias generation:

```bash
# Only include high-confidence matches (90%+)
gitflow-analytics aliases -c config.yaml --confidence-threshold 0.9

# Include all reasonable matches (70%+)
gitflow-analytics aliases -c config.yaml --confidence-threshold 0.7
```

### Review Process

The alias generation command will:

1. **Analyze commit history** - Examine email addresses, names, and commit patterns
2. **Generate suggestions** - Use LLM to identify likely matches with confidence scores
3. **Display results** - Show proposed aliases with reasoning
4. **Prompt for approval** - Ask before updating configuration (unless `--apply` is used)

Example output:

```
🔍 Analyzing developer identities from config.yaml
📊 Confidence threshold: 90%

Found 3 potential aliases:

  John Smith
    Primary: john.smith@company.com
    Aliases: jsmith@gmail.com, 150280367+jsmith@users.noreply.github.com
    Confidence: 95%
    Reasoning: Same name pattern and consistent commit style

  Jane Doe
    Primary: jane.doe@company.com
    Aliases: jane@personal.com
    Confidence: 92%
    Reasoning: Name matches and overlapping commit timestamps

Apply these aliases? [y/N]:
```

## Multiple Configs Sharing Aliases

### Recommended Directory Structure

```
/project/
├── shared/
│   └── aliases.yaml          # Shared aliases
├── team1/
│   └── config-team1.yaml     # Team 1 config
├── team2/
│   └── config-team2.yaml     # Team 2 config
└── combined/
    └── config-all.yaml       # Combined config
```

### Configuration Examples

**Team 1 Config:**
```yaml
# team1/config-team1.yaml
analysis:
  identity:
    aliases_file: ../shared/aliases.yaml
```

**Team 2 Config:**
```yaml
# team2/config-team2.yaml
analysis:
  identity:
    aliases_file: ../shared/aliases.yaml
```

**Combined Config:**
```yaml
# combined/config-all.yaml
analysis:
  identity:
    aliases_file: ../shared/aliases.yaml
```

All three configurations now share the same alias mappings, ensuring consistent identity resolution across all analyses.

## Programmatic Access

You can also work with aliases programmatically:

```python
from pathlib import Path
from gitflow_analytics.config import AliasesManager, DeveloperAlias

# Load existing aliases
manager = AliasesManager(Path("aliases.yaml"))

# Add a new alias
manager.add_alias(DeveloperAlias(
    name="John Smith",
    primary_email="john@company.com",
    aliases=["jsmith@gmail.com", "john.smith@oldcompany.com"],
    confidence=1.0
))

# Save changes
manager.save()

# Get statistics
stats = manager.get_statistics()
print(f"Total aliases: {stats['total_aliases']}")
print(f"Manual aliases: {stats['manual_aliases']}")
print(f"LLM-generated: {stats['llm_aliases']}")
print(f"Average confidence: {stats['avg_confidence']}")
```

## Best Practices

### 1. Start with High Confidence

When using LLM-generated aliases, start with a high confidence threshold (0.9+) and gradually lower it as needed:

```bash
gitflow-analytics aliases -c config.yaml --confidence-threshold 0.95
```

### 2. Review Before Applying

Always review LLM-generated aliases before applying them:

```bash
# Generate and review
gitflow-analytics aliases -c config.yaml

# Apply after review
gitflow-analytics aliases -c config.yaml --apply
```

### 3. Use Shared Aliases for Teams

For team projects, maintain a shared `aliases.yaml` in version control:

```bash
# Add to version control
git add shared/aliases.yaml
git commit -m "feat: add shared developer aliases"
```

### 4. Document Custom Aliases

Add comments to explain non-obvious aliases:

```yaml
developer_aliases:
  # John changed email when company was acquired
  - name: "John Smith"
    primary_email: "john.smith@newcompany.com"
    aliases:
      - "john.smith@oldcompany.com"
```

### 5. Update Aliases Regularly

Re-run alias generation periodically as new developers join:

```bash
# Quarterly alias update
gitflow-analytics aliases -c config.yaml --confidence-threshold 0.9
```

### 6. Combine with Manual Mappings

Use aliases file for common mappings, inline for project-specific:

```yaml
analysis:
  identity:
    # Shared team aliases
    aliases_file: ../shared/team-aliases.yaml

    # Project-specific exceptions
    manual_mappings:
      - name: "External Consultant"
        primary_email: "consultant@external.com"
        aliases:
          - "temp-account@project.com"
```

## Troubleshooting

### Aliases Not Loading

If aliases aren't being applied:

1. **Check file path**: Ensure `aliases_file` path is correct relative to config file
2. **Verify file exists**: Confirm the aliases.yaml file exists at the specified path
3. **Check YAML syntax**: Validate YAML format (no tabs, proper indentation)
4. **Review logs**: Check for warnings about alias loading failures

```bash
# Test config loading
gitflow-analytics analyze -c config.yaml --validate-only
```

### Conflicting Aliases

If multiple aliases specify the same email:

- **Inline mappings take precedence** over aliases file
- **Later entries override earlier entries** in the same file
- **Use primary_email uniquely** - each developer should have one unique primary email

### LLM Generation Errors

If alias generation fails:

1. **Check API access**: Ensure you have a valid OpenRouter or OpenAI API key
2. **Verify network**: Confirm internet connectivity for LLM API calls
3. **Review commit history**: Ensure sufficient commit data exists for analysis
4. **Try lower confidence**: Lower the threshold if no matches are found

## Advanced Topics

### Custom Similarity Threshold

Adjust the identity matching threshold:

```yaml
analysis:
  identity:
    aliases_file: aliases.yaml
    similarity_threshold: 0.90  # Higher = more strict matching
```

### Programmatic Alias Management

Build custom alias management tools:

```python
from gitflow_analytics.config import AliasesManager

# Load and merge multiple alias files
main_mgr = AliasesManager(Path("main-aliases.yaml"))
team_mgr = AliasesManager(Path("team-aliases.yaml"))

# Merge team aliases into main
main_mgr.merge_from_mappings(team_mgr.to_manual_mappings())

# Save consolidated aliases
main_mgr.save()
```

### Alias Statistics and Validation

Get insights about your aliases:

```python
manager = AliasesManager(Path("aliases.yaml"))
stats = manager.get_statistics()

print(f"Tracking {stats['total_email_addresses']} email addresses")
print(f"Consolidated into {stats['total_aliases']} developers")
print(f"Average confidence: {stats['avg_confidence']:.1%}")

# Find low-confidence aliases for review
low_confidence = [
    a for a in manager.aliases
    if a.confidence < 0.8
]

for alias in low_confidence:
    print(f"Review: {alias.name} ({alias.confidence:.0%} confidence)")
```

## Related Documentation

- [Identity Resolution Guide](identity-resolution-enhanced.md) - Core identity resolution concepts
- [Configuration Guide](../reference/configuration.md) - Complete configuration reference
- [Interactive Launcher](interactive-launcher.md) - Using the interactive launcher

## Support

For issues or questions:

- Check [Troubleshooting Guide](troubleshooting.md)
- Review [GitHub Issues](https://github.com/EWTN-Global/gitflow-analytics/issues)
- Consult [Project Documentation](../README.md)
