from __future__ import annotations

from typing import Optional

import httpx
from typing_extensions import Literal

from ...._base_client import make_request_options
from ...._compat import cached_property
from ...._resource import AsyncAPIResource, SyncAPIResource
from ...._response import (
    async_to_raw_response_wrapper,
    async_to_streamed_response_wrapper,
    to_raw_response_wrapper,
    to_streamed_response_wrapper,
)
from ...._types import NOT_GIVEN, Body, Headers, NotGiven, Query
from ...._utils import (
    async_maybe_transform,
    maybe_transform,
)
from ....types.repos.code_scanning import analysis_delete_params, analysis_list_params
from ....types.repos.code_scanning.analysis_delete_response import AnalysisDeleteResponse
from ....types.repos.code_scanning.analysis_list_response import AnalysisListResponse
from ....types.repos.code_scanning.code_scanning_analysis import CodeScanningAnalysis

__all__ = ["AnalysesResource", "AsyncAnalysesResource"]


class AnalysesResource(SyncAPIResource):
    @cached_property
    def with_raw_response(self) -> AnalysesResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return AnalysesResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> AnalysesResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return AnalysesResourceWithStreamingResponse(self)

    def retrieve(
        self,
        analysis_id: int,
        *,
        owner: str,
        repo: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> CodeScanningAnalysis:
        """
        Gets a specified code scanning analysis for a repository.

        The default JSON response contains fields that describe the analysis. This
        includes the Git reference and commit SHA to which the analysis relates, the
        datetime of the analysis, the name of the code scanning tool, and the number of
        alerts.

        The `rules_count` field in the default response give the number of rules that
        were run in the analysis. For very old analyses this data is not available, and
        `0` is returned in this field.

        This endpoint supports the following custom media types. For more information,
        see
        "[Media types](https://docs.github.com/rest/using-the-rest-api/getting-started-with-the-rest-api#media-types)."

        - **`application/sarif+json`**: Instead of returning a summary of the analysis,
          this endpoint returns a subset of the analysis data that was uploaded. The
          data is formatted as
          [SARIF version 2.1.0](https://docs.oasis-open.org/sarif/sarif/v2.1.0/cs01/sarif-v2.1.0-cs01.html).
          It also returns additional data such as the `github/alertNumber` and
          `github/alertUrl` properties.

        OAuth app tokens and personal access tokens (classic) need the `security_events`
        scope to use this endpoint with private or public repositories, or the
        `public_repo` scope to use this endpoint with only public repositories.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return self._get(
            f"/repos/{owner}/{repo}/code-scanning/analyses/{analysis_id}",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=CodeScanningAnalysis,
        )

    def list(
        self,
        repo: str,
        *,
        owner: str,
        direction: Literal["asc", "desc"] | NotGiven = NOT_GIVEN,
        page: int | NotGiven = NOT_GIVEN,
        per_page: int | NotGiven = NOT_GIVEN,
        pr: int | NotGiven = NOT_GIVEN,
        ref: str | NotGiven = NOT_GIVEN,
        sarif_id: str | NotGiven = NOT_GIVEN,
        sort: Literal["created"] | NotGiven = NOT_GIVEN,
        tool_guid: str | None | NotGiven = NOT_GIVEN,
        tool_name: str | NotGiven = NOT_GIVEN,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> AnalysisListResponse:
        """
        Lists the details of all code scanning analyses for a repository, starting with
        the most recent. The response is paginated and you can use the `page` and
        `per_page` parameters to list the analyses you're interested in. By default 30
        analyses are listed per page.

        The `rules_count` field in the response give the number of rules that were run
        in the analysis. For very old analyses this data is not available, and `0` is
        returned in this field.

        > [!WARNING] > **Closing down notice:** The `tool_name` field is closing down
        > and will, in future, not be included in the response for this endpoint. The
        > example response reflects this change. The tool name can now be found inside
        > the `tool` field.

        OAuth app tokens and personal access tokens (classic) need the `security_events`
        scope to use this endpoint with private or public repositories, or the
        `public_repo` scope to use this endpoint with only public repositories.

        Args:
          direction: The direction to sort the results by.

          page: The page number of the results to fetch. For more information, see
              "[Using pagination in the REST API](https://docs.github.com/rest/using-the-rest-api/using-pagination-in-the-rest-api)."

          per_page: The number of results per page (max 100). For more information, see
              "[Using pagination in the REST API](https://docs.github.com/rest/using-the-rest-api/using-pagination-in-the-rest-api)."

          pr: The number of the pull request for the results you want to list.

          ref: The Git reference for the analyses you want to list. The `ref` for a branch can
              be formatted either as `refs/heads/<branch name>` or simply `<branch name>`. To
              reference a pull request use `refs/pull/<number>/merge`.

          sarif_id: Filter analyses belonging to the same SARIF upload.

          sort: The property by which to sort the results.

          tool_guid: The GUID of a code scanning tool. Only results by this tool will be listed. Note
              that some code scanning tools may not include a GUID in their analysis data. You
              can specify the tool by using either `tool_guid` or `tool_name`, but not both.

          tool_name: The name of a code scanning tool. Only results by this tool will be listed. You
              can specify the tool by using either `tool_name` or `tool_guid`, but not both.

          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return self._get(
            f"/repos/{owner}/{repo}/code-scanning/analyses",
            options=make_request_options(
                extra_headers=extra_headers,
                extra_query=extra_query,
                extra_body=extra_body,
                timeout=timeout,
                query=maybe_transform(
                    {
                        "direction": direction,
                        "page": page,
                        "per_page": per_page,
                        "pr": pr,
                        "ref": ref,
                        "sarif_id": sarif_id,
                        "sort": sort,
                        "tool_guid": tool_guid,
                        "tool_name": tool_name,
                    },
                    analysis_list_params.AnalysisListParams,
                ),
            ),
            cast_to=AnalysisListResponse,
        )

    def delete(
        self,
        analysis_id: int,
        *,
        owner: str,
        repo: str,
        confirm_delete: str | None | NotGiven = NOT_GIVEN,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> AnalysisDeleteResponse:
        """
        Deletes a specified code scanning analysis from a repository.

        You can delete one analysis at a time. To delete a series of analyses, start
        with the most recent analysis and work backwards. Conceptually, the process is
        similar to the undo function in a text editor.

        When you list the analyses for a repository, one or more will be identified as
        deletable in the response:

        ```
        "deletable": true
        ```

        An analysis is deletable when it's the most recent in a set of analyses.
        Typically, a repository will have multiple sets of analyses for each enabled
        code scanning tool, where a set is determined by a unique combination of
        analysis values:

        - `ref`
        - `tool`
        - `category`

        If you attempt to delete an analysis that is not the most recent in a set,
        you'll get a 400 response with the message:

        ```
        Analysis specified is not deletable.
        ```

        The response from a successful `DELETE` operation provides you with two
        alternative URLs for deleting the next analysis in the set: `next_analysis_url`
        and `confirm_delete_url`. Use the `next_analysis_url` URL if you want to avoid
        accidentally deleting the final analysis in a set. This is a useful option if
        you want to preserve at least one analysis for the specified tool in your
        repository. Use the `confirm_delete_url` URL if you are content to remove all
        analyses for a tool. When you delete the last analysis in a set, the value of
        `next_analysis_url` and `confirm_delete_url` in the 200 response is `null`.

        As an example of the deletion process, let's imagine that you added a workflow
        that configured a particular code scanning tool to analyze the code in a
        repository. This tool has added 15 analyses: 10 on the default branch, and
        another 5 on a topic branch. You therefore have two separate sets of analyses
        for this tool. You've now decided that you want to remove all of the analyses
        for the tool. To do this you must make 15 separate deletion requests. To start,
        you must find an analysis that's identified as deletable. Each set of analyses
        always has one that's identified as deletable. Having found the deletable
        analysis for one of the two sets, delete this analysis and then continue
        deleting the next analysis in the set until they're all deleted. Then repeat the
        process for the second set. The procedure therefore consists of a nested loop:

        **Outer loop**:

        - List the analyses for the repository, filtered by tool.
        - Parse this list to find a deletable analysis. If found:

          **Inner loop**:

          - Delete the identified analysis.
          - Parse the response for the value of `confirm_delete_url` and, if found, use
            this in the next iteration.

        The above process assumes that you want to remove all trace of the tool's
        analyses from the GitHub user interface, for the specified repository, and it
        therefore uses the `confirm_delete_url` value. Alternatively, you could use the
        `next_analysis_url` value, which would leave the last analysis in each set
        undeleted to avoid removing a tool's analysis entirely.

        OAuth app tokens and personal access tokens (classic) need the `repo` scope to
        use this endpoint with private or public repositories, or the `public_repo`
        scope to use this endpoint with only public repositories.

        Args:
          confirm_delete: Allow deletion if the specified analysis is the last in a set. If you attempt to
              delete the final analysis in a set without setting this parameter to `true`,
              you'll get a 400 response with the message:
              `Analysis is last of its type and deletion may result in the loss of historical alert data. Please specify confirm_delete.`

          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return self._delete(
            f"/repos/{owner}/{repo}/code-scanning/analyses/{analysis_id}",
            options=make_request_options(
                extra_headers=extra_headers,
                extra_query=extra_query,
                extra_body=extra_body,
                timeout=timeout,
                query=maybe_transform({"confirm_delete": confirm_delete}, analysis_delete_params.AnalysisDeleteParams),
            ),
            cast_to=AnalysisDeleteResponse,
        )


class AsyncAnalysesResource(AsyncAPIResource):
    @cached_property
    def with_raw_response(self) -> AsyncAnalysesResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return AsyncAnalysesResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> AsyncAnalysesResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return AsyncAnalysesResourceWithStreamingResponse(self)

    async def retrieve(
        self,
        analysis_id: int,
        *,
        owner: str,
        repo: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> CodeScanningAnalysis:
        """
        Gets a specified code scanning analysis for a repository.

        The default JSON response contains fields that describe the analysis. This
        includes the Git reference and commit SHA to which the analysis relates, the
        datetime of the analysis, the name of the code scanning tool, and the number of
        alerts.

        The `rules_count` field in the default response give the number of rules that
        were run in the analysis. For very old analyses this data is not available, and
        `0` is returned in this field.

        This endpoint supports the following custom media types. For more information,
        see
        "[Media types](https://docs.github.com/rest/using-the-rest-api/getting-started-with-the-rest-api#media-types)."

        - **`application/sarif+json`**: Instead of returning a summary of the analysis,
          this endpoint returns a subset of the analysis data that was uploaded. The
          data is formatted as
          [SARIF version 2.1.0](https://docs.oasis-open.org/sarif/sarif/v2.1.0/cs01/sarif-v2.1.0-cs01.html).
          It also returns additional data such as the `github/alertNumber` and
          `github/alertUrl` properties.

        OAuth app tokens and personal access tokens (classic) need the `security_events`
        scope to use this endpoint with private or public repositories, or the
        `public_repo` scope to use this endpoint with only public repositories.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return await self._get(
            f"/repos/{owner}/{repo}/code-scanning/analyses/{analysis_id}",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=CodeScanningAnalysis,
        )

    async def list(
        self,
        repo: str,
        *,
        owner: str,
        direction: Literal["asc", "desc"] | NotGiven = NOT_GIVEN,
        page: int | NotGiven = NOT_GIVEN,
        per_page: int | NotGiven = NOT_GIVEN,
        pr: int | NotGiven = NOT_GIVEN,
        ref: str | NotGiven = NOT_GIVEN,
        sarif_id: str | NotGiven = NOT_GIVEN,
        sort: Literal["created"] | NotGiven = NOT_GIVEN,
        tool_guid: str | None | NotGiven = NOT_GIVEN,
        tool_name: str | NotGiven = NOT_GIVEN,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> AnalysisListResponse:
        """
        Lists the details of all code scanning analyses for a repository, starting with
        the most recent. The response is paginated and you can use the `page` and
        `per_page` parameters to list the analyses you're interested in. By default 30
        analyses are listed per page.

        The `rules_count` field in the response give the number of rules that were run
        in the analysis. For very old analyses this data is not available, and `0` is
        returned in this field.

        > [!WARNING] > **Closing down notice:** The `tool_name` field is closing down
        > and will, in future, not be included in the response for this endpoint. The
        > example response reflects this change. The tool name can now be found inside
        > the `tool` field.

        OAuth app tokens and personal access tokens (classic) need the `security_events`
        scope to use this endpoint with private or public repositories, or the
        `public_repo` scope to use this endpoint with only public repositories.

        Args:
          direction: The direction to sort the results by.

          page: The page number of the results to fetch. For more information, see
              "[Using pagination in the REST API](https://docs.github.com/rest/using-the-rest-api/using-pagination-in-the-rest-api)."

          per_page: The number of results per page (max 100). For more information, see
              "[Using pagination in the REST API](https://docs.github.com/rest/using-the-rest-api/using-pagination-in-the-rest-api)."

          pr: The number of the pull request for the results you want to list.

          ref: The Git reference for the analyses you want to list. The `ref` for a branch can
              be formatted either as `refs/heads/<branch name>` or simply `<branch name>`. To
              reference a pull request use `refs/pull/<number>/merge`.

          sarif_id: Filter analyses belonging to the same SARIF upload.

          sort: The property by which to sort the results.

          tool_guid: The GUID of a code scanning tool. Only results by this tool will be listed. Note
              that some code scanning tools may not include a GUID in their analysis data. You
              can specify the tool by using either `tool_guid` or `tool_name`, but not both.

          tool_name: The name of a code scanning tool. Only results by this tool will be listed. You
              can specify the tool by using either `tool_name` or `tool_guid`, but not both.

          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return await self._get(
            f"/repos/{owner}/{repo}/code-scanning/analyses",
            options=make_request_options(
                extra_headers=extra_headers,
                extra_query=extra_query,
                extra_body=extra_body,
                timeout=timeout,
                query=await async_maybe_transform(
                    {
                        "direction": direction,
                        "page": page,
                        "per_page": per_page,
                        "pr": pr,
                        "ref": ref,
                        "sarif_id": sarif_id,
                        "sort": sort,
                        "tool_guid": tool_guid,
                        "tool_name": tool_name,
                    },
                    analysis_list_params.AnalysisListParams,
                ),
            ),
            cast_to=AnalysisListResponse,
        )

    async def delete(
        self,
        analysis_id: int,
        *,
        owner: str,
        repo: str,
        confirm_delete: str | None | NotGiven = NOT_GIVEN,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> AnalysisDeleteResponse:
        """
        Deletes a specified code scanning analysis from a repository.

        You can delete one analysis at a time. To delete a series of analyses, start
        with the most recent analysis and work backwards. Conceptually, the process is
        similar to the undo function in a text editor.

        When you list the analyses for a repository, one or more will be identified as
        deletable in the response:

        ```
        "deletable": true
        ```

        An analysis is deletable when it's the most recent in a set of analyses.
        Typically, a repository will have multiple sets of analyses for each enabled
        code scanning tool, where a set is determined by a unique combination of
        analysis values:

        - `ref`
        - `tool`
        - `category`

        If you attempt to delete an analysis that is not the most recent in a set,
        you'll get a 400 response with the message:

        ```
        Analysis specified is not deletable.
        ```

        The response from a successful `DELETE` operation provides you with two
        alternative URLs for deleting the next analysis in the set: `next_analysis_url`
        and `confirm_delete_url`. Use the `next_analysis_url` URL if you want to avoid
        accidentally deleting the final analysis in a set. This is a useful option if
        you want to preserve at least one analysis for the specified tool in your
        repository. Use the `confirm_delete_url` URL if you are content to remove all
        analyses for a tool. When you delete the last analysis in a set, the value of
        `next_analysis_url` and `confirm_delete_url` in the 200 response is `null`.

        As an example of the deletion process, let's imagine that you added a workflow
        that configured a particular code scanning tool to analyze the code in a
        repository. This tool has added 15 analyses: 10 on the default branch, and
        another 5 on a topic branch. You therefore have two separate sets of analyses
        for this tool. You've now decided that you want to remove all of the analyses
        for the tool. To do this you must make 15 separate deletion requests. To start,
        you must find an analysis that's identified as deletable. Each set of analyses
        always has one that's identified as deletable. Having found the deletable
        analysis for one of the two sets, delete this analysis and then continue
        deleting the next analysis in the set until they're all deleted. Then repeat the
        process for the second set. The procedure therefore consists of a nested loop:

        **Outer loop**:

        - List the analyses for the repository, filtered by tool.
        - Parse this list to find a deletable analysis. If found:

          **Inner loop**:

          - Delete the identified analysis.
          - Parse the response for the value of `confirm_delete_url` and, if found, use
            this in the next iteration.

        The above process assumes that you want to remove all trace of the tool's
        analyses from the GitHub user interface, for the specified repository, and it
        therefore uses the `confirm_delete_url` value. Alternatively, you could use the
        `next_analysis_url` value, which would leave the last analysis in each set
        undeleted to avoid removing a tool's analysis entirely.

        OAuth app tokens and personal access tokens (classic) need the `repo` scope to
        use this endpoint with private or public repositories, or the `public_repo`
        scope to use this endpoint with only public repositories.

        Args:
          confirm_delete: Allow deletion if the specified analysis is the last in a set. If you attempt to
              delete the final analysis in a set without setting this parameter to `true`,
              you'll get a 400 response with the message:
              `Analysis is last of its type and deletion may result in the loss of historical alert data. Please specify confirm_delete.`

          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return await self._delete(
            f"/repos/{owner}/{repo}/code-scanning/analyses/{analysis_id}",
            options=make_request_options(
                extra_headers=extra_headers,
                extra_query=extra_query,
                extra_body=extra_body,
                timeout=timeout,
                query=await async_maybe_transform({"confirm_delete": confirm_delete}, analysis_delete_params.AnalysisDeleteParams),
            ),
            cast_to=AnalysisDeleteResponse,
        )


class AnalysesResourceWithRawResponse:
    def __init__(self, analyses: AnalysesResource) -> None:
        self._analyses = analyses

        self.retrieve = to_raw_response_wrapper(
            analyses.retrieve,
        )
        self.list = to_raw_response_wrapper(
            analyses.list,
        )
        self.delete = to_raw_response_wrapper(
            analyses.delete,
        )


class AsyncAnalysesResourceWithRawResponse:
    def __init__(self, analyses: AsyncAnalysesResource) -> None:
        self._analyses = analyses

        self.retrieve = async_to_raw_response_wrapper(
            analyses.retrieve,
        )
        self.list = async_to_raw_response_wrapper(
            analyses.list,
        )
        self.delete = async_to_raw_response_wrapper(
            analyses.delete,
        )


class AnalysesResourceWithStreamingResponse:
    def __init__(self, analyses: AnalysesResource) -> None:
        self._analyses = analyses

        self.retrieve = to_streamed_response_wrapper(
            analyses.retrieve,
        )
        self.list = to_streamed_response_wrapper(
            analyses.list,
        )
        self.delete = to_streamed_response_wrapper(
            analyses.delete,
        )


class AsyncAnalysesResourceWithStreamingResponse:
    def __init__(self, analyses: AsyncAnalysesResource) -> None:
        self._analyses = analyses

        self.retrieve = async_to_streamed_response_wrapper(
            analyses.retrieve,
        )
        self.list = async_to_streamed_response_wrapper(
            analyses.list,
        )
        self.delete = async_to_streamed_response_wrapper(
            analyses.delete,
        )
