from __future__ import annotations

import httpx

from ..._base_client import make_request_options
from ..._compat import cached_property
from ..._resource import AsyncAPIResource, SyncAPIResource
from ..._response import (
    async_to_raw_response_wrapper,
    async_to_streamed_response_wrapper,
    to_raw_response_wrapper,
    to_streamed_response_wrapper,
)
from ..._types import NOT_GIVEN, Body, Headers, NoneType, NotGiven, Query
from ...types.repos.private_vulnerability_reporting_check_enabled_response import (
    PrivateVulnerabilityReportingCheckEnabledResponse,
)

__all__ = ["PrivateVulnerabilityReportingResource", "AsyncPrivateVulnerabilityReportingResource"]


class PrivateVulnerabilityReportingResource(SyncAPIResource):
    @cached_property
    def with_raw_response(self) -> PrivateVulnerabilityReportingResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return PrivateVulnerabilityReportingResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> PrivateVulnerabilityReportingResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return PrivateVulnerabilityReportingResourceWithStreamingResponse(self)

    def check_enabled(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> PrivateVulnerabilityReportingCheckEnabledResponse:
        """
        Returns a boolean indicating whether or not private vulnerability reporting is
        enabled for the repository. For more information, see
        "[Evaluating the security settings of a repository](https://docs.github.com/code-security/security-advisories/working-with-repository-security-advisories/evaluating-the-security-settings-of-a-repository)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return self._get(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=PrivateVulnerabilityReportingCheckEnabledResponse,
        )

    def disable(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Disables private vulnerability reporting for a repository.

        The authenticated
        user must have admin access to the repository. For more information, see
        "[Privately reporting a security vulnerability](https://docs.github.com/code-security/security-advisories/guidance-on-reporting-and-writing/privately-reporting-a-security-vulnerability)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._delete(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    def enable(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Enables private vulnerability reporting for a repository.

        The authenticated user
        must have admin access to the repository. For more information, see
        "[Privately reporting a security vulnerability](https://docs.github.com/code-security/security-advisories/guidance-on-reporting-and-writing/privately-reporting-a-security-vulnerability)."

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._put(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class AsyncPrivateVulnerabilityReportingResource(AsyncAPIResource):
    @cached_property
    def with_raw_response(self) -> AsyncPrivateVulnerabilityReportingResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return AsyncPrivateVulnerabilityReportingResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> AsyncPrivateVulnerabilityReportingResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return AsyncPrivateVulnerabilityReportingResourceWithStreamingResponse(self)

    async def check_enabled(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> PrivateVulnerabilityReportingCheckEnabledResponse:
        """
        Returns a boolean indicating whether or not private vulnerability reporting is
        enabled for the repository. For more information, see
        "[Evaluating the security settings of a repository](https://docs.github.com/code-security/security-advisories/working-with-repository-security-advisories/evaluating-the-security-settings-of-a-repository)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        return await self._get(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=PrivateVulnerabilityReportingCheckEnabledResponse,
        )

    async def disable(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Disables private vulnerability reporting for a repository.

        The authenticated
        user must have admin access to the repository. For more information, see
        "[Privately reporting a security vulnerability](https://docs.github.com/code-security/security-advisories/guidance-on-reporting-and-writing/privately-reporting-a-security-vulnerability)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._delete(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    async def enable(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Enables private vulnerability reporting for a repository.

        The authenticated user
        must have admin access to the repository. For more information, see
        "[Privately reporting a security vulnerability](https://docs.github.com/code-security/security-advisories/guidance-on-reporting-and-writing/privately-reporting-a-security-vulnerability)."

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._put(
            f"/repos/{owner}/{repo}/private-vulnerability-reporting",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class PrivateVulnerabilityReportingResourceWithRawResponse:
    def __init__(self, private_vulnerability_reporting: PrivateVulnerabilityReportingResource) -> None:
        self._private_vulnerability_reporting = private_vulnerability_reporting

        self.check_enabled = to_raw_response_wrapper(
            private_vulnerability_reporting.check_enabled,
        )
        self.disable = to_raw_response_wrapper(
            private_vulnerability_reporting.disable,
        )
        self.enable = to_raw_response_wrapper(
            private_vulnerability_reporting.enable,
        )


class AsyncPrivateVulnerabilityReportingResourceWithRawResponse:
    def __init__(self, private_vulnerability_reporting: AsyncPrivateVulnerabilityReportingResource) -> None:
        self._private_vulnerability_reporting = private_vulnerability_reporting

        self.check_enabled = async_to_raw_response_wrapper(
            private_vulnerability_reporting.check_enabled,
        )
        self.disable = async_to_raw_response_wrapper(
            private_vulnerability_reporting.disable,
        )
        self.enable = async_to_raw_response_wrapper(
            private_vulnerability_reporting.enable,
        )


class PrivateVulnerabilityReportingResourceWithStreamingResponse:
    def __init__(self, private_vulnerability_reporting: PrivateVulnerabilityReportingResource) -> None:
        self._private_vulnerability_reporting = private_vulnerability_reporting

        self.check_enabled = to_streamed_response_wrapper(
            private_vulnerability_reporting.check_enabled,
        )
        self.disable = to_streamed_response_wrapper(
            private_vulnerability_reporting.disable,
        )
        self.enable = to_streamed_response_wrapper(
            private_vulnerability_reporting.enable,
        )


class AsyncPrivateVulnerabilityReportingResourceWithStreamingResponse:
    def __init__(self, private_vulnerability_reporting: AsyncPrivateVulnerabilityReportingResource) -> None:
        self._private_vulnerability_reporting = private_vulnerability_reporting

        self.check_enabled = async_to_streamed_response_wrapper(
            private_vulnerability_reporting.check_enabled,
        )
        self.disable = async_to_streamed_response_wrapper(
            private_vulnerability_reporting.disable,
        )
        self.enable = async_to_streamed_response_wrapper(
            private_vulnerability_reporting.enable,
        )
