from __future__ import annotations

import httpx

from ...._base_client import make_request_options
from ...._compat import cached_property
from ...._resource import AsyncAPIResource, SyncAPIResource
from ...._response import (
    async_to_raw_response_wrapper,
    async_to_streamed_response_wrapper,
    to_raw_response_wrapper,
    to_streamed_response_wrapper,
)
from ...._types import NOT_GIVEN, Body, Headers, NoneType, NotGiven, Query

__all__ = ["SuspendedResource", "AsyncSuspendedResource"]


class SuspendedResource(SyncAPIResource):
    @cached_property
    def with_raw_response(self) -> SuspendedResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return SuspendedResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> SuspendedResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return SuspendedResourceWithStreamingResponse(self)

    def suspend(
        self,
        installation_id: int,
        *,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """
        Suspends a GitHub App on a user, organization, or business account, which blocks
        the app from accessing the account's resources. When a GitHub App is suspended,
        the app's access to the GitHub API or webhook events is blocked for that
        account.

        You must use a
        [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app)
        to access this endpoint.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._put(
            f"/app/installations/{installation_id}/suspended",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    def unsuspend(
        self,
        installation_id: int,
        *,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """
        Removes a GitHub App installation suspension.

        You must use a
        [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app)
        to access this endpoint.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._delete(
            f"/app/installations/{installation_id}/suspended",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class AsyncSuspendedResource(AsyncAPIResource):
    @cached_property
    def with_raw_response(self) -> AsyncSuspendedResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return AsyncSuspendedResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> AsyncSuspendedResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return AsyncSuspendedResourceWithStreamingResponse(self)

    async def suspend(
        self,
        installation_id: int,
        *,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """
        Suspends a GitHub App on a user, organization, or business account, which blocks
        the app from accessing the account's resources. When a GitHub App is suspended,
        the app's access to the GitHub API or webhook events is blocked for that
        account.

        You must use a
        [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app)
        to access this endpoint.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._put(
            f"/app/installations/{installation_id}/suspended",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    async def unsuspend(
        self,
        installation_id: int,
        *,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """
        Removes a GitHub App installation suspension.

        You must use a
        [JWT](https://docs.github.com/apps/building-github-apps/authenticating-with-github-apps/#authenticating-as-a-github-app)
        to access this endpoint.

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._delete(
            f"/app/installations/{installation_id}/suspended",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class SuspendedResourceWithRawResponse:
    def __init__(self, suspended: SuspendedResource) -> None:
        self._suspended = suspended

        self.suspend = to_raw_response_wrapper(
            suspended.suspend,
        )
        self.unsuspend = to_raw_response_wrapper(
            suspended.unsuspend,
        )


class AsyncSuspendedResourceWithRawResponse:
    def __init__(self, suspended: AsyncSuspendedResource) -> None:
        self._suspended = suspended

        self.suspend = async_to_raw_response_wrapper(
            suspended.suspend,
        )
        self.unsuspend = async_to_raw_response_wrapper(
            suspended.unsuspend,
        )


class SuspendedResourceWithStreamingResponse:
    def __init__(self, suspended: SuspendedResource) -> None:
        self._suspended = suspended

        self.suspend = to_streamed_response_wrapper(
            suspended.suspend,
        )
        self.unsuspend = to_streamed_response_wrapper(
            suspended.unsuspend,
        )


class AsyncSuspendedResourceWithStreamingResponse:
    def __init__(self, suspended: AsyncSuspendedResource) -> None:
        self._suspended = suspended

        self.suspend = async_to_streamed_response_wrapper(
            suspended.suspend,
        )
        self.unsuspend = async_to_streamed_response_wrapper(
            suspended.unsuspend,
        )
