from __future__ import annotations

import httpx

from ..._base_client import make_request_options
from ..._compat import cached_property
from ..._resource import AsyncAPIResource, SyncAPIResource
from ..._response import (
    async_to_raw_response_wrapper,
    async_to_streamed_response_wrapper,
    to_raw_response_wrapper,
    to_streamed_response_wrapper,
)
from ..._types import NOT_GIVEN, Body, Headers, NoneType, NotGiven, Query

__all__ = ["VulnerabilityAlertsResource", "AsyncVulnerabilityAlertsResource"]


class VulnerabilityAlertsResource(SyncAPIResource):
    @cached_property
    def with_raw_response(self) -> VulnerabilityAlertsResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return VulnerabilityAlertsResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> VulnerabilityAlertsResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return VulnerabilityAlertsResourceWithStreamingResponse(self)

    def check_alerts_enabled(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Shows whether dependency alerts are enabled or disabled for a repository.

        The
        authenticated user must have admin read access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._get(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    def disable_alerts(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Disables dependency alerts and the dependency graph for a repository.

        The
        authenticated user must have admin access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._delete(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    def enable_alerts(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Enables dependency alerts and the dependency graph for a repository.

        The
        authenticated user must have admin access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return self._put(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class AsyncVulnerabilityAlertsResource(AsyncAPIResource):
    @cached_property
    def with_raw_response(self) -> AsyncVulnerabilityAlertsResourceWithRawResponse:
        """
        This property can be used as a prefix for any HTTP method call to return
        the raw response object instead of the parsed content.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#accessing-raw-response-data-eg-headers
        """
        return AsyncVulnerabilityAlertsResourceWithRawResponse(self)

    @cached_property
    def with_streaming_response(self) -> AsyncVulnerabilityAlertsResourceWithStreamingResponse:
        """
        An alternative to `.with_raw_response` that doesn't eagerly read the response body.

        For more information, see https://www.github.com/stainless-sdks/github_api_sdk-python#with_streaming_response
        """
        return AsyncVulnerabilityAlertsResourceWithStreamingResponse(self)

    async def check_alerts_enabled(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Shows whether dependency alerts are enabled or disabled for a repository.

        The
        authenticated user must have admin read access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._get(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    async def disable_alerts(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Disables dependency alerts and the dependency graph for a repository.

        The
        authenticated user must have admin access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._delete(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )

    async def enable_alerts(
        self,
        repo: str,
        *,
        owner: str,
        # Use the following arguments if you need to pass additional parameters to the API that aren't available via kwargs.
        # The extra values given here take precedence over values defined on the client or passed to this method.
        extra_headers: Headers | None = None,
        extra_query: Query | None = None,
        extra_body: Body | None = None,
        timeout: float | httpx.Timeout | None | NotGiven = NOT_GIVEN,
    ) -> None:
        """Enables dependency alerts and the dependency graph for a repository.

        The
        authenticated user must have admin access to the repository. For more
        information, see
        "[About security alerts for vulnerable dependencies](https://docs.github.com/articles/about-security-alerts-for-vulnerable-dependencies)".

        Args:
          extra_headers: Send extra headers

          extra_query: Add additional query parameters to the request

          extra_body: Add additional JSON properties to the request

          timeout: Override the client-level default timeout for this request, in seconds
        """
        if not owner:
            raise ValueError(f"Expected a non-empty value for `owner` but received {owner!r}")
        if not repo:
            raise ValueError(f"Expected a non-empty value for `repo` but received {repo!r}")
        extra_headers = {"Accept": "*/*", **(extra_headers or {})}
        return await self._put(
            f"/repos/{owner}/{repo}/vulnerability-alerts",
            options=make_request_options(extra_headers=extra_headers, extra_query=extra_query, extra_body=extra_body, timeout=timeout),
            cast_to=NoneType,
        )


class VulnerabilityAlertsResourceWithRawResponse:
    def __init__(self, vulnerability_alerts: VulnerabilityAlertsResource) -> None:
        self._vulnerability_alerts = vulnerability_alerts

        self.check_alerts_enabled = to_raw_response_wrapper(
            vulnerability_alerts.check_alerts_enabled,
        )
        self.disable_alerts = to_raw_response_wrapper(
            vulnerability_alerts.disable_alerts,
        )
        self.enable_alerts = to_raw_response_wrapper(
            vulnerability_alerts.enable_alerts,
        )


class AsyncVulnerabilityAlertsResourceWithRawResponse:
    def __init__(self, vulnerability_alerts: AsyncVulnerabilityAlertsResource) -> None:
        self._vulnerability_alerts = vulnerability_alerts

        self.check_alerts_enabled = async_to_raw_response_wrapper(
            vulnerability_alerts.check_alerts_enabled,
        )
        self.disable_alerts = async_to_raw_response_wrapper(
            vulnerability_alerts.disable_alerts,
        )
        self.enable_alerts = async_to_raw_response_wrapper(
            vulnerability_alerts.enable_alerts,
        )


class VulnerabilityAlertsResourceWithStreamingResponse:
    def __init__(self, vulnerability_alerts: VulnerabilityAlertsResource) -> None:
        self._vulnerability_alerts = vulnerability_alerts

        self.check_alerts_enabled = to_streamed_response_wrapper(
            vulnerability_alerts.check_alerts_enabled,
        )
        self.disable_alerts = to_streamed_response_wrapper(
            vulnerability_alerts.disable_alerts,
        )
        self.enable_alerts = to_streamed_response_wrapper(
            vulnerability_alerts.enable_alerts,
        )


class AsyncVulnerabilityAlertsResourceWithStreamingResponse:
    def __init__(self, vulnerability_alerts: AsyncVulnerabilityAlertsResource) -> None:
        self._vulnerability_alerts = vulnerability_alerts

        self.check_alerts_enabled = async_to_streamed_response_wrapper(
            vulnerability_alerts.check_alerts_enabled,
        )
        self.disable_alerts = async_to_streamed_response_wrapper(
            vulnerability_alerts.disable_alerts,
        )
        self.enable_alerts = async_to_streamed_response_wrapper(
            vulnerability_alerts.enable_alerts,
        )
