"""
Phase 3: Contract Tests (TypeScript ↔ Python)

Tests that validate:
1. TypeScript client expectations match Python server output
2. Response format parsing works correctly
3. Tool schemas match between client and server
"""

import pytest
import re
from pathlib import Path
from typing import List, Set

# Import the MCP server
import sys

project_root = Path(__file__).parent.parent
sys.path.insert(0, str(project_root))

from src.github_mcp.tools import __all__ as all_tool_names  # noqa: E402
from src.github_mcp import tools as tools_module  # noqa: E402
import inspect  # noqa: E402


def read_typescript_tool_list() -> List[str]:
    """Read READ_TOOLS_WITH_JSON_SUPPORT from client-deno.ts."""
    client_deno_path = project_root / "src" / "github_mcp" / "servers" / "client-deno.ts"

    if not client_deno_path.exists():
        return []

    content = client_deno_path.read_text(encoding="utf-8")

    # Extract the array
    match = re.search(
        r"const READ_TOOLS_WITH_JSON_SUPPORT = \[(.*?)\];", content, re.DOTALL
    )
    if not match:
        return []

    tools = []
    for line in match.group(1).split("\n"):
        line = line.strip()
        # Skip empty lines and comments
        if not line or line.startswith("//"):
            continue
        # Extract tool name using regex: match 'tool_name' or "tool_name" with optional trailing comma
        tool_match = re.search(r"['\"]([^'\"]+)['\"]", line)
        if tool_match:
            tool_name = tool_match.group(1).strip()
            if tool_name:
                tools.append(tool_name)

    return tools


def get_python_tools_with_response_format() -> Set[str]:
    """Get all Python tools that support response_format."""
    tools = set()

    # Get all tools from tools module
    for tool_name in all_tool_names:
        if hasattr(tools_module, tool_name):
            tool_func = getattr(tools_module, tool_name)
            if inspect.iscoroutinefunction(tool_func) or inspect.isfunction(tool_func):
                sig = inspect.signature(tool_func)
                params = sig.parameters

                if "params" in params:
                    param_type = params["params"].annotation
                    if inspect.isclass(param_type):
                        if hasattr(param_type, "model_fields"):
                            fields = param_type.model_fields
                            if "response_format" in fields:
                                tools.add(tool_name)

    return tools


class TestClientServerContract:
    """Test that TypeScript client and Python server agree on contracts."""

    def test_typescript_client_matches_python_server(self):
        """Verify TypeScript client expectations match Python server output."""
        typescript_tools = set(read_typescript_tool_list())
        python_tools = get_python_tools_with_response_format()

        # Tools in TypeScript should be in Python
        missing_in_python = typescript_tools - python_tools

        if missing_in_python:
            pytest.fail(
                "Tools in READ_TOOLS_WITH_JSON_SUPPORT but don't support response_format in Python:\n"
                + "\n".join(f"  - {tool}" for tool in sorted(missing_in_python))
                + "\n\nNote: If workspace tools are missing, they may need to be implemented."
            )

        # All tools match (verified by assertion above)

    def test_python_tools_in_typescript_list(self):
        """Verify Python tools with response_format are in TypeScript list."""
        typescript_tools = set(read_typescript_tool_list())
        python_tools = get_python_tools_with_response_format()

        # Tools in Python should be in TypeScript (or at least documented)
        missing_in_typescript = python_tools - typescript_tools

        if missing_in_typescript:
            # Note: Some tools may be intentionally excluded
            for tool in sorted(missing_in_typescript):
                print(f"  - {tool}")
            print(
                "  (This is a warning, not an error - tools may be intentionally excluded)"
            )

    def test_response_format_enum_consistency(self):
        """Verify response_format uses consistent enum values."""
        from src.github_mcp.models import ResponseFormat

        # Check enum values
        assert hasattr(ResponseFormat, "JSON"), "ResponseFormat should have JSON value"
        assert hasattr(ResponseFormat, "MARKDOWN"), (
            "ResponseFormat should have MARKDOWN value"
        )

        # Verify enum values match expected strings
        assert ResponseFormat.JSON == "json", "ResponseFormat.JSON should be 'json'"
        assert ResponseFormat.MARKDOWN == "markdown", (
            "ResponseFormat.MARKDOWN should be 'markdown'"
        )

        # ResponseFormat enum is consistent (verified by assertions above)


class TestResponseFormatParsing:
    """Test response format parsing works correctly."""

    def test_json_response_structure(self):
        """Verify JSON responses have expected structure."""
        # This would require actual API calls or mocking
        # For now, verify the pattern exists in code

        # Check that tools with response_format='json' return parseable data
        tools_with_json = read_typescript_tool_list()

        assert len(tools_with_json) > 0, (
            "Should have tools that support JSON response format"
        )

        # Tools support JSON response format (verified by assertion above)

    def test_markdown_response_structure(self):
        """Verify Markdown responses have expected structure."""
        # Markdown responses should be strings
        # This is verified by the fact that response_format is optional
        # and defaults to markdown for most tools

        # Markdown response format is supported


if __name__ == "__main__":
    pytest.main([__file__, "-v"])
