"""Log parsing and filtering functionality for webhook server logs."""

import asyncio
import datetime
import re
from collections.abc import AsyncGenerator
from dataclasses import dataclass
from pathlib import Path
from typing import Any

from simple_logger.logger import get_logger


@dataclass
class LogEntry:
    """Represents a parsed log entry with structured data."""

    timestamp: datetime.datetime
    level: str
    logger_name: str
    message: str
    hook_id: str | None = None
    event_type: str | None = None
    repository: str | None = None
    pr_number: int | None = None
    github_user: str | None = None
    task_id: str | None = None
    task_type: str | None = None
    task_status: str | None = None
    token_spend: int | None = None

    def to_dict(self) -> dict[str, Any]:
        """Convert LogEntry to dictionary for JSON serialization."""
        return {
            "timestamp": self.timestamp.isoformat(),
            "level": self.level,
            "logger_name": self.logger_name,
            "message": self.message,
            "hook_id": self.hook_id,
            "event_type": self.event_type,
            "repository": self.repository,
            "pr_number": self.pr_number,
            "github_user": self.github_user,
            "task_id": self.task_id,
            "task_type": self.task_type,
            "task_status": self.task_status,
            "token_spend": self.token_spend,
        }


class LogParser:
    """Parser for webhook server log files.

    Parses logs generated by GithubWebhook.prepare_log_prefix() function which creates
    structured log prefixes for webhook processing.

    Log files are typically stored in the configured data directory under a 'logs' subdirectory.
    """

    def __init__(self) -> None:
        """Initialize LogParser with logger."""
        self.logger = get_logger(name="log_parser")

    # Regex pattern for parsing production logs from prepare_log_prefix() in github_api.py
    # Format from prepare_log_prefix():
    #   With PR: "{colored_repo} [{event}][{delivery_id}][{user}][PR {number}]: {message}"
    #   Without PR: "{colored_repo} [{event}][{delivery_id}][{user}]: {message}"
    # Full log format: "timestamp logger level colored_repo [event][delivery_id][user][PR number]: message"
    # Example: "2025-07-31T10:30:00.123000 GithubWebhook INFO repo-name
    #           [pull_request][abc123][user][PR 123]: Processing webhook"
    # Supports:
    #   - Optional fractional seconds
    #   - Optional timezone (Z or ±HH:MM format, e.g., +00:00, -05:00)
    #   - Flexible whitespace between fields
    #   - Logger names with dots/hyphens
    LOG_PATTERN = re.compile(
        r"^(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(?:\.\d+)?(?:Z|[+-]\d{2}:\d{2})?)\s+"
        r"([\w.-]+)\s+(?:\x1b\[[\d;]*m)?([\w.-]+)(?:\x1b\[[\d;]*m)?\s+(.+)$"
    )

    # Pattern to extract GitHub context from prepare_log_prefix format
    # Matches: colored_repo [event][delivery_id][user][PR number]: message
    GITHUB_CONTEXT_PATTERN = re.compile(
        r"(?:\x1b\[[0-9;]*m)?([^\x1b\[\s]+)(?:\x1b\[[0-9;]*m)? "
        r"\[([^\]]+)\]\[([^\]]+)\]\[([^\]]+)\](?:\[PR (\d+)\])?: (.+)"
    )

    ANSI_ESCAPE_PATTERN = re.compile(r"\x1b\[[0-9;]*m")

    # Precompiled patterns for task field extraction (performance optimization)
    # Handle escaped brackets: (?:\\.|[^\]])+ matches escaped chars or non-] chars
    TASK_ID_PATTERN = re.compile(r"\[task_id=((?:\\.|[^\]])+)\]")
    TASK_TYPE_PATTERN = re.compile(r"\[task_type=((?:\\.|[^\]])+)\]")
    TASK_STATUS_PATTERN = re.compile(r"\[task_status=((?:\\.|[^\]])+)\]")
    # Pattern for token spend: handles both original and masked formats
    # Original: "Token spend: 35 API calls"
    # Masked: "token ***** 35 API calls" (when "token" is redacted by secret masking)
    TOKEN_SPEND_PATTERN = re.compile(r"(?:Token spend|token\s+\*+)\s*:?\s*(\d+)\s+API calls")

    def is_workflow_step(self, entry: LogEntry) -> bool:
        """
        Check if a log entry is a workflow milestone step.

        Only entries with task_id AND task_status are considered workflow milestones.
        This filters out internal/initialization steps and only shows meaningful business events.

        Args:
            entry: LogEntry to check

        Returns:
            True if this is a workflow milestone entry (has task_id and task_status)
        """
        return bool(entry.task_id and entry.task_status)

    def extract_workflow_steps(self, entries: list[LogEntry], hook_id: str) -> list[LogEntry]:
        """
        Extract workflow step entries for a specific hook ID.

        Args:
            entries: List of log entries to filter
            hook_id: Hook ID to filter by

        Returns:
            List of workflow step entries for the specified hook ID
        """
        return [entry for entry in entries if entry.hook_id == hook_id and self.is_workflow_step(entry)]

    def parse_log_entry(self, log_line: str) -> LogEntry | None:
        """
        Parse a single log line into a LogEntry object.

        Args:
            log_line: Raw log line string

        Returns:
            LogEntry object if parsing successful, None otherwise
        """
        if not log_line.strip():
            return None

        # Parse production log format
        match = self.LOG_PATTERN.match(log_line.strip())
        if not match:
            return None

        timestamp_str, logger_name, level, message = match.groups()

        # Parse ISO timestamp format: "2025-07-31T10:30:00.123000"
        # Handle 'Z' timezone suffix which fromisoformat doesn't accept
        try:
            normalized_timestamp = timestamp_str.replace("Z", "+00:00")
            timestamp = datetime.datetime.fromisoformat(normalized_timestamp)
            # Ensure timestamp is timezone-aware (assume UTC if naive)
            if timestamp.tzinfo is None:
                timestamp = timestamp.replace(tzinfo=datetime.UTC)
        except ValueError:
            return None

        # Extract GitHub webhook context from prepare_log_prefix format
        repository, event_type, hook_id, github_user, pr_number, cleaned_message = self._extract_github_context(message)

        # Extract task correlation fields from message and strip them from the message
        task_id, task_type, task_status, final_message = self._extract_task_fields(cleaned_message)

        # Extract token spend from message
        token_spend = self.extract_token_spend(final_message)

        return LogEntry(
            timestamp=timestamp,
            level=level,
            logger_name=logger_name,
            message=final_message,
            hook_id=hook_id,
            event_type=event_type,
            repository=repository,
            pr_number=pr_number,
            github_user=github_user,
            task_id=task_id,
            task_type=task_type,
            task_status=task_status,
            token_spend=token_spend,
        )

    def _extract_github_context(
        self, message: str
    ) -> tuple[str | None, str | None, str | None, str | None, int | None, str]:
        """Extract GitHub context from prepare_log_prefix format.

        Returns:
            Tuple of (repository, event_type, hook_id, github_user, pr_number, cleaned_message)
        """
        match = self.GITHUB_CONTEXT_PATTERN.search(message)
        if match:
            repository = match.group(1)
            event_type = match.group(2)
            hook_id = match.group(3)
            github_user = match.group(4)
            pr_number_str = match.group(5)  # Optional PR number
            cleaned_message = match.group(6)

            # Parse PR number if present
            pr_number = None
            if pr_number_str:
                try:
                    pr_number = int(pr_number_str)
                except ValueError:
                    pass

            # Clean ANSI codes from message
            cleaned_message = self.ANSI_ESCAPE_PATTERN.sub("", cleaned_message)

            return repository, event_type, hook_id, github_user, pr_number, cleaned_message

        # No GitHub context found, return original message cleaned of ANSI codes
        cleaned_message = self.ANSI_ESCAPE_PATTERN.sub("", message)
        return None, None, None, None, None, cleaned_message

    def _extract_task_fields(self, message: str) -> tuple[str | None, str | None, str | None, str]:
        """Extract task correlation fields from log message.

        Extracts task_id, task_type, and task_status from patterns like:
        [task_id=check_tox] [task_type=ci_check] [task_status=started]

        The task tokens are removed from the returned message to avoid duplication
        and improve free-text search, as these values are stored in dedicated fields.

        Args:
            message: Log message to extract from

        Returns:
            Tuple of (task_id, task_type, task_status, cleaned_message)
        """
        task_id = None
        task_type = None
        task_status = None
        cleaned_message = message

        # Extract task_id using precompiled pattern
        if task_id_match := self.TASK_ID_PATTERN.search(cleaned_message):
            task_id = self._unescape_task_value(task_id_match.group(1))
            cleaned_message = self.TASK_ID_PATTERN.sub("", cleaned_message, count=1).strip()

        # Extract task_type using precompiled pattern
        if task_type_match := self.TASK_TYPE_PATTERN.search(cleaned_message):
            task_type = self._unescape_task_value(task_type_match.group(1))
            cleaned_message = self.TASK_TYPE_PATTERN.sub("", cleaned_message, count=1).strip()

        # Extract task_status using precompiled pattern
        if task_status_match := self.TASK_STATUS_PATTERN.search(cleaned_message):
            task_status = self._unescape_task_value(task_status_match.group(1))
            cleaned_message = self.TASK_STATUS_PATTERN.sub("", cleaned_message, count=1).strip()

        return task_id, task_type, task_status, cleaned_message

    @staticmethod
    def _unescape_task_value(value: str) -> str:
        """Unescape brackets in task field values."""
        return value.replace("\\]", "]").replace("\\[", "[")

    def extract_token_spend(self, message: str) -> int | None:
        """Extract token spend from log message.

        Parses messages like:
        - "Token spend: 35 API calls (initial: 2831, final: 2796, remaining: 2796)"
        - "token ***** 35 API calls (initial: 2831, final: 2796, remaining: 2796)" (when masked)

        Args:
            message: Log message to extract from

        Returns:
            Token spend as integer, or None if not found
        """
        match = self.TOKEN_SPEND_PATTERN.search(message)
        if match:
            try:
                return int(match.group(1))
            except ValueError:
                return None
        return None

    def parse_log_file(self, file_path: Path) -> list[LogEntry]:
        """
        Parse an entire log file and return list of LogEntry objects.

        Args:
            file_path: Path to the log file

        Returns:
            List of successfully parsed LogEntry objects
        """
        entries: list[LogEntry] = []
        total_lines = 0
        failed_lines = 0

        try:
            with open(file_path, encoding="utf-8") as f:
                for _line_num, line in enumerate(f, 1):
                    total_lines += 1
                    entry = self.parse_log_entry(line)
                    if entry:
                        entries.append(entry)
                    else:
                        failed_lines += 1

        except OSError as e:
            self.logger.error(f"Failed to read log file {file_path}: {e}")
        except UnicodeDecodeError as e:
            self.logger.error(f"Failed to decode log file {file_path}: {e}")

        return entries

    async def tail_log_file(self, file_path: Path, follow: bool = True) -> AsyncGenerator[LogEntry]:
        """
        Tail a log file and yield new LogEntry objects as they are added.

        Args:
            file_path: Path to the log file to monitor
            follow: Whether to continue monitoring for new entries

        Yields:
            LogEntry objects for new log lines
        """
        # Start from the end of the file
        if not file_path.exists():
            return

        with open(file_path, encoding="utf-8") as f:
            # Move to end of file
            f.seek(0, 2)

            while True:
                line = f.readline()
                if line:
                    entry = self.parse_log_entry(line)
                    if entry:
                        yield entry
                elif follow:
                    # No new data, wait a bit before checking again
                    await asyncio.sleep(0.1)
                else:
                    # Not following, exit when no more data
                    break

    async def monitor_log_directory(self, log_dir: Path, pattern: str = "*.log") -> AsyncGenerator[LogEntry]:
        """
        Monitor a directory for log files and yield new entries from all files.

        Args:
            log_dir: Directory path containing log files
            pattern: Glob pattern for log files (default: "*.log")

        Yields:
            LogEntry objects from all monitored log files
        """
        if not log_dir.exists() or not log_dir.is_dir():
            return

        # Find all existing log files including rotated ones
        log_files: list[Path] = []
        log_files.extend(log_dir.glob(pattern))
        # Only monitor current log file, not rotated ones for real-time
        current_log_files = [
            f for f in log_files if not any(f.name.endswith(ext) for ext in [".1", ".2", ".3", ".4", ".5"])
        ]

        if not current_log_files:
            return

        # Monitor the most recent current log file (not rotated)
        # Sort by modification time to get the most recent file
        current_log_files.sort(key=lambda f: f.stat().st_mtime, reverse=True)
        most_recent_file = current_log_files[0]

        async for entry in self.tail_log_file(most_recent_file, follow=True):
            yield entry


class LogFilter:
    """Filter log entries based on various criteria."""

    def filter_entries(
        self,
        entries: list[LogEntry],
        hook_id: str | None = None,
        pr_number: int | None = None,
        repository: str | None = None,
        event_type: str | None = None,
        github_user: str | None = None,
        level: str | None = None,
        start_time: datetime.datetime | None = None,
        end_time: datetime.datetime | None = None,
        search_text: str | None = None,
        limit: int | None = None,
        offset: int | None = None,
    ) -> list[LogEntry]:
        """
        Filter log entries based on provided criteria.

        Args:
            entries: List of LogEntry objects to filter
            hook_id: Filter by exact hook ID match
            pr_number: Filter by exact PR number match
            repository: Filter by exact repository match
            event_type: Filter by exact event type match
            github_user: Filter by exact GitHub user match
            level: Filter by exact log level match
            start_time: Filter entries after this timestamp
            end_time: Filter entries before this timestamp
            search_text: Filter by text search in message (case-insensitive)
            limit: Maximum number of entries to return
            offset: Number of entries to skip (for pagination)

        Returns:
            Filtered list of LogEntry objects
        """
        filtered = entries[:]

        # Apply filters
        if hook_id is not None:
            filtered = [e for e in filtered if e.hook_id == hook_id]

        if pr_number is not None:
            filtered = [e for e in filtered if e.pr_number == pr_number]

        if repository is not None:
            filtered = [e for e in filtered if e.repository == repository]

        if event_type is not None:
            filtered = [e for e in filtered if e.event_type == event_type]

        if github_user is not None:
            filtered = [e for e in filtered if e.github_user == github_user]

        if level is not None:
            filtered = [e for e in filtered if e.level == level]

        if start_time is not None:
            filtered = [e for e in filtered if e.timestamp >= start_time]

        if end_time is not None:
            filtered = [e for e in filtered if e.timestamp <= end_time]

        if search_text is not None:
            search_lower = search_text.lower()
            filtered = [e for e in filtered if search_lower in e.message.lower()]

        # Apply pagination
        if offset is not None:
            filtered = filtered[offset:]

        if limit is not None:
            filtered = filtered[:limit]

        return filtered

    def get_unique_values(self, entries: list[LogEntry], field: str) -> list[str]:
        """
        Get unique values for a specific field across all entries.

        Args:
            entries: List of LogEntry objects
            field: Field name to get unique values for

        Returns:
            List of unique non-None values for the specified field
        """
        values = set()
        for entry in entries:
            value = getattr(entry, field, None)
            if value is not None:
                values.add(str(value))
        return sorted(list(values))

    def get_entry_count_by_field(self, entries: list[LogEntry], field: str) -> dict[str, int]:
        """
        Get count of entries grouped by a specific field.

        Args:
            entries: List of LogEntry objects
            field: Field name to group by

        Returns:
            Dictionary mapping field values to entry counts
        """
        counts: dict[str, int] = {}
        for entry in entries:
            value = getattr(entry, field, None)
            if value is not None:
                key = str(value)
                counts[key] = counts.get(key, 0) + 1
        return counts
