from typing import Dict, List, Any, Optional
import re
from dataclasses import dataclass
from collections import defaultdict

@dataclass
class CodeChange:
    file_path: str
    change_type: str  # 'addition', 'deletion', 'modification'
    impact_score: float
    complexity_change: float
    dependencies_affected: List[str]
    api_changes: List[str]
    test_coverage_change: float
    documentation_changes: bool
    security_impact: float
    performance_impact: float

class CodeAnalyzer:
    def __init__(self):
        self.patterns = {
            'api_change': r'(def|class)\s+(\w+)',
            'dependency': r'(import|from)\s+(\w+)',
            'security': r'(password|token|key|auth|security)',
            'performance': r'(optimize|performance|speed|efficiency)',
            'test': r'(test|assert|mock)',
            'documentation': r'(docstring|comment|documentation)'
        }

    def analyze_commit(self, commit: Dict[str, Any]) -> Dict[str, Any]:
        """Analyze a single commit for deeper insights."""
        changes = []
        # Use the new file_changes structure from enhanced_git_analyzer
        file_changes = commit.get('file_changes', [])
        
        # Debug: Print file changes structure for first few commits
        if len(changes) < 3:  # Only debug first 3 commits
            print(f"DEBUG: Commit {commit.get('hash', 'unknown')[:8]} has {len(file_changes)} file changes")
            for i, fc in enumerate(file_changes[:2]):  # Show first 2 file changes
                print(f"  File {i}: {fc.get('file_path', 'unknown')} - Patch: {'Yes' if fc.get('patch') else 'No'}")
        
        for file_change in file_changes:
            change = self._analyze_file_change(file_change)
            if change:
                changes.append(change)
        
        return {
            'changes': changes,
            'patterns': self._identify_patterns(commit),
            'dependencies': self._analyze_dependencies(commit),
            'api_changes': self._analyze_api_changes(commit)
        }

    def _analyze_file_change(self, file_change: Dict[str, Any]) -> Optional[CodeChange]:
        """Analyze changes in a single file."""
        # Get patch content, handle both string and bytes
        patch = file_change.get('patch')
        if patch is None:
            # If no patch, try to analyze based on file path and change type
            return self._analyze_file_without_patch(file_change)
        
        # Convert patch to string if it's bytes
        if isinstance(patch, bytes):
            patch = patch.decode('utf-8', errors='ignore')
        patch = str(patch)
        
        # Analyze the patch content
        api_changes = self._extract_api_changes(patch)
        dependencies = self._extract_dependencies(patch)
        
        return CodeChange(
            file_path=file_change['file_path'],
            change_type=self._determine_change_type(file_change),
            impact_score=0.0,  # Removed scoring
            complexity_change=0.0,  # Removed scoring
            dependencies_affected=dependencies,
            api_changes=api_changes,
            test_coverage_change=0.0,  # Removed scoring
            documentation_changes=self._has_documentation_changes(patch),
            security_impact=0.0,  # Removed scoring
            performance_impact=0.0  # Removed scoring
        )

    def _extract_api_changes(self, patch: str) -> List[str]:
        """Extract API changes from the patch."""
        api_changes = []
        
        # Look for function and class definitions
        for line in patch.split('\n'):
            if line.startswith('+') or line.startswith('-'):
                match = re.search(self.patterns['api_change'], line)
                if match:
                    api_changes.append(match.group(2))
                    
        return api_changes

    def _extract_dependencies(self, patch: str) -> List[str]:
        """Extract dependency changes from the patch."""
        dependencies = []
        
        for line in patch.split('\n'):
            if line.startswith('+') or line.startswith('-'):
                match = re.search(self.patterns['dependency'], line)
                if match:
                    dependencies.append(match.group(2))
                    
        return dependencies

    def _has_documentation_changes(self, patch: str) -> bool:
        """Check if there are documentation changes."""
        for line in patch.split('\n'):
            if line.startswith('+') or line.startswith('-'):
                if re.search(self.patterns['documentation'], line.lower()):
                    return True
        return False

    def _determine_change_type(self, file_change: Dict[str, Any]) -> str:
        """Determine the type of change."""
        change_type = file_change.get('change_type', 'M')
        if change_type == 'A':
            return 'addition'
        elif change_type == 'D':
            return 'deletion'
        elif change_type == 'R':
            return 'renamed'
        else:
            return 'modification'

    def _identify_patterns(self, commit: Dict[str, Any]) -> Dict[str, Any]:
        """Identify patterns in the commit."""
        patterns = defaultdict(int)
        
        # Analyze commit message
        message = commit.get('message', '').lower()
        for pattern_name, pattern in self.patterns.items():
            if re.search(pattern, message):
                patterns[pattern_name] += 1
                
        return dict(patterns)

    def _analyze_dependencies(self, commit: Dict[str, Any]) -> Dict[str, Any]:
        """Analyze dependency changes in the commit."""
        dependencies = defaultdict(int)
        
        for file_change in commit.get('file_changes', []):
            if file_change.get('patch'):
                deps = self._extract_dependencies(file_change['patch'])
                for dep in deps:
                    dependencies[dep] += 1
                    
        return dict(dependencies)

    def _analyze_api_changes(self, commit: Dict[str, Any]) -> Dict[str, Any]:
        """Analyze API changes in the commit."""
        api_changes = defaultdict(int)
        
        for file_change in commit.get('file_changes', []):
            if file_change.get('patch'):
                apis = self._extract_api_changes(file_change['patch'])
                for api in apis:
                    api_changes[api] += 1
                    
        return dict(api_changes)

    def _analyze_file_without_patch(self, file_change: Dict[str, Any]) -> Optional[CodeChange]:
        """Analyze file changes when patch is not available."""
        file_path = file_change.get('file_path', '')
        change_type = self._determine_change_type(file_change)
        
        # Try to analyze based on file path and change type
        api_changes = []
        dependencies = []
        
        # Check file extension for potential API changes
        if file_path.endswith(('.py', '.js', '.ts', '.java', '.cpp', '.c')):
            # Could be API changes
            api_changes.append('potential_api_change')
        
        # Check for dependency files
        if any(dep_file in file_path.lower() for dep_file in ['requirements', 'setup.py', 'package.json', 'pom.xml', 'build.gradle']):
            dependencies.append('dependency_file')
        
        # Check for test files
        test_coverage_change = 1.0 if any(test_indicator in file_path.lower() for test_indicator in ['test', 'spec', 'specs']) else 0.0
        
        return CodeChange(
            file_path=file_path,
            change_type=change_type,
            impact_score=0.0,
            complexity_change=0.0,
            dependencies_affected=dependencies,
            api_changes=api_changes,
            test_coverage_change=test_coverage_change,
            documentation_changes='readme' in file_path.lower() or 'doc' in file_path.lower(),
            security_impact=0.0,
            performance_impact=0.0
        ) 