# coding: utf-8

"""
Class for reading and processing Digital RF (DRF) data generated by Grape2 PSWS stations.
Author: Cuong Nguyen
"""

"""
TODO: Add support for DRF input of arbitrary sampling rate
TODO: Add support for specifying number of hours to read
TODO: Think about metadata (e.g. include in cache or not)
TODO:
    - Add support for logging
    - Log exceptions
    - Configure logging level at the command-line 
TODO: Test with complex DRF input
TODO: To see if cache directory is automatically removed if the object is destroyed or closed in anyway
TODO: Get memory profile for the initial run
TODO: cache command line option
TODO: add support for Grape 1 and RX888 receivers
"""
"""
Caching is always enabled.
However, optional parameter to remove cache when done.
"""

import os
import atexit
import shutil
import digital_rf as drf
import math
import numpy as np
from tqdm import tqdm
import datetime
import pytz
import pickle
import logging
from typing import Optional, Dict, Any, List
from scipy import signal

logger = logging.getLogger(__name__)

class Reader:
    """ Class for reading and processing Digital RF (DRF) data generated by Grape2 PSWS stations."""

    def __init__(self, datadir: str, cachedir: str = "cache", cleanup_cache: bool = True, target_bandwidth: int = 10, batch_size_mins: int = 30):
        self.datadir = datadir
        self.target_bandwidth = target_bandwidth
        self.resampled_fs = target_bandwidth*2
        self.cleanup_cache = cleanup_cache
        self.f_c = 1000 # TODO: what does this actually mean?

        if batch_size_mins > 60:
            error_message = "Batch size must be less than or equal to 60 minutes"
            logging.error(error_message)
            raise ValueError(error_message)
        else:
            self.batch_size_mins = batch_size_mins

        self.dro = drf.DigitalRFReader(datadir)
        self.dmr = drf.DigitalMetadataReader(os.path.join(datadir, "ch0", "metadata"))
        self.fs = int(self.dmr.get_samples_per_second())
        self.start_index, self.end_index = self.dro.get_bounds("ch0")
        self.utc_date = self._get_initial_date()
        self.metadata = {}
        self._extract_metadata()

        # Create cache directory in the same directory as the source code
        source_dir = os.path.dirname(os.path.abspath(__file__))
        self.cachedir = os.path.join(source_dir, cachedir)
        os.makedirs(self.cachedir, exist_ok=True)
        self._ensure_data_cached()


        atexit.register(self._cleanup)

    def _get_initial_date(self) -> datetime.date:
        """Retrieve the initial UTC timestamp and return channel_indexit as a date."""
        rf_properties = self.dro.get_properties("ch0", sample=self.start_index)
        init_timestamp = rf_properties["init_utc_timestamp"]
        return datetime.datetime.fromtimestamp(init_timestamp, tz=pytz.utc)

    def _extract_metadata(self):
        """Extract metadata such as station, node, and center frequencies."""
        try:
            start_global_index = self.dmr.get_bounds()[0]
            header_meta = self.dmr.read(start_global_index, start_global_index + 1)[start_global_index]
            self.metadata = header_meta
            print(self.metadata)
            self.metadata.update({
                "station": header_meta["callsign"],
                "center_frequencies": header_meta["center_frequencies"],
                "lat": header_meta["lat"],
                "lon": header_meta["long"],
                "grid": header_meta["grid_square"],
                "utc_date": self.utc_date
            })
            if "station_node_number" in header_meta:
                self.metadata["node"] = header_meta["station_node_number"]
            if "ad_sample_rate"in header_meta:
                self.metadata["sampling_rate"] = header_meta["ad_sample_rate"]
            else:
                self.metadata["sampling_rate"] = self.dmr.get_samples_per_second()


        except Exception as e:
            error_message = f"Error extracting metadata: {e}"
            logging.error(error_message)
            raise ValueError(error_message)

    def _cleanup(self):
        """Remove the cache directory if 'remove_cache_on_exit' is True."""
        if self.cleanup_cache and os.path.exists(self.cachedir):
            shutil.rmtree(self.cachedir)
            log_message = f"Removed cache directory: {self.cachedir}"
            logging.info(log_message)
            print(log_message)

    def _ensure_data_cached(self):
        """ Ensure that rawdata is resampled and cached."""
        rawdata_cached = True
        for channel_index in range(len(self.metadata["center_frequencies"])):
            if not os.path.exists(self._get_cache_file_path(channel_index)):
                rawdata_cached = False
                break
        if rawdata_cached:
            return            
        else:
            log_message = "Rawdata not cached. Caching rawdata..."
            logger.info(log_message)
            print(log_message + " This will take some time. Snacks are not included.")
            for channel_index in range(len(self.metadata["center_frequencies"])):
                log_message = f"Reading DRF rawdata for {self.metadata["center_frequencies"][channel_index]} MHz channel..."
                logger.info(log_message)
                print(log_message)
                raw_data = self._read_rawdata_channel(channel_index)
                resampled_data = self._resample(raw_data)
                self._cache_data(
                    self._get_cache_file_path(channel_index), resampled_data
                )

    def _read_rawdata_channel(self, channel_index: int):
        """Read raw data and store it using np.memmap to reduce memory usage."""
        cont_data_arr = self.dro.get_continuous_blocks(
            self.start_index, self.end_index, "ch0"
        )
        batch_size_samples = self.fs * 60 * self.batch_size_mins
        read_iters = math.ceil((self.end_index - self.start_index) / batch_size_samples)

        start_sample = list(cont_data_arr.keys())[0]
        memmap_file = f"{self.cachedir}/raw_data_channel_{channel_index}.memmap"
        memmap_array = np.memmap(
            memmap_file,
            dtype="float32",
            mode="w+",
            shape=(self.end_index - self.start_index + 1,),
        )

        for i in tqdm(range(read_iters), desc="Reading Batches"):
            batch = self.dro.read_vector(
                start_sample + i * batch_size_samples,
                batch_size_samples,
                "ch0",
                channel_index,
            )
            memmap_array[i * batch_size_samples : (i + 1) * batch_size_samples] = batch

        return memmap_array

    def _cache_data(self, path: str, data):
        """Cache the data to a pickle file."""
        pickle.dump(data, open(path, "wb"))
        log_message = f"Cached data at: {path}"
        logging.info(log_message)
        print(log_message)

    def _get_cache_file_path(self, channel: int) -> str:
        """Generate the file path for the cached data."""
        center_frequency = str(self.metadata["center_frequencies"][channel]).replace(".", "p")
        return os.path.join(
            self.cachedir,
            f"{self.metadata["utc_date"].date()}_{self.metadata["node"]}_RAWDATA_fs-{self.resampled_fs}Hz_cf-{center_frequency}MHz.ba.pkl",
        )

    def _resample(self, data):
        decimation_factor = math.ceil(self.fs / self.resampled_fs)
        # Process data in chunks to save memory
        chunk_size = min(10**6, len(data))  # Adjust chunk size based on your system's memory limits
        resampled_chunks = []

        for i in range(0, len(data), chunk_size):
            chunk = data[i:i + chunk_size]
            t_chunk = np.arange(i, i + len(chunk)) / self.fs

            # Use broadcasting to avoid intermediate large arrays
            cos_wave = np.cos(2 * np.pi * (self.f_c - self.target_bandwidth / 2) * t_chunk)
            shifted_chunk = chunk * cos_wave

            # Use FIR filter for memory-efficient resampling
            resampled_chunk = signal.decimate(
                shifted_chunk, decimation_factor, ftype="fir", zero_phase=True
            )
            resampled_chunks.append(resampled_chunk)

        # Concatenate chunks at the end to minimize memory footprint
        return np.concatenate(resampled_chunks)

    def read_data(self, channel_index: int) -> np.ndarray:
        """ Read the cached datata for the specified channel. """
        cache_path = self._get_cache_file_path(channel_index)
        return pickle.load(open(cache_path, "rb"))

    def get_metadata(self) -> Dict[str, Any]:
        """Return metadata information."""
        return self.metadata

if __name__ == "__main__":
    # data_dir = sys.argv[1]
    # cache_dir = 'cache'
    # data_reader = PSWSDataReader(data_dir, cachedir=cache_dir, resampled_fs=2000) 
    # for i in range(3): 
    #     data_reader.read_data(i)
    logging.basicConfig(
        filename="reader.log",
        level=logging.DEBUG,  # TODO:  Add support for setting logging level at command-line level. Default: WARNING
        format="%(asctime)s | %(levelname)-10s | %(name)-15s | %(funcName)-30s | %(message)s",
        force=True
    )
    datadir = "/home/cuong/drive/GRAPE2-SFTP/w2naf"
    cachedir = "cache"
    reader = Reader(datadir, cachedir)
