#!/usr/bin/env python3
"""
Natural Language Query Examples

Demonstrates the natural language query interface with realistic examples.
Run with: python natural_query_examples.py
"""

from graph_api import ElementStore, query
from graph_api.base_element import (
    BaseElement,
    ElementDetails,
    ElementProperties,
    NodeTypes,
)


def create_sample_store() -> ElementStore:
    """Create a sample knowledge graph for demonstrations."""
    store = ElementStore()

    # Create sample people
    people_data = [
        {
            "id": "p1",
            "name": "Alice Johnson",
            "age": 32,
            "role": "senior engineer",
            "salary": 120000,
        },
        {
            "id": "p2",
            "name": "Bob Smith",
            "age": 28,
            "role": "junior engineer",
            "salary": 75000,
        },
        {
            "id": "p3",
            "name": "Charlie Davis",
            "age": 45,
            "role": "manager",
            "salary": 150000,
        },
        {
            "id": "p4",
            "name": "Diana Wilson",
            "age": 35,
            "role": "senior engineer",
            "salary": 130000,
        },
        {
            "id": "p5",
            "name": "Eve Martinez",
            "age": 26,
            "role": "intern",
            "salary": 50000,
        },
    ]

    # Create sample organizations
    org_data = [
        {
            "id": "o1",
            "name": "TechCorp",
            "founded": 2015,
            "employees": 500,
            "type": "technology",
        },
        {
            "id": "o2",
            "name": "StartupXYZ",
            "founded": 2020,
            "employees": 25,
            "type": "startup",
        },
        {
            "id": "o3",
            "name": "DataInc",
            "founded": 2018,
            "employees": 150,
            "type": "analytics",
        },
        {
            "id": "o4",
            "name": "CloudSoft",
            "founded": 2019,
            "employees": 80,
            "type": "cloud",
        },
    ]

    # Create sample equipment
    equipment_data = [
        {
            "id": "e1",
            "name": "Laptop Dell XPS 15",
            "type": "computer",
            "year": 2023,
            "value": 1500,
        },
        {
            "id": "e2",
            "name": "Server HP ProLiant",
            "type": "server",
            "year": 2022,
            "value": 5000,
        },
        {
            "id": "e3",
            "name": "Rifle M16A2",
            "type": "weapon",
            "year": 1995,
            "value": 500,
        },
        {
            "id": "e4",
            "name": "Camera Canon EOS",
            "type": "camera",
            "year": 2021,
            "value": 800,
        },
        {
            "id": "e5",
            "name": "Printer Xerox",
            "type": "printer",
            "year": 2020,
            "value": 300,
        },
    ]

    # Add people to store
    for data in people_data:
        element_id = data.pop("id")
        class_id = "person"
        props = ElementProperties(**data)
        details = ElementDetails(
            id=element_id, class_id=class_id, type=NodeTypes.NODE, properties=props
        )
        element = BaseElement(details, store)
        store.elements[element_id] = element

    # Add organizations to store
    for data in org_data:
        element_id = data.pop("id")
        class_id = "organization"
        props = ElementProperties(**data)
        details = ElementDetails(
            id=element_id, class_id=class_id, type=NodeTypes.NODE, properties=props
        )
        element = BaseElement(details, store)
        store.elements[element_id] = element

    # Add equipment to store
    for data in equipment_data:
        element_id = data.pop("id")
        class_id = "equipment"
        props = ElementProperties(**data)
        details = ElementDetails(
            id=element_id, class_id=class_id, type=NodeTypes.NODE, properties=props
        )
        element = BaseElement(details, store)
        store.elements[element_id] = element

    return store


def example_1_basic_queries(store: ElementStore):
    """Example 1: Basic queries - get all elements of a type."""
    print("=" * 70)
    print("EXAMPLE 1: Basic Queries")
    print("=" * 70)

    # Query 1a: All people
    print("\n1a. Query: 'all people'")
    hq = query(store, "all people")
    results = hq.execute()
    print(f"   Result: Found {len(results)} people")
    for person in results:
        print(f"     - {person.properties.get('name')}")

    # Query 1b: All organizations
    print("\n1b. Query: 'organizations'")
    hq = query(store, "organizations")
    results = hq.execute()
    print(f"   Result: Found {len(results)} organizations")
    for org in results:
        print(
            f"     - {org.properties.get('name')} ({org.properties.get('employees')} employees)"
        )

    # Query 1c: All equipment
    print("\n1c. Query: 'equipment'")
    hq = query(store, "equipment")
    results = hq.execute()
    print(f"   Result: Found {len(results)} equipment items")


def example_2_comparisons(store: ElementStore):
    """Example 2: Comparison operators - filter by numeric values."""
    print("\n" + "=" * 70)
    print("EXAMPLE 2: Comparison Operators")
    print("=" * 70)

    # Query 2a: People older than 30
    print("\n2a. Query: 'people older than 30'")
    hq = query(store, "people older than 30")
    results = hq.execute()
    print(f"   Result: Found {len(results)} people older than 30")
    for person in results:
        print(
            f"     - {person.properties.get('name')}: age {person.properties.get('age')}"
        )

    # Query 2b: People younger than 30
    print("\n2b. Query: 'people younger than 30'")
    hq = query(store, "people younger than 30")
    results = hq.execute()
    print(f"   Result: Found {len(results)} people younger than 30")
    for person in results:
        print(
            f"     - {person.properties.get('name')}: age {person.properties.get('age')}"
        )

    # Query 2c: Organizations with more than 100 employees
    print("\n2c. Query: 'organizations with more than 100 employees'")
    hq = query(store, "organizations with more than 100 employees")
    results = hq.execute()
    print(f"   Result: Found {len(results)} large organizations")
    for org in results:
        print(
            f"     - {org.properties.get('name')}: {org.properties.get('employees')} employees"
        )


def example_3_text_search(store: ElementStore):
    """Example 3: Text search - find by keywords."""
    print("\n" + "=" * 70)
    print("EXAMPLE 3: Text Search")
    print("=" * 70)

    # Query 3a: Equipment containing "laptop"
    print("\n3a. Query: 'equipment containing laptop'")
    hq = query(store, "equipment containing laptop")
    results = hq.execute()
    print(f"   Result: Found {len(results)} items")
    for item in results:
        print(f"     - {item.properties.get('name')}")

    # Query 3b: Equipment containing "server"
    print("\n3b. Query: 'equipment containing server'")
    hq = query(store, "equipment containing server")
    results = hq.execute()
    print(f"   Result: Found {len(results)} server items")
    for item in results:
        print(f"     - {item.properties.get('name')}")

    # Query 3c: Organizations with "Tech" in name
    print("\n3c. Query: 'organizations containing tech'")
    hq = query(store, "organizations containing tech")
    results = hq.execute()
    print(f"   Result: Found {len(results)} tech organizations")
    for org in results:
        print(f"     - {org.properties.get('name')}")


def example_4_sorting(store: ElementStore):
    """Example 4: Sorting - order results by property."""
    print("\n" + "=" * 70)
    print("EXAMPLE 4: Sorting")
    print("=" * 70)

    # Query 4a: People sorted by age ascending
    print("\n4a. Query: 'people sorted by age ascending'")
    hq = query(store, "people sorted by age ascending")
    results = hq.execute()
    print(f"   Result: {len(results)} people sorted by age")
    for person in results:
        print(
            f"     - {person.properties.get('name')}: {person.properties.get('age')} years old"
        )

    # Query 4b: People sorted by salary descending
    print("\n4b. Query: 'people sorted by salary descending'")
    hq = query(store, "people sorted by salary descending")
    results = hq.execute()
    print(f"   Result: {len(results)} people sorted by salary (highest first)")
    for person in results:
        print(
            f"     - {person.properties.get('name')}: ${person.properties.get('salary')}"
        )

    # Query 4c: Organizations sorted by employees
    print("\n4c. Query: 'organizations sorted by employees'")
    hq = query(store, "organizations sorted by employees")
    results = hq.execute()
    print("   Result: Organizations by size")
    for org in results:
        print(
            f"     - {org.properties.get('name')}: {org.properties.get('employees')} employees"
        )


def example_5_limiting(store: ElementStore):
    """Example 5: Limiting - get top N results."""
    print("\n" + "=" * 70)
    print("EXAMPLE 5: Limiting Results")
    print("=" * 70)

    # Query 5a: First 3 people
    print("\n5a. Query: 'people first 3'")
    hq = query(store, "people first 3")
    results = hq.execute()
    print(f"   Result: Top {len(results)} people")
    for person in results:
        print(f"     - {person.properties.get('name')}")

    # Query 5b: Top 2 organizations
    print("\n5b. Query: 'organizations top 2'")
    hq = query(store, "organizations top 2")
    results = hq.execute()
    print(f"   Result: {len(results)} organizations")
    for org in results:
        print(f"     - {org.properties.get('name')}")


def example_6_complex_queries(store: ElementStore):
    """Example 6: Complex queries - combine multiple filters."""
    print("\n" + "=" * 70)
    print("EXAMPLE 6: Complex Combined Queries")
    print("=" * 70)

    # Query 6a: People older than 25 sorted by age first 3
    print("\n6a. Query: 'people older than 25 sorted by age first 3'")
    hq = query(store, "people older than 25 sorted by age first 3")
    results = hq.execute()
    print(f"   Result: Top {len(results)} people over 25, sorted by age")
    for person in results:
        print(f"     - {person.properties.get('name')}: {person.properties.get('age')}")

    # Query 6b: Organizations with more than 50 employees sorted by name
    print("\n6b. Query: 'organizations with more than 50 employees sorted by name'")
    hq = query(store, "organizations with more than 50 employees sorted by name")
    results = hq.execute()
    print(f"   Result: {len(results)} medium/large organizations")
    for org in results:
        print(
            f"     - {org.properties.get('name')}: {org.properties.get('employees')} employees"
        )

    # Query 6c: Equipment from recent years sorted by value descending
    print("\n6c. Query: 'equipment sorted by year descending first 3'")
    hq = query(store, "equipment sorted by year descending first 3")
    results = hq.execute()
    print(f"   Result: {len(results)} most recent equipment")
    for item in results:
        print(
            f"     - {item.properties.get('name')}: {item.properties.get('year')} (${item.properties.get('value')})"
        )


def example_7_counting(store: ElementStore):
    """Example 7: Counting - get totals without retrieving data."""
    print("\n" + "=" * 70)
    print("EXAMPLE 7: Counting")
    print("=" * 70)

    # Query 7a: Count all people
    print("\n7a. Query: 'people' (count)")
    hq = query(store, "people")
    count = hq.count()
    print(f"   Result: Total of {count} people in store")

    # Query 7b: Count all organizations
    print("\n7b. Query: 'organizations' (count)")
    hq = query(store, "organizations")
    count = hq.count()
    print(f"   Result: Total of {count} organizations")

    # Query 7c: Show GraphQuery representation
    print("\n7c. Query representation:")
    hq = query(store, "people older than 30")
    print("   Natural query: 'people older than 30'")
    print(f"   GraphQuery:   {repr(hq)}")


def example_8_analytics(store: ElementStore):
    """Example 8: Real-world analytics scenarios."""
    print("\n" + "=" * 70)
    print("EXAMPLE 8: Real-World Analytics")
    print("=" * 70)

    # Scenario 1: Find high earners
    print("\n8a. Scenario: Find senior engineers making over $125,000")
    hq = query(store, "people with role senior engineer sorted by salary descending")
    results = hq.execute()
    print(f"   Result: {len(results)} senior engineers")
    for person in results:
        print(
            f"     - {person.properties.get('name')}: ${person.properties.get('salary')}"
        )

    # Scenario 2: Organizations founded recently
    print("\n8b. Scenario: Find startups (founded after 2018)")
    hq = query(store, "organizations sorted by founded descending first 5")
    results = hq.execute()
    print(f"   Result: {len(results)} recent organizations")
    for org in results:
        print(
            f"     - {org.properties.get('name')}: Founded {org.properties.get('founded')}"
        )

    # Scenario 3: Expensive equipment
    print("\n8c. Scenario: Find high-value equipment over $1000")
    hq = query(store, "equipment sorted by value descending")
    results = hq.execute()
    expensive = [e for e in results if e.properties.get("value", 0) > 1000]
    print(f"   Result: {len(expensive)} items over $1000")
    for item in expensive:
        print(f"     - {item.properties.get('name')}: ${item.properties.get('value')}")


def main():
    """Run all examples."""
    print("\n")
    print("╔" + "=" * 68 + "╗")
    print("║" + " " * 68 + "║")
    print("║" + "  Natural Language Query Examples".center(68) + "║")
    print("║" + " " * 68 + "║")
    print("╚" + "=" * 68 + "╝")

    # Create sample store
    store = create_sample_store()
    print(f"\n✓ Created sample knowledge graph with {len(store.elements)} elements")

    # Run examples
    example_1_basic_queries(store)
    example_2_comparisons(store)
    example_3_text_search(store)
    example_4_sorting(store)
    example_5_limiting(store)
    example_6_complex_queries(store)
    example_7_counting(store)
    example_8_analytics(store)

    print("\n" + "=" * 70)
    print("✅ All examples completed successfully!")
    print("=" * 70 + "\n")


if __name__ == "__main__":
    main()
