"""
Tests for Natural Language Query Interface

Tests the human-friendly query parser and execution.
Run with: python -m pytest test_natural_query.py -v
"""

import pytest

from graph_api.base_element import (
    BaseElement,
    ElementDetails,
    ElementProperties,
    NodeTypes,
)
from graph_api.edge import Edge, EdgeDetails
from graph_api.element_store import ElementStore
from graph_api.meta import (
    MetaNode,
    MetaNodeDetails,
    MetaNodeProperties,
    MetaPropertyType,
    PropValueType,
)
from graph_api.natural_query import HumanQuery, NaturalQueryParser, query


@pytest.fixture
def populated_store():
    """Create a store with realistic test data."""
    store = ElementStore()

    # Create test elements (nodes)
    elements_data = [
        {
            "id": "1",
            "class_id": "person",
            "name": "Alice Johnson",
            "age": 32,
            "email": "alice@example.com",
            "role": "senior developer",
        },
        {
            "id": "2",
            "class_id": "person",
            "name": "Bob Smith",
            "age": 28,
            "email": "bob@example.com",
            "role": "junior developer",
        },
        {
            "id": "3",
            "class_id": "person",
            "name": "Charlie Brown",
            "age": 45,
            "email": "charlie@example.com",
            "role": "manager",
        },
        {
            "id": "4",
            "class_id": "organization",
            "name": "TechCorp",
            "founded": 2015,
            "employees": 250,
            "type": "technology",
        },
        {
            "id": "5",
            "class_id": "organization",
            "name": "StartupXYZ",
            "founded": 2020,
            "employees": 15,
            "type": "startup",
        },
        {
            "id": "6",
            "class_id": "equipment",
            "name": "Laptop Dell XPS",
            "type": "computer",
            "year": 2023,
            "value": 1200,
        },
        {
            "id": "7",
            "class_id": "equipment",
            "name": "Rifle M16",
            "type": "weapon",
            "year": 1995,
            "value": 500,
        },
        # Additional people for richer traversal / filtering
        {
            "id": "8",
            "class_id": "person",
            "name": "Diana Prince",
            "age": 38,
            "email": "diana@example.com",
            "role": "security analyst",
        },
        {
            "id": "9",
            "class_id": "person",
            "name": "Evan Wright",
            "age": 41,
            "email": "evan@example.com",
            "role": "data scientist",
        },
        # Additional organizations
        {
            "id": "10",
            "class_id": "organization",
            "name": "SecureCorp",
            "founded": 2010,
            "employees": 80,
            "type": "security",
        },
        {
            "id": "11",
            "class_id": "organization",
            "name": "DataWorks",
            "founded": 2012,
            "employees": 120,
            "type": "analytics",
        },
        # Additional equipment
        {
            "id": "12",
            "class_id": "equipment",
            "name": "Server Rack 1",
            "type": "server",
            "year": 2021,
            "value": 5000,
        },
        {
            "id": "13",
            "class_id": "equipment",
            "name": "Firewall Appliance",
            "type": "network",
            "year": 2022,
            "value": 3000,
        },
    ]

    for elem_data in elements_data:
        element_id = elem_data.pop("id")
        class_id = elem_data.pop("class_id")
        properties = ElementProperties(**elem_data)
        details = ElementDetails(
            id=element_id, class_id=class_id, type=NodeTypes.NODE, properties=properties
        )
        element = BaseElement(details, store)
        store.elements[element_id] = element

    # Create a richer edge structure between people, organizations, and equipment
    edges_data = [
        # Employment relationships
        {"id": "e1", "class_id": "employment", "from_id": "1", "to_id": "4"},
        {"id": "e2", "class_id": "employment", "from_id": "2", "to_id": "4"},
        {"id": "e3", "class_id": "employment", "from_id": "3", "to_id": "4"},
        {"id": "e4", "class_id": "employment", "from_id": "8", "to_id": "10"},
        {"id": "e5", "class_id": "employment", "from_id": "9", "to_id": "11"},
        # Equipment assignment
        {"id": "e6", "class_id": "assigned_to", "from_id": "6", "to_id": "1"},
        {"id": "e7", "class_id": "assigned_to", "from_id": "7", "to_id": "3"},
        {"id": "e8", "class_id": "assigned_to", "from_id": "12", "to_id": "10"},
        {"id": "e9", "class_id": "assigned_to", "from_id": "13", "to_id": "11"},
        # Colleague-like relationships between people
        {"id": "e10", "class_id": "colleague", "from_id": "1", "to_id": "2"},
        {"id": "e11", "class_id": "colleague", "from_id": "2", "to_id": "3"},
        {"id": "e12", "class_id": "colleague", "from_id": "1", "to_id": "8"},
        {"id": "e13", "class_id": "colleague", "from_id": "8", "to_id": "9"},
    ]

    for e in edges_data:
        edge_props = ElementProperties()
        ed = EdgeDetails(
            id=e["id"],
            class_id=e["class_id"],
            type=NodeTypes.EDGE,
            properties=edge_props,
            from_id=e["from_id"],
            to_id=e["to_id"],
        )
        edge = Edge(ed, store)

        # Attach edge to node endpoints for traversal-related behavior
        src = store.elements.get(e["from_id"])
        dst = store.elements.get(e["to_id"])
        if src is not None and hasattr(src, "outgoing_edges"):
            src.outgoing_edges.append(edge)
        if dst is not None and hasattr(dst, "incoming_edges"):
            dst.incoming_edges.append(edge)

    # Add a simple MetaNode with an options property to exercise
    # option-type and alias-based natural queries.
    meta_properties = MetaNodeProperties(
        name="Event",
        type="node",
        property_types={
            "disorder_type": MetaPropertyType(
                type=PropValueType.OPTIONS,
                label="Disorder type",
                key="disorder_type",
                options=[
                    "Political violence",
                    "Demonstrations",
                    "Strategic developments",
                ],
                aliases=["disorder"],
            )
        },
    )
    meta_details = MetaNodeDetails(
        id="meta_event",
        class_id="event",
        type=NodeTypes.META,
        properties=meta_properties,
    )
    meta_node = MetaNode(meta_details, store)
    store.elements[meta_node.id] = meta_node

    # Add a couple of event elements using the options
    for idx, opt in enumerate(
        meta_properties.property_types["disorder_type"].options, start=1
    ):
        ev_props = ElementProperties(
            name=f"Event {idx}",
            disorder_type=opt,
        )
        ev_details = ElementDetails(
            id=f"e{idx}",
            class_id="event",
            type=NodeTypes.NODE,
            properties=ev_props,
        )
        ev = BaseElement(ev_details, store)
        store.elements[ev.id] = ev

    return store


class TestNaturalQueryParser:
    """Test the natural query parser."""

    def test_parser_creation(self, populated_store):
        """Test creating a parser."""
        parser = NaturalQueryParser(populated_store)
        assert parser is not None
        assert parser.store == populated_store

    def test_extract_class_from_query(self, populated_store):
        """Test extracting class ID from natural queries."""
        parser = NaturalQueryParser(populated_store)

        # Test basic class extraction
        class_id = parser._extract_class("all people older than 30")
        assert class_id == "person"

        class_id = parser._extract_class("organizations with more than 100 employees")
        assert class_id == "organization"

    def test_extract_property(self, populated_store):
        """Test extracting property names."""
        parser = NaturalQueryParser(populated_store)

        prop = parser._extract_property("people older than 30")
        assert prop in ["age", "older"]

        prop = parser._extract_property("organizations with more than 100 employees")
        assert prop == "employees"

    def test_extract_greater_than_operator(self, populated_store):
        """Test extracting 'greater than' comparisons."""
        parser = NaturalQueryParser(populated_store)

        operator, value = parser._extract_operator_and_value("people older than 30")
        assert value == 30

    def test_extract_less_than_operator(self, populated_store):
        """Test extracting 'less than' comparisons."""
        parser = NaturalQueryParser(populated_store)

        operator, value = parser._extract_operator_and_value("people younger than 40")
        assert value == 40

    def test_extract_text_search(self, populated_store):
        """Test extracting text search terms."""
        parser = NaturalQueryParser(populated_store)

        text = parser._extract_text_search("equipment containing laptop")
        assert text == "laptop"

        text = parser._extract_text_search("search for dell")
        assert text == "dell"

    def test_extract_sort(self, populated_store):
        """Test extracting sort instructions."""
        parser = NaturalQueryParser(populated_store)

        prop, direction = parser._extract_sort("sorted by name ascending")
        assert prop == "name"
        assert direction == "asc"

        prop, direction = parser._extract_sort("order by age descending")
        assert prop == "age"
        assert direction == "desc"

    def test_extract_limit(self, populated_store):
        """Test extracting limit/first/top."""
        parser = NaturalQueryParser(populated_store)

        limit = parser._extract_limit("first 5")
        assert limit == 5

        limit = parser._extract_limit("top 10")
        assert limit == 10

        limit = parser._extract_limit("limit 20")
        assert limit == 20

    def test_parse_simple_query(self, populated_store):
        """Test parsing a simple query."""
        parser = NaturalQueryParser(populated_store)
        human_query = parser.parse("all people")

        assert human_query.class_id == "person"
        assert human_query.operator is None

    def test_parse_complex_query(self, populated_store):
        """Test parsing a complex query with multiple filters."""
        parser = NaturalQueryParser(populated_store)
        human_query = parser.parse("people older than 30 sorted by name")

        assert human_query.class_id == "person"
        assert human_query.value == 30
        assert human_query.sort_property == "name"


class TestHumanQuery:
    """Test the HumanQuery execution class."""

    def test_human_query_creation(self, populated_store):
        """Test creating a HumanQuery object."""
        hq = HumanQuery(populated_store)
        assert hq is not None
        assert hq.store == populated_store

    def test_convert_to_graph_query(self, populated_store):
        """Test converting HumanQuery to GraphQuery."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"

        graph_query = hq.to_graph_query()
        assert graph_query is not None
        results = graph_query.r()
        # 5 people in store after fixture expansion
        assert len(results) == 5

    def test_execute_simple_query(self, populated_store):
        """Test executing a simple query."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"

        results = hq.execute()
        assert len(results) == 5
        assert all(el.class_id == "person" for el in results)

    def test_execute_with_comparison(self, populated_store):
        """Test executing query with comparison operator."""
        from graph_api.graph_query import FilterOperator

        hq = HumanQuery(populated_store)
        hq.class_id = "person"
        hq.property = "age"
        hq.operator = FilterOperator.GT
        hq.value = 30

        results = hq.execute()
        # People older than 30 in expanded fixture
        assert len(results) == 4
        assert all(el.properties.get("age", 0) > 30 for el in results)

    def test_execute_with_text_search(self, populated_store):
        """Test executing query with text search."""
        hq = HumanQuery(populated_store)
        hq.class_id = "equipment"
        hq.text_search = "dell"

        results = hq.execute()
        assert len(results) == 1
        assert "dell" in results[0].properties.get("name", "").lower()

    def test_execute_with_sort(self, populated_store):
        """Test executing query with sorting."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"
        hq.sort_property = "age"
        hq.sort_direction = "asc"

        results = hq.execute()
        ages = [el.properties.get("age") for el in results]
        assert ages == sorted(ages)  # Should be ascending

    def test_execute_with_limit(self, populated_store):
        """Test executing query with limit."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"
        hq.limit = 2

        results = hq.execute()
        assert len(results) == 2

    def test_count_results(self, populated_store):
        """Test counting results."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"

        count = hq.count()
        assert count == 5

    def test_query_repr(self, populated_store):
        """Test string representation of query."""
        hq = HumanQuery(populated_store)
        hq.class_id = "person"

        repr_str = repr(hq)
        assert "classId" in repr_str
        assert "person" in repr_str


class TestConvenientQueryFunction:
    """Test the convenient query() function."""

    def test_query_function_simple(self, populated_store):
        """Test query function with simple query."""
        hq = query(populated_store, "all people")
        assert hq.class_id == "person"

    def test_query_function_complex(self, populated_store):
        """Test query function with complex query."""
        hq = query(populated_store, "people older than 30")
        assert hq.class_id == "person"
        assert hq.value == 30

    def test_query_function_execute(self, populated_store):
        """Test query function and execute."""
        hq = query(populated_store, "people older than 30")
        results = hq.execute()
        assert len(results) == 4
        assert all(el.class_id == "person" for el in results)
        assert all(el.properties.get("age", 0) > 30 for el in results)

    def test_query_function_with_limit(self, populated_store):
        """Test query function with limit."""
        hq = query(populated_store, "people first 2")
        results = hq.execute()
        assert len(results) == 2

    def test_query_function_with_sort(self, populated_store):
        """Test query function with sort."""
        hq = query(populated_store, "organizations sorted by employees ascending")
        results = hq.execute()
        employee_counts = [el.properties.get("employees", 0) for el in results]
        assert employee_counts == sorted(employee_counts)


class TestNaturalQueryIntegration:
    """Integration tests combining parser, conversion, and execution."""

    def test_people_older_than_30(self, populated_store):
        """Test: show me all people older than 30"""
        hq = query(populated_store, "all people older than 30")
        results = hq.execute()

        # Alice (32), Charlie (45), Diana (38), Evan (41)
        assert len(results) == 4
        names = [el.properties.get("name") for el in results]
        assert "Alice Johnson" in names
        assert "Charlie Brown" in names

    def test_equipment_search(self, populated_store):
        """Test: find equipment containing 'dell'"""
        hq = query(populated_store, "equipment containing dell")
        results = hq.execute()

        # Only Laptop Dell XPS matches
        assert len(results) == 1
        assert results[0].properties.get("name") == "Laptop Dell XPS"

    def test_top_3_results_aliases(self, populated_store):
        """Test flexible limit phrases mapping to first(3)."""
        # We'll just verify parsing doesn't error and limit is set.
        parser = NaturalQueryParser(populated_store)

        for q in [
            "top 3 people",
            "first 3 people",
            "at most 3 people",
        ]:
            hq = parser.parse(q)
            assert hq.class_id == "person"
            assert hq.limit == 3

    def test_organizations_by_size(self, populated_store):
        """Test: organizations with more than 100 employees sorted by name"""
        hq = query(
            populated_store, "organizations with more than 100 employees sorted by name"
        )
        results = hq.execute()

        # TechCorp (250) and DataWorks (120)
        assert len(results) == 2
        names = {el.properties.get("name") for el in results}
        assert {"TechCorp", "DataWorks"}.issubset(names)

    def test_recent_equipment(self, populated_store):
        """Test: equipment from 2023 or later, first 5"""
        hq = query(populated_store, "equipment from 2023 first 5")
        results = hq.execute()

        assert len(results) <= 5
        assert all(el.class_id == "equipment" for el in results)

    def test_count_people(self, populated_store):
        """Test: count all people"""
        hq = query(populated_store, "people")
        count = hq.count()

        assert count == 5

    def test_senior_developers(self, populated_store):
        """Test: find senior developers"""
        hq = query(populated_store, "people with role senior developer")
        results = hq.execute()

        assert len(results) == 1
        assert results[0].properties.get("role") == "senior developer"

    def test_full_text_search_in_role(self, populated_store):
        """Test: full-text search scoped to role property."""
        hq = query(populated_store, "people full text search developer in role")
        results = hq.execute()

        # Alice (senior developer) and Bob (junior developer)
        names = {el.properties.get("name") for el in results}
        assert {"Alice Johnson", "Bob Smith"}.issubset(names)

    def test_multiple_filters(self, populated_store):
        """Test combining filters: young, junior developers"""
        hq = query(populated_store, "people younger than 35 with role junior developer")
        # This tests multi-filter capability
        results = hq.execute()
        # Should find Bob (28, junior developer)
        assert any(el.properties.get("age", 0) < 35 for el in results)

    def test_include_and_exclude_terms(self, populated_store):
        """Test: containing bob but not alice."""
        hq = query(populated_store, "people containing bob but not alice")
        results = hq.execute()

        names = [el.properties.get("name", "").lower() for el in results]
        # Should include Bob, not Alice
        assert any("bob" in n for n in names)
        assert all("alice" not in n for n in names)

    def test_empty_result_set(self, populated_store):
        """Test query that returns no results."""
        hq = query(populated_store, "people older than 100")
        results = hq.execute()

        assert len(results) == 0

    def test_all_of_type(self, populated_store):
        """Test: get all items of a type."""
        hq = query(populated_store, "all equipment")
        results = hq.execute()

        assert len(results) == 4
        assert all(el.class_id == "equipment" for el in results)

    def test_top_2_oldest_people(self, populated_store):
        """Test: top 2 oldest people sorted by age desc."""
        hq = query(populated_store, "top 2 people sorted by age desc")
        results = hq.execute()

        # Oldest two: Charlie (45), Alice (32)
        assert len(results) == 2
        ages = [el.properties.get("age") for el in results]
        assert ages == sorted(ages, reverse=True)

    def test_last_2_people_by_age(self, populated_store):
        """Test: last 2 people by age ascending."""
        hq = query(populated_store, "people sorted by age asc last 2")
        results = hq.execute()

        # Youngest two people when sorted ascending by age
        assert len(results) == 2
        ages = [el.properties.get("age") for el in results]
        assert ages == sorted(ages)

    def test_search_by_text_name(self, populated_store):
        """Test: text search on name field via natural language."""
        hq = query(populated_store, "people search by name bob")
        results = hq.execute()

        assert len(results) == 1
        assert results[0].properties.get("name") == "Bob Smith"

    def test_full_text_search_developer(self, populated_store):
        """Test: full-text search for 'developer' across properties."""
        hq = query(populated_store, "people full text search developer")
        results = hq.execute()

        # Alice (senior developer) and Bob (junior developer)
        names = {el.properties.get("name") for el in results}
        assert {"Alice Johnson", "Bob Smith"}.issubset(names)

    def test_wildcard_search_email(self, populated_store):
        """Test: wildcard search on email domain."""
        hq = query(populated_store, "people wildcard search *@example.com on email")
        results = hq.execute()

        assert len(results) == 5
        assert all(
            el.properties.get("email", "").endswith("@example.com") for el in results
        )

    def test_related_to_person(self, populated_store):
        """Test: filter by related elements for a given person.

        This only verifies the parser wiring; actual relation behavior
        depends on edges and is covered by graph_query tests.
        """
        hq = query(populated_store, "elements related to person 1")
        # No edges in this fixture, but we at least ensure execution works
        results = hq.execute()
        assert isinstance(results, list)

    def test_colleagues_of_person_alias(self, populated_store):
        """Test: colleagues of person 1 → related_to person 1.

        Still no edges here; just ensure parsing and execution work.
        """
        hq = query(populated_store, "colleagues of person 1")
        results = hq.execute()
        assert isinstance(results, list)

    def test_option_property_query_by_label(self, populated_store):
        """Test querying option-type property using its label.

        Example: events with disorder type Political violence
        """

        hq = query(populated_store, "events with disorder type Political violence")
        # Focus on end results; internal condition structure may evolve.
        assert hq.class_id == "event"

        results = hq.execute()
        assert len(results) == 1
        assert results[0].properties.get("disorder_type") == "Political violence"

    def test_option_property_query_by_alias(self, populated_store):
        """Test querying option-type property using its alias name.

        Example: events with disorder Political violence
        """

        hq = query(populated_store, "events with disorder Political violence")
        assert hq.class_id == "event"

        results = hq.execute()
        assert len(results) == 1
        assert results[0].properties.get("disorder_type") == "Political violence"

    def test_multiple_disorder_values(self, populated_store):
        """Test querying events for each disorder option using current behavior.

        This exercises multiple option-like values without relying on meta.
        """
        # Political violence
        hq1 = query(populated_store, "events with disorder Political violence")
        res1 = hq1.execute()
        assert any(
            r.properties.get("disorder_type") == "Political violence" for r in res1
        )

        # Demonstrations
        hq2 = query(populated_store, "events with disorder Demonstrations")
        res2 = hq2.execute()
        assert any(r.properties.get("disorder_type") == "Demonstrations" for r in res2)

        # Strategic developments
        hq3 = query(populated_store, "events with disorder Strategic developments")
        res3 = hq3.execute()
        assert any(
            r.properties.get("disorder_type") == "Strategic developments" for r in res3
        )

    def test_events_with_disorder_type_phrase(self, populated_store):
        """Test variations including the phrase 'disorder type' in the query."""
        hq = query(populated_store, "events with disorder type Demonstrations")
        results = hq.execute()

        assert any(
            r.properties.get("disorder_type") == "Demonstrations" for r in results
        )

    def test_organizations_filtered_by_type(self, populated_store):
        """Test organizations filtered by their 'type' property via with-clause."""
        # SecureCorp is a security organization
        hq_sec = query(populated_store, "organizations with type security")
        results_sec = hq_sec.execute()
        assert any(el.properties.get("name") == "SecureCorp" for el in results_sec)

        # DataWorks is an analytics organization
        hq_analytics = query(populated_store, "organizations with type analytics")
        results_analytics = hq_analytics.execute()
        assert any(el.properties.get("name") == "DataWorks" for el in results_analytics)

    def test_equipment_filtered_by_type(self, populated_store):
        """Test equipment filtered by 'type' using with-clause."""
        hq_server = query(populated_store, "equipment with type server")
        res_server = hq_server.execute()
        assert any(el.properties.get("name") == "Server Rack 1" for el in res_server)

        hq_network = query(populated_store, "equipment with type network")
        res_network = hq_network.execute()
        assert any(
            el.properties.get("name") == "Firewall Appliance" for el in res_network
        )

    def test_combined_employees_and_type_filter(self, populated_store):
        """Test combining numeric and string filters in a single query."""
        hq = query(
            populated_store,
            "organizations with more than 50 employees with type analytics",
        )
        results = hq.execute()

        # DataWorks should be included when both conditions apply
        assert any(el.properties.get("name") == "DataWorks" for el in results)

    def test_show_me_all_people(self, populated_store):
        """show me all people  same as all people."""
        hq = query(populated_store, "show me all people")
        results = hq.execute()

        assert all(el.class_id == "person" for el in results)

    def test_list_all_organizations(self, populated_store):
        """list all organizations  all org nodes."""
        hq = query(populated_store, "list all organizations")
        results = hq.execute()

        assert all(el.class_id == "organization" for el in results)

    def test_are_there_any_events(self, populated_store):
        """are there any events with disorder Political violence."""
        hq = query(
            populated_store,
            "are there any events with disorder Political violence",
        )
        results = hq.execute()

        assert any(
            r.properties.get("disorder_type") == "Political violence" for r in results
        )

    # --- Additional natural-language integration variations ---

    def test_people_older_than_25(self, populated_store):
        hq = query(populated_store, "people older than 25")
        results = hq.execute()
        assert all(el.properties.get("age", 0) > 25 for el in results)

    def test_people_older_than_35(self, populated_store):
        hq = query(populated_store, "people older than 35")
        results = hq.execute()
        assert all(el.properties.get("age", 0) > 35 for el in results)

    def test_people_younger_than_45(self, populated_store):
        hq = query(populated_store, "people younger than 45")
        results = hq.execute()
        assert all(el.properties.get("age", 0) < 45 for el in results)

    def test_people_younger_than_30(self, populated_store):
        hq = query(populated_store, "people younger than 30")
        results = hq.execute()
        assert all(el.properties.get("age", 0) < 30 for el in results)

    def test_people_with_role_manager(self, populated_store):
        hq = query(populated_store, "people with role manager")
        results = hq.execute()
        assert any(el.properties.get("role") == "manager" for el in results)

    def test_people_with_role_data_scientist(self, populated_store):
        hq = query(populated_store, "people with role data scientist")
        results = hq.execute()
        assert any(el.properties.get("role") == "data scientist" for el in results)

    def test_people_with_role_security_analyst(self, populated_store):
        hq = query(populated_store, "people with role security analyst")
        results = hq.execute()
        assert any(el.properties.get("role") == "security analyst" for el in results)

    def test_organizations_with_type_startup(self, populated_store):
        hq = query(populated_store, "organizations with type startup")
        results = hq.execute()
        assert any(el.properties.get("name") == "StartupXYZ" for el in results)

    def test_organizations_with_type_technology(self, populated_store):
        hq = query(populated_store, "organizations with type technology")
        results = hq.execute()
        assert any(el.properties.get("name") == "TechCorp" for el in results)

    def test_organizations_with_type_security(self, populated_store):
        hq = query(populated_store, "organizations with type security")
        results = hq.execute()
        assert any(el.properties.get("name") == "SecureCorp" for el in results)

    def test_equipment_containing_rifle(self, populated_store):
        hq = query(populated_store, "equipment containing rifle")
        results = hq.execute()
        assert any("rifle" in el.properties.get("name", "").lower() for el in results)

    def test_equipment_containing_server(self, populated_store):
        hq = query(populated_store, "equipment containing server")
        results = hq.execute()
        assert any("server" in el.properties.get("name", "").lower() for el in results)

    def test_people_full_text_search_manager(self, populated_store):
        hq = query(populated_store, "people full text search manager")
        results = hq.execute()
        assert any("manager" in (el.properties.get("role", "") or "") for el in results)

    def test_people_full_text_search_security(self, populated_store):
        hq = query(populated_store, "people full text search security")
        results = hq.execute()
        assert any(
            "security" in (el.properties.get("role", "") or "") for el in results
        )

    def test_people_full_text_search_data(self, populated_store):
        hq = query(populated_store, "people full text search data")
        results = hq.execute()
        assert any("data" in (el.properties.get("role", "") or "") for el in results)

    def test_people_wildcard_example_domain(self, populated_store):
        hq = query(populated_store, "people wildcard search *@example.com on email")
        results = hq.execute()
        assert all(
            el.properties.get("email", "").endswith("@example.com") for el in results
        )

    def test_people_wildcard_specific_user(self, populated_store):
        hq = query(
            populated_store, "people wildcard search alice*@example.com on email"
        )
        results = hq.execute()
        assert any(el.properties.get("email") == "alice@example.com" for el in results)

    def test_people_search_by_name_alice(self, populated_store):
        hq = query(populated_store, "people search by name alice")
        results = hq.execute()
        assert any(el.properties.get("name") == "Alice Johnson" for el in results)

    def test_people_search_by_name_charlie(self, populated_store):
        hq = query(populated_store, "people search by name charlie")
        results = hq.execute()
        assert any(el.properties.get("name") == "Charlie Brown" for el in results)

    def test_people_containing_ev(self, populated_store):
        hq = query(populated_store, "people containing ev")
        results = hq.execute()
        assert any("ev" in el.properties.get("name", "").lower() for el in results)

    def test_people_containing_an_but_not_bob(self, populated_store):
        hq = query(populated_store, "people containing an but not bob")
        results = hq.execute()
        names = [el.properties.get("name", "").lower() for el in results]
        assert any("an" in n for n in names)
        assert all("bob" not in n for n in names)

    def test_all_people_sorted_by_name(self, populated_store):
        hq = query(populated_store, "people sorted by name ascending")
        results = hq.execute()
        names = [el.properties.get("name") for el in results]
        assert names == sorted(names)

    def test_people_sorted_by_age_desc(self, populated_store):
        hq = query(populated_store, "people sorted by age desc")
        results = hq.execute()
        ages = [el.properties.get("age") for el in results]
        assert ages == sorted(ages, reverse=True)

    def test_top_1_person_by_age_desc(self, populated_store):
        hq = query(populated_store, "top 1 people sorted by age desc")
        results = hq.execute()
        assert len(results) == 1

    def test_first_2_people_by_name(self, populated_store):
        hq = query(populated_store, "first 2 people sorted by name ascending")
        results = hq.execute()
        assert len(results) <= 2

    def test_last_3_people_by_age(self, populated_store):
        hq = query(populated_store, "people sorted by age asc last 3")
        results = hq.execute()
        assert len(results) <= 3

    def test_organizations_sorted_by_employees_desc(self, populated_store):
        hq = query(populated_store, "organizations sorted by employees desc")
        results = hq.execute()
        counts = [el.properties.get("employees", 0) for el in results]
        assert counts == sorted(counts, reverse=True)

    def test_top_2_organizations_by_employees(self, populated_store):
        hq = query(populated_store, "top 2 organizations sorted by employees desc")
        results = hq.execute()
        assert len(results) == 2

    def test_equipment_sorted_by_year_desc(self, populated_store):
        hq = query(populated_store, "equipment sorted by year desc")
        results = hq.execute()
        years = [el.properties.get("year") for el in results]
        assert years == sorted(years, reverse=True)

    def test_equipment_first_1_sorted_by_value(self, populated_store):
        hq = query(populated_store, "equipment sorted by value desc first 1")
        results = hq.execute()
        assert len(results) == 1

    def test_elements_related_to_person_2(self, populated_store):
        hq = query(populated_store, "elements related to person 2")
        results = hq.execute()
        assert isinstance(results, list)

    def test_colleagues_of_person_2(self, populated_store):
        hq = query(populated_store, "colleagues of person 2")
        results = hq.execute()
        assert isinstance(results, list)

    def test_elements_related_to_person_via_phrase(self, populated_store):
        """Ensure 'via <edge phrase>' is parsed and edge_label is set.

        This exercises the extended related-to pattern that maps the
        trailing phrase through edge meta aliases into HumanQuery.edge_label.
        """
        parser = NaturalQueryParser(populated_store)

        # We don't assert a specific edge_label value here because that
        # depends on meta configuration; we only verify that the parser
        # recognizes the pattern and preserves a related_element_id.
        hq = parser.parse("elements related to person 1 via works at")
        assert hq.related_element_id == "1"
        # edge_label may be None if no matching edge meta exists, but the
        # attribute must be present on HumanQuery.
        assert hasattr(hq, "edge_label")

    def test_find_people_younger_than_35(self, populated_store):
        """find people younger than 35."""
        hq = query(populated_store, "find people younger than 35")
        results = hq.execute()

        assert any(el.properties.get("name") == "Bob Smith" for el in results)
        assert all(el.properties.get("age", 0) < 35 for el in results)

    def test_list_equipment_containing_laptop(self, populated_store):
        """list equipment containing laptop."""
        hq = query(populated_store, "list equipment containing laptop")
        results = hq.execute()

        assert len(results) == 1
        assert results[0].properties.get("name") == "Laptop Dell XPS"

    def test_show_organizations_sorted_by_employees_desc(self, populated_store):
        """show organizations sorted by employees descending."""
        hq = query(
            populated_store,
            "show organizations sorted by employees descending",
        )
        results = hq.execute()

        counts = [el.properties.get("employees", 0) for el in results]
        assert counts == sorted(counts, reverse=True)


if __name__ == "__main__":
    pytest.main([__file__, "-v"])
