"""ham.genai.models.language.types.language_model_response"""

from __future__ import annotations

from typing import (
    List,
    TypeVar,
    Optional,
    Any,
    Dict,
    Union,
    Literal,
)

try:
    from ham.core.cache import cached
    from ham.core.typing import get_type_description
except ImportError:
    from ......cache import cached  # type: ignore
    from ......typing import get_type_description  # type: ignore

from ...model_provider import litellm
from ....types.base import BaseGenAIModelResponse
from openai.types.chat import (
    ChatCompletionContentPartParam,
    ChatCompletionMessageParam,
    ChatCompletionMessageToolCall,
)

__all__ = [
    "LanguageModelResponse",
]


T = TypeVar("T")


class LanguageModelResponse(BaseGenAIModelResponse[T]):
    """A response generated by a language model. This response is unified
    to represent both standard completions as well as structured outputs."""

    type: Literal["language_model"] = "language_model"

    model: str
    """The model that generated this response."""

    output: T
    """The 'final' or primary response content from the language model, this is
    in the type requested by the user.
    
    NOTE:
    In many cases with tool calling, message content is not present, in these cases
    this field will **NOT** represent tool calls, and will be returned as `None`."""

    completion: Any
    """The raw Chat Completion (`litellm.ModelResponse`) object returned by the
    language model."""

    content: Optional[str] = None
    """The actual response content of the completion. This is the string that
    was generated by the model."""

    tool_calls: Optional[List["litellm.ChatCompletionMessageToolCall"]] = None
    """The tool calls that were made by the model. This is a list of tool calls
    that were made by the model."""

    refusal: Optional[str] = None
    """The refusal message generated by the model. This is the string that
    was generated by the model when it refused to generate the completion."""

    def get_content(
        self, choice: int = 0
    ) -> Union[str, List["ChatCompletionContentPartParam"], None]:
        """The 'raw' message content generated by the language model, this
        can be either a string, a list of content parts, or `None`."""

        if not self.completion or not self.completion.choices:
            return None

        if choice >= len(self.completion.choices):
            return None

        return self.completion.choices[choice].message.content

    def get_tool_calls(
        self, *, name: Optional[str] = None, id: Optional[str] = None, choice: int = 0
    ) -> Optional[List["ChatCompletionMessageToolCall"]]:
        """The tool calls generated by the language model, this is a list of
        `ChatCompletionMessageToolCall` objects. Optionally can be filtered
        by name or ID to return specific tool calls.

        NOTE: Only one of `name` or `id` can be provided."""
        if not self.completion or not self.completion.choices:
            return None

        if choice >= len(self.completion.choices):
            return None

        tool_calls = self.completion.choices[choice].message.tool_calls
        if not tool_calls:
            return None

        # Filter by name or id if provided
        if name is not None and id is not None:
            raise ValueError("Only one of 'name' or 'id' can be provided, not both")

        if name is not None:
            return [call for call in tool_calls if call.function.name == name]

        if id is not None:
            return [call for call in tool_calls if call.id == id]

        return tool_calls

    def has_tool_calls(self, name: Optional[str] = None, choice: int = 0) -> bool:
        """Checks if the response has tool calls, optionally filtered by name.
        If `name` is provided, it will check if the tool call with that name
        exists in the response."""
        if not self.completion or not self.completion.choices:
            return False

        tool_calls = self.get_tool_calls(name=name, choice=choice)
        return tool_calls is not None and len(tool_calls) > 0

    def get_tool_call_parameters(
        self, tool: Optional[str] = None, choice: int = 0
    ) -> Optional[Dict[str, Any]]:
        """Returns the generated parameters for a tool call within a response.
        If the response has multiple tool calls, and no tool is specified,
        an error will be raised.

        Args:
            tool: The name of the tool to get the parameters for.
            choice: The choice index to get tool calls from.

        Returns:
            Dict[str, Any]: The generated parameters for the tool call.
        """
        tool_calls = self.get_tool_calls(choice=choice)
        if not tool_calls:
            return None

        if tool is None:
            if len(tool_calls) > 1:
                raise ValueError(
                    "Multiple tool calls found in response, and no tool specified."
                )
            tool = tool_calls[0].function.name

        for tool_call in tool_calls:
            if tool_call.function.name == tool:
                import json

                return json.loads(tool_call.function.arguments)
        return None

    def to_message(
        self, format_tool_calls: bool = False, choice: int = 0
    ) -> "ChatCompletionMessageParam":
        """Converts the LanguageModelResponse to a Chat Completions
        message object.

        If the `format_tool_calls` parameter is True, the tool calls
        will be cleanly formatted and added to the message content
        with something similar to:

        'I called the function `get_weather` with the following arguments:
        {arguments}'
        """
        if not self.completion or not self.completion.choices:
            return {"role": "assistant", "content": ""}

        if choice >= len(self.completion.choices):
            return {"role": "assistant", "content": ""}

        choice_message = self.completion.choices[choice].message

        # Base message structure
        message: "ChatCompletionMessageParam" = {
            "role": "assistant",
            "content": choice_message.content or "",
        }

        # Add tool calls if they exist and format_tool_calls is False
        if choice_message.tool_calls and not format_tool_calls:
            message["tool_calls"] = choice_message.tool_calls

        # Format tool calls into content if format_tool_calls is True
        elif choice_message.tool_calls and format_tool_calls:
            content_parts = []
            if choice_message.content:
                content_parts.append(choice_message.content)

            for tool_call in choice_message.tool_calls:
                formatted_call = f"I called the function `{tool_call.function.name}` with the following arguments:\n{tool_call.function.arguments}"
                content_parts.append(formatted_call)

            message["content"] = "\n\n".join(content_parts)

        return message

    def to_content_part(self) -> "ChatCompletionContentPartParam":
        """Converts the LanguageModelResponse to a Chat Completions
        content part object."""
        return {"type": "text", "text": self.content}

    @cached
    def __str__(self) -> str:
        """Pretty prints the response object."""
        output = ">>> LanguageModelResponse:"

        if self.output or self.content:
            output += f"\n{self.output if self.output else self.content}"
        else:
            output += f"\n{self.completion}"

        output += f"\n\n>>> Model: {self.model}"
        output += f"\n>>> Type: {get_type_description(type(self.output))}"
        output += f"\n>>> Tool Calls: {len(self.tool_calls) if self.tool_calls else 0}"

        return output
