import argparse
import sys
from argparse import Namespace
from enum import Enum
from typing import Optional


class MatchType(Enum):
    BEGIN = "begin"
    CONTAIN = "contain"
    END = "end"
    REGEX = "regex"


class HashCommitArgs(Namespace):
    hash: Optional[str]
    message: Optional[str]
    match_type: MatchType
    version: bool
    verbose: int
    overwrite: bool
    no_preserve_author: bool
    commit: Optional[str]


def parse_args() -> HashCommitArgs:
    parser = argparse.ArgumentParser(
        description="Generate a Git commit with a specific hash prefix."
    )
    parser.add_argument("--hash", help="Desired hash string.", type=str)
    parser.add_argument("--message", help="Commit message.", type=str)
    parser.add_argument(
        "--match-type",
        type=lambda mt: MatchType[mt.upper()],
        choices=list(MatchType),
        default=MatchType.BEGIN,
        help="Match type for the hash (begin, contain, end, regex). If 'regex', the --hash argument is treated as a regular expression.",
    )
    parser.add_argument(
        "--version", action="store_true", help="Show the version of hashcommit."
    )
    parser.add_argument(
        "-v", "--verbose", action="count", default=0, help="Increase verbosity level."
    )
    parser.add_argument(
        "--overwrite",
        action="store_true",
        help="Overwrite the existing commit instead of creating a new one.",
    )
    parser.add_argument(
        "--no-preserve-author",
        action="store_true",
        help="Do not preserve the original commit author when overwriting.",
    )
    parser.add_argument(
        "--commit",
        help="Commit hash to overwrite. If not provided, the last commit will be used.",
        type=str,
    )
    if len(sys.argv) == 1:
        parser.print_help(sys.stdout)
        sys.exit(0)
    return parser.parse_args(namespace=HashCommitArgs())
