from typing import Any, Dict, List, NamedTuple
import logging
import time
import datetime

import aiobotocore.session
import newrelic.agent


logger = logging.getLogger(__name__)


class FeatureRetrievalMeta(NamedTuple):
    cache_misses: int
    retrieval_ms: float
    success: bool
    cache_delay_minutes: float


@newrelic.agent.function_trace()
async def async_batch_get(
    dynamo_client, table_name: str, keys: List[Dict[str, Any]]
) -> List[Dict[str, Any]]:
    """
    Asynchronous batch_get_item with chunking for requests > 100 keys
    and handling for unprocessed keys.
    """
    all_items: List[Dict[str, Any]] = []
    # DynamoDB's BatchGetItem has a 100-item limit per request.
    CHUNK_SIZE = 100

    # Split the keys into chunks of 100
    for i in range(0, len(keys), CHUNK_SIZE):
        chunk_keys = keys[i : i + CHUNK_SIZE]
        to_fetch = {table_name: {"Keys": chunk_keys}}

        # Inner loop to handle unprocessed keys for the current chunk
        # Max retries of 3
        retries = 3
        while to_fetch and retries > 0:
            retries -= 1
            try:
                resp = await dynamo_client.batch_get_item(RequestItems=to_fetch)

                if "Responses" in resp and table_name in resp["Responses"]:
                    all_items.extend(resp["Responses"][table_name])

                unprocessed = resp.get("UnprocessedKeys", {})
                # If there are unprocessed keys, set them to be fetched in the next iteration
                if unprocessed and unprocessed.get(table_name):
                    logger.warning(
                        "Retrying %d unprocessed keys.",
                        len(unprocessed[table_name]["Keys"]),
                    )
                    to_fetch = unprocessed
                else:
                    # All keys in the chunk were processed, exit the inner loop
                    to_fetch = {}

            except Exception as e:
                logger.error("Error during batch_get_item for a chunk: %s", e)
                # Stop trying to process this chunk on error and move to the next
                to_fetch = {}

    return all_items


@newrelic.agent.function_trace()
def parse_dynamo_item(item: Dict[str, Any]) -> Dict[str, Any]:
    """Parse a DynamoDB attribute map (low-level) to Python types."""
    out: Dict[str, Any] = {}
    for k, v in item.items():
        if "N" in v:
            out[k] = float(v["N"])
        elif "S" in v:
            out[k] = v["S"]
        elif "SS" in v:
            out[k] = v["SS"]
        elif "NS" in v:
            out[k] = [float(n) for n in v["NS"]]
        elif "BOOL" in v:
            out[k] = v["BOOL"]
        elif "NULL" in v:
            out[k] = None
        elif "L" in v:
            out[k] = [parse_dynamo_item({"value": i})["value"] for i in v["L"]]
        elif "M" in v:
            out[k] = parse_dynamo_item(v["M"])
    return out


@newrelic.agent.function_trace()
async def set_stream_features(
    *,
    streams: List[Dict[str, Any]],
    stream_features: List[str],
    features_cache,
    features_table: str,
    stream_pk_prefix: str,
    cache_sep: str,
    aio_session: aiobotocore.session.Session | None = None,
) -> FeatureRetrievalMeta:
    time_start = time.perf_counter_ns()
    """Fetch missing features for streams from DynamoDB and fill them into streams."""
    if not streams or not stream_features:
        return FeatureRetrievalMeta(
            cache_misses=0,
            retrieval_ms=(time.perf_counter_ns() - time_start) * 1e-6,
            success=True,
            cache_delay_minutes=0,
        )

    cache_miss: Dict[str, Dict[str, Any]] = {}
    cache_delay_obj: dict[str, float] = {f: 0 for f in stream_features}
    now = datetime.datetime.utcnow()
    for f in stream_features:
        for s in streams:
            key = f"{s['streamUrl']}{cache_sep}{f}"
            if key in features_cache:
                # Only set if value is not None
                cached = features_cache.get(key)
                if cached["value"] is not None:
                    s[f] = cached["value"]
                    cache_delay_obj[f] = max(
                        cache_delay_obj[f], (now - cached["updated_at"]).total_seconds()
                    )
            else:
                cache_miss[key] = s
    valid_cache_delays = list(v for v in cache_delay_obj.values() if v > 0)
    cache_delay = min(valid_cache_delays) if valid_cache_delays else 0

    if not cache_miss:
        return FeatureRetrievalMeta(
            cache_misses=0,
            retrieval_ms=(time.perf_counter_ns() - time_start) * 1e-6,
            success=True,
            cache_delay_minutes=cache_delay / 60,
        )
    cache_misses = len(cache_miss)
    logger.info("Cache miss for %d items", cache_misses)

    # Prepare keys
    keys = []
    for k in cache_miss.keys():
        stream_url, sk = k.split(cache_sep, 1)
        pk = f"{stream_pk_prefix}{stream_url}"
        keys.append({"pk": {"S": pk}, "sk": {"S": sk}})

    session = aio_session or aiobotocore.session.get_session()
    async with session.create_client("dynamodb") as dynamodb:
        try:
            items = await async_batch_get(dynamodb, features_table, keys)
        except Exception as e:
            logger.error("DynamoDB batch_get failed: %s", e)
            return FeatureRetrievalMeta(
                cache_misses=cache_misses,
                retrieval_ms=(time.perf_counter_ns() - time_start) * 1e-6,
                success=False,
                cache_delay_minutes=cache_delay / 60,
            )

    updated_keys = set()
    for item in items:
        stream_url = item["pk"]["S"].removeprefix(stream_pk_prefix)
        feature_name = item["sk"]["S"]
        cache_key = f"{stream_url}{cache_sep}{feature_name}"
        parsed = parse_dynamo_item(item)

        features_cache[cache_key] = {
            "value": parsed.get("value"),
            "cache_ttl_in_seconds": int(parsed.get("cache_ttl_in_seconds", -1)),
            "updated_at": datetime.datetime.fromisoformat(
                parsed.get("updated_at")
            ).replace(tzinfo=None),
        }
        if cache_key in cache_miss:
            cache_miss[cache_key][feature_name] = parsed.get("value")
            updated_keys.add(cache_key)

    # Save keys that were not found in DynamoDB with None value
    if len(updated_keys) < len(cache_miss):
        missing_keys = set(cache_miss.keys()) - updated_keys
        for k in missing_keys:
            features_cache[k] = {"value": None, "cache_ttl_in_seconds": 300}
    return FeatureRetrievalMeta(
        cache_misses=cache_misses,
        retrieval_ms=(time.perf_counter_ns() - time_start) * 1e-6,
        success=True,
        cache_delay_minutes=cache_delay / 60,
    )
