import logging
from collections.abc import Sequence
from copy import deepcopy

from hdx.location.adminlevel import AdminLevel
from hdx.location.country import Country

from ..outputs.base import BaseOutput
from ..runner import Runner
from .sources import Sources

try:
    from pandas import DataFrame
except ImportError:
    DataFrame = None

logger = logging.getLogger(__name__)


class Writer:
    """obtain output data and write it to specified outputs.

    Args:
        runner: Runner object
        outputs: Mapping from names to output objects
    """

    regional_headers = (("regionnames",), ("#region+name",))
    national_headers = (
        ["iso3", "countryname"],
        ["#country+code", "#country+name"],
    )
    subnational_headers = {
        1: (
            ("iso3", "countryname", "adm1_pcode", "adm1_name"),
            ("#country+code", "#country+name", "#adm1+code", "#adm1+name"),
        ),
        2: (
            ("iso3", "countryname", "adm2_pcode", "adm2_name"),
            ("#country+code", "#country+name", "#adm2+code", "#adm2+name"),
        ),
        3: (
            ("iso3", "countryname", "adm3_pcode", "adm3_name"),
            ("#country+code", "#country+name", "#adm3+code", "#adm3+name"),
        ),
        4: (
            ("iso3", "countryname", "adm4_pcode", "adm4_name"),
            ("#country+code", "#country+name", "#adm4+code", "#adm4+name"),
        ),
    }
    sources_headers = (
        ("Indicator", "Date", "Source", "Url"),
        ("#indicator+name", "#date", "#meta+source", "#meta+url"),
    )

    def __init__(self, runner: Runner, outputs: dict[str, BaseOutput]):
        self.runner = runner
        self.outputs = outputs

    def update(self, name: str, data: list | DataFrame) -> None:
        """Update JSON key or tab in Excel or Google Sheets.

        Args:
            name: Name of tab (key in JSON) to update
            data: Data to output

        Returns:
            None
        """

        if not data:
            return
        logger.info(f"Updating tab: {name}")
        for output in self.outputs.values():
            output.update_tab(name, data)

    def get_toplevel_rows(
        self,
        names: Sequence[str] | None = None,
        overrides: dict[str, dict] = {},
        toplevel: str = "allregions",
    ) -> list[list]:
        """Get rows for the given toplevel for scrapers limiting to those in
        names if given. Rows include header row, HXL hashtag row and a value
        row.  Sometimes it may be necessary to map alternative level names to
        the top level and this can be done using overrides. It is a dictionary
        with keys being scraper names and values being dictionaries which map
        level names to output levels.

        Args:
            names: Names of scrapers. Default is None.
            overrides: Dictionary mapping scrapers to level mappings. Default is {}.
            toplevel: Name of top level such as "global". Default is "allregions".

        Returns:
            Rows for a given level
        """
        return self.runner.get_rows(
            toplevel, ("value",), names=names, overrides=overrides
        )

    def get_regional_rows(
        self,
        regional: Sequence[str],
        names: Sequence[str] | None = None,
        overrides: dict[str, dict] = {},
        level: str = "regional",
    ):
        """Get regional rows for scrapers limiting to those in names if given using the
        level name given by level. Rows include header row, HXL hashtag row and value rows,
        one for each regional admin unit. The parameter regional is a list of regional
        admin names. Sometimes it may be necessary to map alternative level names to the
        regional level and this can be done using overrides. It is a dictionary with keys
        being scraper names and values being dictionaries which map level names to output
        levels.

        Args:
            regional: Regional admin names
            names: Names of scrapers. Default is None.
            overrides: Dictionary mapping scrapers to level mappings. Default is {}.
            level: Name of regional level. Default is "regional".

        Returns:
            Rows for a given level
        """
        return self.runner.get_rows(
            level,
            regional,
            self.regional_headers,
            (lambda adm: adm,),
            names=names,
            overrides=overrides,
        )

    def update_toplevel(
        self,
        toplevel_rows: list[list],
        tab: str = "allregions",
        regional_rows: list[list] | None = None,
        regional_adm: str = "ALL",
        regional_hxltags: Sequence[str] | None = None,
        regional_first: bool = False,
    ) -> None:
        """Update the top level tab (or key in JSON) in the outputs. Optionally, further
        rows to output as top level can be obtained from the regional rows.

        Args:
            toplevel_rows: Header row, HXL tags row and top level value row
            tab: Name of tab (key in JSON) to update. Default is "allregions".
            regional_rows: Header, HXL tags and regional values. Default is None.
            regional_adm: The admin name of the top level in the regional data. Default is "ALL".
            regional_hxltags: What regional HXL tags to include. Default is None (all tags).
            regional_first: Whether regional rows are output before top level rows. Default is False.

        Returns:
            None
        """
        if regional_rows is None:
            regional_rows = []
        if not toplevel_rows:
            toplevel_rows = [[], [], []]
        if regional_rows:
            adm_header = regional_rows[1].index("#region+name")
            rows_to_insert = ([], [], [])
            for row in regional_rows[2:]:
                if row[adm_header] == regional_adm:
                    for i, hxltag in enumerate(regional_rows[1]):
                        if hxltag == "#region+name":
                            continue
                        if regional_hxltags and hxltag not in regional_hxltags:
                            continue
                        rows_to_insert[0].append(regional_rows[0][i])
                        rows_to_insert[1].append(hxltag)
                        rows_to_insert[2].append(row[i])
            if regional_first:
                toplevel_rows[0] = rows_to_insert[0] + toplevel_rows[0]
                toplevel_rows[1] = rows_to_insert[1] + toplevel_rows[1]
                toplevel_rows[2] = rows_to_insert[2] + toplevel_rows[2]
            else:
                toplevel_rows[0] += rows_to_insert[0]
                toplevel_rows[1] += rows_to_insert[1]
                toplevel_rows[2] += rows_to_insert[2]
        self.update(tab, toplevel_rows)

    def update_regional(
        self,
        regional_rows: list[list],
        toplevel_rows: list[list] | None = None,
        toplevel_hxltags: Sequence[str] | None = None,
        tab: str = "regional",
        toplevel: str = "allregions",
    ) -> None:
        """Update the regional tab (or key in JSON) in the outputs. Optionally, further
        rows to output as regional can be obtained from the top level rows.

        Args:
            regional_rows: Header row, HXL tags row and regional value rows
            toplevel_rows: Header, HXL tags and top level values. Default is None.
            toplevel_hxltags: What top level HXL tags to include. Default is None (all tags).
            tab: Name of tab (key in JSON) to update. Default is "regional".
            toplevel: Name of top level such as "global". Default is "allregions".

        Returns:
            None
        """
        if not regional_rows:
            return
        toplevel_values = {}
        toplevel_headers = {}
        if toplevel_rows:
            for i, hxltag in enumerate(toplevel_rows[1]):
                if toplevel_hxltags and hxltag not in toplevel_hxltags:
                    continue
                toplevel_values[hxltag] = toplevel_rows[2][i]
                if hxltag not in regional_rows[1]:
                    toplevel_headers[hxltag] = toplevel_rows[0][i]
        adm_header = regional_rows[1].index("#region+name")
        found_adm = False

        def add_value(row):
            value_found = False
            for i, hxltag in enumerate(regional_rows[1]):
                value = toplevel_values.get(hxltag)
                if value is None:
                    continue
                row[i] = value
                value_found = True
            for hxltag, header in toplevel_headers.items():
                value = toplevel_values.get(hxltag)
                if value is None:
                    continue
                regional_rows[0].append(header)
                regional_rows[1].append(hxltag)
                row.append(value)
                value_found = True
            return value_found

        for row in regional_rows[2:]:
            if row[adm_header] == toplevel:
                add_value(row)
                found_adm = True
                break
        if not found_adm:
            row = [toplevel]
            for _ in regional_rows[0][1:]:
                row.append(None)
            if add_value(row):
                regional_rows.append(row)
        length = len(regional_rows[0])
        for row in regional_rows[2:]:
            while len(row) < length:
                row.append(None)
        self.update(tab, regional_rows)

    def update_national(
        self,
        countries: Sequence[str],
        names: Sequence[str] | None = None,
        flag_countries: dict | None = None,
        iso3_to_region: dict | None = None,
        ignore_regions: Sequence[str] = tuple(),
        level="national",
        tab="national",
    ) -> None:
        """Update the national tab (or key in JSON) in the outputs for scrapers limiting to
        those in names. Certain additional columns can be added. One shows countries to be
        flagged (given a Y or N) and is configured using flag_countries, a dictionary which
        has keys header, hxltag and countries (whose corresponding value is a list or tuple
        of countries). Another shows regions a country is in and is specified by the mapping
        iso3_to_region. Some regions can be ignored using ignore_regions.

        Args:
            countries: Country names
            names: Names of scrapers. Default is None.
            flag_countries: Countries to flag. Default is None.
            iso3_to_region: Mapping from iso3 to region. Default is None.
            ignore_regions: Regions to ignore. Default is tuple().
            level: Name of national level. Default is "national".
            tab: Name of tab (key in JSON) to update. Default is "national".

        Returns:
            None
        """
        headers = deepcopy(self.national_headers)
        fns = [
            lambda adm: adm,
            Country.get_country_name_from_iso3,
        ]

        if flag_countries:
            headers[0].append(flag_countries["header"])
            headers[1].append(flag_countries["hxltag"])

            def isfc_fn(adm):
                return "Y" if adm in flag_countries["countries"] else "N"

            fns.append(isfc_fn)

        if iso3_to_region:
            headers[0].append("region")
            headers[1].append("#region+name")

            def region_fn(adm):
                regions = sorted(iso3_to_region[adm])
                for region in reversed(regions):
                    if ignore_regions and region in ignore_regions:
                        regions.remove(region)
                return "|".join(regions)

            fns.append(region_fn)

        rows = self.runner.get_rows(level, countries, headers, fns, names=names)
        if rows:
            self.update(tab, rows)

    def update_subnational(
        self,
        adminlevel: AdminLevel,
        names: Sequence[str] | None = None,
        level: str = "subnational",
        tab: str = "subnational",
    ) -> None:
        """Update the subnational tab (or key in JSON) in the outputs for scrapers limiting
        to those in names.

        Args:
            adminlevel: AdminLevel object from HDX Python Country library
            names: Names of scrapers. Default is None.
            level: Name of subnational level. Default is "subnational".
            tab: Name of tab (key in JSON) to update. Default is "subnational".

        Returns:
            None
        """

        def get_country_name(adm):
            countryiso3 = adminlevel.pcode_to_iso3[adm]
            return Country.get_country_name_from_iso3(countryiso3)

        fns = (
            lambda adm: adminlevel.pcode_to_iso3[adm],
            get_country_name,
            lambda adm: adm,
            lambda adm: adminlevel.pcode_to_name[adm],
        )
        rows = self.runner.get_rows(
            level,
            adminlevel.pcodes,
            self.subnational_headers[
                adminlevel.admin_level
            ],  # use the main admin level (don't worry about overrides)
            fns,
            names=names,
        )
        self.update(tab, rows)

    def update_sources(
        self,
        additional_sources: Sequence[dict] = tuple(),
        names: Sequence[str] | None = None,
        secondary_runner: Runner | None = None,
        custom_sources: Sequence[tuple] = tuple(),
        tab: str = "sources",
        should_overwrite_sources: bool | None = None,
        sources_to_delete: Sequence[str] = tuple(),
    ) -> None:
        """Update the sources tab (or key in JSON) in the outputs for scrapers
        limiting to those in names. Additional sources can be added. Each is a
        dictionary with indicator (specified with HXL hashtag), dataset or
        source and source_url as well as the source_date or whether to
        force_date_today. Custom sources can be directly passed to be appended.
        They are of form (indicator, date, source, source_url). By default, if
        the same indicator (HXL hashtag) appears more than once in the list of
        sources, then the first is used, but there is the option to enable
        overwriting of sources. A list of sources to delete can be supplied
        where each source is specified using its HXL hashtag (or a part of the
        hashtag).

        Args:
            additional_sources: Additional sources to add. Default is empty tuple.
            names: Names of scrapers. Default is None.
            secondary_runner: Secondary Runner object. Default is None.
            custom_sources: Custom sources to add. Default is empty tuple.
            tab: Name of tab (key in JSON) to update. Default is "sources".
            should_overwrite_sources: Whether to overwrite sources. Default is None (class default).
            sources_to_delete: Sources tags to delete. Default is empty tuple.

        Returns:
            None
        """
        sources = self.runner.get_sources(
            names=names,
            additional_sources=additional_sources,
            should_overwrite_sources=should_overwrite_sources,
        )
        hxltags = [source[0] for source in sources]
        if secondary_runner:
            Sources.add_sources_overwrite(
                hxltags,
                sources,
                secondary_runner.get_sources(),
                logger,
                should_overwrite_sources,
            )
        Sources.add_sources_overwrite(
            hxltags,
            sources,
            custom_sources,
            logger,
            should_overwrite_sources,
        )
        for hxltag in sources_to_delete:
            for i in range(len(sources) - 1, -1, -1):
                if hxltag in sources[i][0]:
                    del sources[i]

        self.update(tab, list(self.sources_headers) + sources)
