# HTTP Status Code Exception Library

The **HTTP Status Code Exception Library** is a Python library designed to provide a structured and object-oriented approach to handling HTTP status codes as exceptions. This library simplifies the process of working with HTTP responses by mapping each status code to a corresponding exception class.

---

## Table of Contents

1. [Features](#features)
2. [Installation](#installation)
3. [Usage](#usage)
4. [Examples](#examples)
5. [Contributing](#contributing)
6. [License](#license)

---

## Features

- **Comprehensive Coverage**: Includes all standard HTTP status codes (1xx, 2xx, 3xx, 4xx, 5xx).
- **Object-Oriented Design**: Each status code is represented as a distinct exception class.
- **Customizable Messages**: Allows for custom error messages when raising exceptions.
- **Easy Integration**: Designed to be easily integrated into web applications or APIs.
- **Extensible**: Easily extendable to include additional functionality or custom status codes.

---

## Installation

You can install the library using `pip`:

```bash
pip install http-status-code-exception
```

Alternatively, if you are using [Poetry](https://python-poetry.org/), add it to your project:

```bash
poetry add http-status-code-exception
```

---

## Usage

The library provides a base exception class `HTTPException` and specific subclasses for each HTTP status code. You can import and use these classes directly in your application.

### Importing Exceptions

```python
from http_status_code_exception.client_error import NotFound
```

### Raising Exceptions

```python
raise NotFound("The requested resource was not found")
```

### Retrieving Exceptions Dynamically

If you need to retrieve an exception class based on a status code, use the `get` function:

```python
from http_status_code_exception import get

status_code: int = 404
exception_class = get(status_code)
raise exception_class("Page not found")
```

---

## Examples

### Example 1: Handling Client Errors

```python
from http_status_code_exception.client_error import BadRequest, Unauthorized, Forbidden

def handle_client_errors():
    try:
        # Simulate a bad request
        raise BadRequest("Invalid request data")
    except BadRequest as e:
        print(f"Bad Request: {e}")
    except Unauthorized as e:
        print(f"Unauthorized: {e}")
    except Forbidden as e:
        print(f"Forbidden: {e}")
```

### Example 2: Handling Server Errors

```python
from http_status_code_exception.error_server import InternalServerError, ServiceUnavailable

def handle_server_errors():
    try:
        # Simulate an internal server error
        raise InternalServerError("Database connection failed")
    except InternalServerError as e:
        print(f"Internal Server Error: {e}")
    except ServiceUnavailable as e:
        print(f"Service Unavailable: {e}")
```

---

## Contributing

 Contributions are welcome! If you find a bug or have a feature request, please open an issue or submit a pull request.

1. Fork the repository.
2. Create a new branch for your changes (`git checkout -b feature/new-feature`).
3. Commit your changes (`git commit -am 'Add new feature'`).
4. Push to the branch (`git push origin feature/new-feature`).
5. Submit a pull request.

---

## License

This project is licensed under the MIT License. See the [LICENSE](LICENSE) file for details.

---

## Acknowledgments

- Thanks to the Python community for inspiring this library.
- Special thanks to contributors who have helped improve this project.

---

Feel free to reach out if you have any questions or need further assistance!