"""
Module for handling HTTP Redirection responses (3xx).

This module provides a class to represent and handle redirection HTTP status codes (3xx).
These status codes indicate that further action needs to be taken by the user agent in order to fulfill the request.

Attributes:
    VERSION (str): Version of the module.

Classes:
    MultipleChoices
    MovedPermanently
    Found
    SeeOther
    NotModified
    UseProxy
    TemporaryRedirect
    PermanentRedirect

"""

VERSION: str = "1.0.0"

from http import HTTPStatus
from typing import Self

import http_status_code_exception.base_class

class MultipleChoices(http_status_code_exception.base_class.HTTPException):
    """300 Multiple Choices: The request has more than one possible responses."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.MULTIPLE_CHOICES, message)


class MovedPermanently(http_status_code_exception.base_class.HTTPException):
    """301 Moved Permanently: The resource has been moved permanently."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.MOVED_PERMANENTLY, message)


class Found(http_status_code_exception.base_class.HTTPException):
    """302 Found: The resource has been temporarily moved to another URI."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.FOUND, message)


class SeeOther(http_status_code_exception.base_class.HTTPException):
    """303 See Other: The response can be found under another URI."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.SEE_OTHER, message)


class NotModified(http_status_code_exception.base_class.HTTPException):
    """304 Not Modified: The resource has not been modified since the last request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NOT_MODIFIED, message)


class UseProxy(http_status_code_exception.base_class.HTTPException):
    """305 Use Proxy: The requested resource must be accessed through a proxy."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.USE_PROXY, message)


class TemporaryRedirect(http_status_code_exception.base_class.HTTPException):
    """307 Temporary Redirect: The resource has been temporarily moved to another URI."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.TEMPORARY_REDIRECT, message)


class PermanentRedirect(http_status_code_exception.base_class.HTTPException):
    """308 Permanent Redirect: The resource has been permanently moved to another URI."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.PERMANENT_REDIRECT, message)
