"""
Module for handling HTTP Server Error responses (5xx).

This module provides a class to represent and handle server error HTTP status codes (5xx).
These status codes indicate that the server failed to fulfill an apparently valid request.

Attributes:
    VERSION (str): Version of the module.

Classes:
    InternalServerError
    NotImplemented
    BadGateway
    ServiceUnavailable
    GatewayTimeout
    HttpVersionNotSupported
    VariantAlsoNegotiates
    InsufficientStorage
    LoopDetected
    NotExtended
    NetworkAuthenticationRequired

"""

VERSION: str = "1.0.0"

from http import HTTPStatus
from typing import Self

import http_status_code_exception.base_class

class InternalServerError(http_status_code_exception.base_class.HTTPException):
    """500 Internal Server Error: The server encountered an unexpected condition."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.INTERNAL_SERVER_ERROR, message)


class NotImplemented(http_status_code_exception.base_class.HTTPException):
    """501 Not Implemented: The server does not support the functionality required to fulfill the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NOT_IMPLEMENTED, message)


class BadGateway(http_status_code_exception.base_class.HTTPException):
    """502 Bad Gateway: The server received an invalid response while acting as a gateway or proxy."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.BAD_GATEWAY, message)


class ServiceUnavailable(http_status_code_exception.base_class.HTTPException):
    """503 Service Unavailable: The server is currently unable to handle the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.SERVICE_UNAVAILABLE, message)


class GatewayTimeout(http_status_code_exception.base_class.HTTPException):
    """504 Gateway Timeout: The server did not receive a timely response from an upstream server."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.GATEWAY_TIMEOUT, message)


class HTTPVersionNotSupported(http_status_code_exception.base_class.HTTPException):
    """505 HTTP Version Not Supported: The server does not support the HTTP protocol version used in the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.HTTP_VERSION_NOT_SUPPORTED, message)


class VariantAlsoNegotiates(http_status_code_exception.base_class.HTTPException):
    """506 Variant Also Negotiates: A variant for the requested resource exists and has itself been negotiated."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.VARIANT_ALSO_NEGOTIATES, message)


class InsufficientStorage(http_status_code_exception.base_class.HTTPException):
    """507 Insufficient Storage: The method could not be performed on the resource due to insufficient storage."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.INSUFFICIENT_STORAGE, message)


class LoopDetected(http_status_code_exception.base_class.HTTPException):
    """508 Loop Detected: The server detected an infinite loop while processing the request."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.LOOP_DETECTED, message)


class NotExtended(http_status_code_exception.base_class.HTTPException):
    """510 Not Extended: Further extensions to the request are required for the server to fulfill it."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NOT_EXTENDED, message)


class NetworkAuthenticationRequired(http_status_code_exception.base_class.HTTPException):
    """511 Network Authentication Required: The client needs to authenticate to gain network access."""
    def __init__(self: Self, message: str | None = None) -> Self:
        super().__init__(HTTPStatus.NETWORK_AUTHENTICATION_REQUIRED, message)
