"""Schema utilities for tool definitions."""

from __future__ import annotations

import logging
from typing import Any

__all__ = [
    "ensure_strict_schema",
    "json_type_to_python",
    "schema_to_pydantic",
    "validate_openai_schema",
]

logger = logging.getLogger(__name__)


def ensure_strict_schema(schema: dict[str, Any]) -> dict[str, Any]:
    """Ensure a JSON schema is compatible with OpenAI's strict mode.

    OpenAI strict mode requires:
    - additionalProperties: false on all objects
    - All properties must be in required

    Args:
        schema: Original JSON schema.

    Returns:
        Modified schema for strict mode.
    """
    schema = dict(schema)

    if schema.get("type") == "object":
        schema["additionalProperties"] = False

        if "properties" in schema:
            # All properties must be required
            schema["required"] = list(schema["properties"].keys())

            # Recursively process nested objects
            for prop_schema in schema["properties"].values():
                if isinstance(prop_schema, dict):
                    _ensure_strict_recursive(prop_schema)

    return schema


def _ensure_strict_recursive(schema: dict[str, Any]) -> None:
    """Recursively apply strict mode to nested schemas."""
    if schema.get("type") == "object":
        schema["additionalProperties"] = False
        if "properties" in schema:
            schema["required"] = list(schema["properties"].keys())
            for prop_schema in schema["properties"].values():
                if isinstance(prop_schema, dict):
                    _ensure_strict_recursive(prop_schema)

    elif schema.get("type") == "array" and "items" in schema:
        if isinstance(schema["items"], dict):
            _ensure_strict_recursive(schema["items"])


def schema_to_pydantic(name: str, schema: dict[str, Any]) -> type:
    """Convert JSON schema to a Pydantic model.

    Args:
        name: Model name (used for class name).
        schema: JSON schema with properties.

    Returns:
        Dynamically created Pydantic model class.
    """
    from pydantic import Field, create_model

    properties = schema.get("properties", {})
    required = set(schema.get("required", []))

    fields = {}
    for prop_name, prop_schema in properties.items():
        prop_type = json_type_to_python(prop_schema.get("type", "string"))
        default = ... if prop_name in required else None
        description = prop_schema.get("description", "")
        fields[prop_name] = (prop_type, Field(default=default, description=description))

    return create_model(f"{name}Input", **fields)


def json_type_to_python(json_type: str) -> type:
    """Map JSON schema type to Python type.

    Args:
        json_type: JSON schema type string.

    Returns:
        Corresponding Python type.
    """
    mapping = {
        "string": str,
        "integer": int,
        "number": float,
        "boolean": bool,
        "array": list,
        "object": dict,
    }
    return mapping.get(json_type, str)


def validate_openai_schema(
    schema: dict[str, Any],
    tool_name: str = "unknown",
    path: str = "",
) -> list[str]:
    """Validate a JSON schema for OpenAI API compatibility.

    OpenAI's API has specific requirements for tool schemas:
    - Arrays must have 'items' (not 'prefixItems' which tuples generate)
    - Certain schema features like 'prefixItems' are not supported

    Args:
        schema: JSON schema to validate.
        tool_name: Name of the tool (for error messages).
        path: Current path in schema (for error context).

    Returns:
        List of validation error messages. Empty if valid.
    """
    errors: list[str] = []

    if not isinstance(schema, dict):
        return errors

    # Check for prefixItems (generated by tuple types)
    if "prefixItems" in schema:
        errors.append(
            f"Tool '{tool_name}' has 'prefixItems' at {path or 'root'} "
            "(likely from tuple type). Use list[Model] instead of tuple."
        )

    # Check arrays have 'items'
    if schema.get("type") == "array" and "items" not in schema and "prefixItems" not in schema:
        errors.append(
            f"Tool '{tool_name}' has array at {path or 'root'} without 'items'. "
            "OpenAI requires 'items' for array schemas."
        )

    # Recursively check nested schemas
    # Check properties
    if "properties" in schema:
        for prop_name, prop_schema in schema["properties"].items():
            prop_path = f"{path}.{prop_name}" if path else prop_name
            errors.extend(validate_openai_schema(prop_schema, tool_name, prop_path))

    # Check items
    if "items" in schema and isinstance(schema["items"], dict):
        items_path = f"{path}[items]" if path else "[items]"
        errors.extend(validate_openai_schema(schema["items"], tool_name, items_path))

    # Check anyOf/oneOf/allOf
    for key in ("anyOf", "oneOf", "allOf"):
        if key in schema:
            for i, sub_schema in enumerate(schema[key]):
                sub_path = f"{path}.{key}[{i}]" if path else f"{key}[{i}]"
                errors.extend(validate_openai_schema(sub_schema, tool_name, sub_path))

    # Check $defs (definitions)
    if "$defs" in schema:
        for def_name, def_schema in schema["$defs"].items():
            def_path = f"$defs.{def_name}"
            errors.extend(validate_openai_schema(def_schema, tool_name, def_path))

    return errors
