"""
This module provides classes and functions for setting up and running a Flask-based server
to interact with the HumatronWorker. It handles incoming HTTP requests, processes them, and returns responses.
"""

"""
" ██╗  ██╗██╗   ██╗███╗   ███╗ █████╗ ████████╗██████╗  ██████╗ ███╗   ██╗
" ██║  ██║██║   ██║████╗ ████║██╔══██╗╚══██╔══╝██╔══██╗██╔═══██╗████╗  ██║
" ███████║██║   ██║██╔████╔██║███████║   ██║   ██████╔╝██║   ██║██╔██╗ ██║
" ██╔══██║██║   ██║██║╚██╔╝██║██╔══██║   ██║   ██╔══██╗██║   ██║██║╚██╗██║
" ██║  ██║╚██████╔╝██║ ╚═╝ ██║██║  ██║   ██║   ██║  ██║╚██████╔╝██║ ╚████║
" ╚═╝  ╚═╝ ╚═════╝ ╚═╝     ╚═╝╚═╝  ╚═╝   ╚═╝   ╚═╝  ╚═╝ ╚═════╝ ╚═╝  ╚═══╝
"
"                   Copyright (C) 2023 Humatron, Inc.
"                          All rights reserved.
"""

import logging
from typing import NamedTuple, Optional

import flask
import werkzeug
from flask import Flask, abort, request, jsonify, make_response
from werkzeug.serving import make_server, BaseWSGIServer

from humatron.worker.classes import HumatronWorker, Request
from humatron.worker.rest_utils import check_request_token, set_response_token

_logger = logging.getLogger('humatron.sdk.flask')


class SSLContextData(NamedTuple):
    """
    Represents SSL context data for secure communication over HTTPS.
    """
    crt_path: str
    """The file path to the SSL certificate."""
    key_path: str
    """The file path to the SSL key."""


def make_flask_server(
    humatron_worker: HumatronWorker,
    req_token: str,
    resp_token: str,
    rest_host: str,
    rest_port: int,
    rest_url: str,
    methods: list[str] = None,
    ssl_ctx_data: Optional[SSLContextData] = None
) -> BaseWSGIServer:
    """
    Creates a Flask-based HTTP server to handle requests for the HumatronWorker.

    The server listens on the specified host and port, and processes incoming requests
    through the HumatronWorker instance. SSL can be configured for secure communication.

    @param humatron_worker :
        The worker instance that handles requests sent to the server.
    @param req_token :
        The token used for validating incoming requests (for security purposes).
    @param resp_token :
        The token to include in outgoing responses (for security purposes).
    @param rest_host :
        The host address where the server will listen (e.g., '0.0.0.0').
    @param rest_port :
        The port number where the server will listen for incoming requests.
    @param rest_url :
        The URL endpoint for processing requests (e.g., 'api/v1/worker').
    @param methods :
        The HTTP methods allowed for the endpoint (defaults to ['POST']).
    @param ssl_ctx_data :
        SSL context data for enabling HTTPS communication. If None, HTTP is used.

    @return:
        A WSGI server instance ready to handle incoming HTTP requests.
    """
    app = Flask(__name__)

    # Set up SSL context if provided
    ssl_context = (ssl_ctx_data.crt_path, ssl_ctx_data.key_path) if ssl_ctx_data else None
    srv = make_server(rest_host, rest_port, app, ssl_context=ssl_context)

    @app.route(f'/{rest_url}', methods=methods if methods else ['POST'])
    def _process() -> flask.Response:
        """
        Processes incoming HTTP requests and returns a response.

        This function validates the incoming request's token, processes the request using
        the HumatronWorker, and sets the appropriate response token before sending it back
        to the client.

        @return:
            The HTTP response generated by processing the request.
        """
        try:
            # Check if the request token is valid
            if not check_request_token(request.headers, req_token):
                abort(make_response(jsonify({}), 401))

            # Process the request using the HumatronWorker
            resp_spec = humatron_worker.post_request(Request.from_dict(request.json))

            # Prepare the response object
            resp_flask = flask.make_response(resp_spec.to_dict()) if resp_spec else flask.jsonify({})

            # Set the response token
            set_response_token(resp_flask.headers, resp_token)

            return resp_flask
        except werkzeug.exceptions.HTTPException as e:
            raise e
        except KeyError as e:
            abort(make_response(jsonify({'error': str(e)}), 400))
        except Exception as e:
            abort(make_response(jsonify({'error': str(e)}), 500))

    return srv


def start_flask_server(
    humatron_worker: HumatronWorker,
    req_token: str,
    resp_token: str,
    rest_host: str,
    rest_port: int,
    rest_url: str,
    methods: list[str] = None,
    ssl_ctx_data: Optional[SSLContextData] = None
) -> None:
    """
    Starts the Flask server and begins handling requests indefinitely.

    This function creates the Flask server, starts it, and logs the server details. It will
    continue to handle incoming requests until it is manually shut down.

    @param humatron_worker :
        The worker instance that handles requests.
    @param req_token :
        The token used for validating incoming requests.
    @param resp_token :
        The token to include in outgoing responses.
    @param rest_host :
        The host address for the server (e.g., '0.0.0.0').
    @param rest_port :
        The port number for the server.
    @param rest_url :
        The URL endpoint for handling requests.
    @param methods :
        HTTP methods allowed for the endpoint (defaults to ['POST']).
    @param ssl_ctx_data :
        SSL context data for HTTPS. If None, HTTP will be used.

    """
    srv = make_flask_server(
        humatron_worker, req_token, resp_token, rest_host, rest_port, rest_url, methods, ssl_ctx_data
    )

    # Determine the protocol (http or https) based on whether SSL context is provided
    protocol = 'http' if ssl_ctx_data is None else 'https'

    try:
        _logger.info(f'Server is running at {protocol}://{rest_host}:{rest_port}/{rest_url}')
        # Start the server and listen for requests indefinitely
        srv.serve_forever()
    finally:
        _logger.info('Shutting down the server.')
        humatron_worker.close()  # Clean up worker resources
        srv.shutdown()  # Shutdown the server
