"""hyper flow commands - Simplified render flows"""
import typer
from typing import Optional
from hypercli import HyperCLI
from .output import output, console, success, spinner

app = typer.Typer(help="Simplified render flows (text-to-speech, audio-to-text, etc.)")


def get_client() -> HyperCLI:
    return HyperCLI()


@app.command("text-to-speech")
def text_to_speech(
    text: str = typer.Argument(..., help="Text to synthesize"),
    mode: str = typer.Option("design", "--mode", "-m", help="TTS mode: custom, design, or clone"),
    language: str = typer.Option("Auto", "--language", "-l", help="Language (Auto, English, Chinese, etc.)"),
    # CustomVoice options
    speaker: Optional[str] = typer.Option(None, "--speaker", "-s", help="Speaker for custom mode (Ryan, Serena, etc.)"),
    style: Optional[str] = typer.Option(None, "--style", help="Style instruction for custom mode"),
    model_size: Optional[str] = typer.Option(None, "--model-size", help="Model size: 0.6B or 1.7B"),
    # VoiceDesign options
    voice_description: Optional[str] = typer.Option(None, "--voice-desc", "-d", help="Voice description for design mode"),
    # VoiceClone options
    ref_audio_url: Optional[str] = typer.Option(None, "--ref-audio", help="Reference audio URL for clone mode"),
    ref_text: Optional[str] = typer.Option(None, "--ref-text", help="Reference audio transcript for clone mode"),
    use_xvector_only: bool = typer.Option(False, "--xvector-only", help="Use x-vector only for clone mode"),
    # Output
    notify_url: Optional[str] = typer.Option(None, "--notify", help="Webhook URL for completion"),
    fmt: str = typer.Option("table", "--output", "-o", help="Output format: table|json"),
):
    """
    Generate speech from text using Qwen3-TTS.

    Three modes:

      design  - Describe any voice in natural language (default)

      custom  - Use predefined speakers (Ryan, Serena, etc.)

      clone   - Clone a voice from reference audio

    Examples:

      hyper flow text-to-speech "Hello world" -d "A young Indian male voice"

      hyper flow text-to-speech "Hello" -m custom -s Serena --style "Speak warmly"

      hyper flow text-to-speech "Hello" -m clone --ref-audio https://... --ref-text "..."
    """
    client = get_client()

    kwargs = dict(
        text=text,
        mode=mode,
        language=language,
        speaker=speaker,
        style=style,
        model_size=model_size,
        voice_description=voice_description,
        ref_audio_url=ref_audio_url,
        ref_text=ref_text,
        use_xvector_only=use_xvector_only if use_xvector_only else None,
        notify_url=notify_url,
    )

    with spinner("Creating TTS render..."):
        render = client.renders.text_to_speech(**kwargs)

    if fmt == "json":
        output(render, "json")
    else:
        console.print(f"[bold green]✓[/bold green] TTS render created: [cyan]{render.render_id}[/cyan]")
        console.print(f"  State: {render.state}")
        console.print(f"  Mode: {mode}")
        if render.result_url:
            console.print(f"  Result: {render.result_url}")


@app.command("audio-to-text")
def audio_to_text(
    audio_url: str = typer.Argument(..., help="URL of audio file to transcribe"),
    notify_url: Optional[str] = typer.Option(None, "--notify", help="Webhook URL for completion"),
    fmt: str = typer.Option("table", "--output", "-o", help="Output format: table|json"),
):
    """
    Transcribe audio to text using WhisperX.

    Example:

      hyper flow audio-to-text https://example.com/recording.mp3
    """
    client = get_client()

    with spinner("Creating transcription render..."):
        render = client.renders.audio_to_text(audio_url=audio_url, notify_url=notify_url)

    if fmt == "json":
        output(render, "json")
    else:
        console.print(f"[bold green]✓[/bold green] Transcription render created: [cyan]{render.render_id}[/cyan]")
        console.print(f"  State: {render.state}")
        if render.result_url:
            console.print(f"  Result: {render.result_url}")
