/*
    A painting canvas with double buffer. Windows.

    Sergei L. Kosakovsky Pond, June 2000-June 2002.
*/

#include "HYGraphicPane.h"
#include "HYComponent.h"
//#include "HYPulldown.h"
#include "HYUtils.h"

#include "math.h"

HBRUSH    bgBrush = CreateSolidBrush (GetSysColor (COLOR_WINDOW));
#define   SQR(A) (A)*(A)

//__________________________________________________________________

RECT    HYRect2Rect (_HYRect& hr)
{
    RECT r;
    r.left = hr.left;
    r.right = hr.right;
    r.top = hr.top;
    r.bottom = hr.bottom;
    return r;
}

//__________________________________________________________________

COLORREF  HYColor2ColorRef (_HYColor&hc, HDC thePane)
{
    return GetNearestColor(thePane,RGB(hc.R,hc.G,hc.B));
}
//__________________________________________________________________

_HYPlatformGraphicPane::_HYPlatformGraphicPane(int h, int w, int d)
{
    thePane = CreateCompatibleDC(0);
    if (thePane) {
        if (d<=1) {
            paneBitMap = CreateCompatibleBitmap(thePane,w,h);
        } else {
            paneBitMap = CreateCompatibleBitmap(GetDC(NULL),w,h);
        }

        if (paneBitMap) {
            oldMap = (HBITMAP)SelectObject (thePane,paneBitMap);
        }
    }
    if ((!thePane)||(!paneBitMap)) {
        _String errMsg = _String("Windows Error ") & (long)GetLastError() & (" while trying to allocate memory for GraphicPane");
        FlagError (errMsg);
    }
    fillColor = NULL;
    backColor = NULL;
    oldFont   = NULL;
    oldBrush  = NULL;
    oldPen    = NULL;

    SetBkMode (thePane, TRANSPARENT);
}

//__________________________________________________________________

_HYPlatformGraphicPane::~_HYPlatformGraphicPane(void)
{
    if (oldMap) {
        DeleteObject (SelectObject (thePane,oldMap));
    }
    if (oldFont) {
        DeleteObject (SelectObject (thePane,oldFont));
    }
    if (oldPen) {
        DeleteObject (SelectObject (thePane,oldPen));
    }
    if (oldBrush) {
        DeleteObject (SelectObject (thePane,oldBrush));
    }
    DeleteDC         (thePane);
    if (fillColor) {
        DeleteObject (fillColor);
    }
    if (fillColor) {
        DeleteObject (backColor);
    }
    //DeleteObject   (paneBitMap);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetPaneSize  (int h,int w, int d)
{
    SelectObject (thePane,oldBrush);
    //DeleteDC (thePane);
    DeleteObject (paneBitMap);
    if (fillColor) {
        DeleteObject (fillColor);
    }

    //thePane = CreateCompatibleDC(NULL);
    if (thePane) {
        if (d<=1) {
            paneBitMap = CreateCompatibleBitmap(thePane,w,h);
        } else {
            paneBitMap = CreateCompatibleBitmap(GetDC(NULL),w,h);
        }

        /*BITMAPINFOHEADER bih;
        bih.biSize   = sizeof (BITMAPINFOHEADER);
        bih.biWidth  = w;
        bih.biHeight = h;
        bih.biPlanes = 1;
        bih.biBitCount = d;
        bih.biCompression = BI_RGB;
        bih.biSizeImage   = 0;
        bih.biXPelsPerMeter = bih.biYPelsPerMeter = 72 * 100/2.54;
        bih.biClrUsed = 0;
        bih.biClrImportant = 0;

        paneBitMap = CreateDIBitmap (thePane, & bih, 0, nil, nil, DIB_RGB_COLORS);*/

        if (paneBitMap) {
            DeleteObject ((HBITMAP)SelectObject (thePane,paneBitMap));
            fillColor = NULL;
        }
    }

    //printf ("Resize bitmap\n");

    if ((!thePane)||(!paneBitMap)) {
        _String errMsg = _String("Windows Error ") & (long)GetLastError() & (" while trying to resize GraphicPane");

        if (!thePane) {
            errMsg = errMsg & ". Failed to create DC.";
        } else {
            errMsg = errMsg & ". Failed to create bitmap.";
        }

        FlagError (errMsg);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::CheckPen (_HYRect& r, bool force)
{
    _HYGraphicPane * theParent = (_HYGraphicPane*)this;
    if (force||(r.width!=lastPenSize)) {
        lastPenSize = r.width;
        HPEN        newPen = (HPEN)CreatePen (PS_SOLID,lastPenSize,HYColor2ColorRef (theParent->color,thePane));
        if (oldPen) {
            DeleteObject ((HPEN)SelectObject(thePane,newPen));
        } else {
            oldPen = (HPEN)SelectObject(thePane,newPen);
        }
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawLine (_HYRect lineDesc)
{
    CheckPen (lineDesc);
    MoveToEx  (thePane, lineDesc.left, lineDesc.top, NULL);
    LineTo  (thePane,lineDesc.right,lineDesc.bottom);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawHatchedLine (_HYRect lineDesc)
{
    _HYGraphicPane * theParent = (_HYGraphicPane*)this;
    HPEN    newPen = (HPEN)CreatePen (PS_DOT,1,HYColor2ColorRef (theParent->color,thePane));
    HPEN    savePen = ((HPEN)SelectObject(thePane,newPen));
    MoveToEx  (thePane, lineDesc.left, lineDesc.top, NULL);
    LineTo  (thePane,lineDesc.right,lineDesc.bottom);
    DeleteObject (SelectObject (thePane,savePen));
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_DisplayText    (_String theText,int t, int l, bool dir)
{
    SetTextAlign (thePane,TA_BASELINE);
    TextOut(thePane,l,t,theText.sData,theText.sLength);
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_DisplayText    (_String& theText,_HYRect& r, char align)
{
    RECT    tb = HYRect2Rect (r);

    tb.top  += ((_HYGraphicPane*)this)->font.size;
    tb.left += ((_HYGraphicPane*)this)->font.size/3;

    _EraseRect (r);

    DrawText (thePane,
              theText.sData,
              theText.sLength,
              &tb,
              DT_NOCLIP|DT_WORDBREAK|DT_END_ELLIPSIS|
              ((align==HY_ALIGN_LEFT)?DT_LEFT:
               (align==HY_ALIGN_RIGHT?DT_RIGHT:DT_CENTER)));
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_DisplayChar  (char c,int t, int l)
{
    SetTextAlign (thePane,TA_BASELINE);
    TextOut(thePane,l,t,&c,1);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawRect (_HYRect rectDesc)
{
    CheckPen (rectDesc);

    rectDesc.right-=rectDesc.width;
    rectDesc.bottom-=rectDesc.width;
    MoveToEx (thePane,rectDesc.left,rectDesc.top,NULL);
    LineTo(thePane,rectDesc.right,rectDesc.top);
    LineTo(thePane,rectDesc.right,rectDesc.bottom);
    LineTo(thePane,rectDesc.left,rectDesc.bottom);
    LineTo(thePane,rectDesc.left,rectDesc.top);
}
//__________________________________________________________________

void _HYPlatformGraphicPane::_FillRect (_HYRect rectDesc)
{
    RECT    r= HYRect2Rect (rectDesc);
    FillRect (thePane,&r,fillColor);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_EraseRect (_HYRect rectDesc)
{
    RECT    r= HYRect2Rect (rectDesc);
    FillRect (thePane,&r,backColor);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawOval (_HYRect rectDesc)
{
    CheckPen (rectDesc);
    SelectObject (thePane,backColor);
    Ellipse (thePane,rectDesc.left,rectDesc.top,rectDesc.right,rectDesc.bottom);
    SelectObject (thePane,fillColor);
}
//__________________________________________________________________

void _HYPlatformGraphicPane::_FillOval (_HYRect rectDesc)
{
    CheckPen (rectDesc);
    Ellipse (thePane,rectDesc.left,rectDesc.top,rectDesc.right,rectDesc.bottom);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_EraseOval (_HYRect rectDesc)
{
    // TBI
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawArc (_HYRect rectDesc, int s, int f)
{
    CheckPen (rectDesc);
    _Parameter  startA = s*3.1415926/180, endA = (s+f)*3.1415926/180,
                diag = .5*sqrt(SQR(rectDesc.right-rectDesc.left)+
                               SQR(rectDesc.bottom-rectDesc.top));
    int         sX =   (rectDesc.right+rectDesc.left)/2+diag*sin(startA),
                sY =   (rectDesc.bottom+rectDesc.top)/2-diag*cos(startA),
                fX =   (rectDesc.right+rectDesc.left)/2+diag*sin(endA),
                fY =   (rectDesc.bottom+rectDesc.top)/2-diag*cos(endA);

    if (f<0) {
        SetArcDirection (thePane,AD_COUNTERCLOCKWISE);
        Arc (thePane,rectDesc.left,rectDesc.top,rectDesc.right,rectDesc.bottom,
             sX,sY,fX,fY);
    } else {
        SetArcDirection (thePane,AD_CLOCKWISE);
        Arc (thePane,rectDesc.left,rectDesc.top,rectDesc.right,rectDesc.bottom,
             fX,fY,sX,sY);
    }

}
//__________________________________________________________________

void _HYPlatformGraphicPane::_FillArc (_HYRect rectDesc, int s, int f)
{
    CheckPen (rectDesc);

    _Parameter  startA  = (90-s)*3.1415926/180,
                endA    = (90-s-f)*3.1415926/180,
                diag    = .5*sqrt(SQR(rectDesc.right-rectDesc.left)+
                                  SQR(rectDesc.bottom-rectDesc.top));

    int         sX =   (rectDesc.right+rectDesc.left)/2+diag*sin(startA),
                sY =   (rectDesc.bottom+rectDesc.top)/2+diag*cos(startA),
                fX =   (rectDesc.right+rectDesc.left)/2+diag*sin(endA),
                fY =   (rectDesc.bottom+rectDesc.top)/2+diag*cos(endA);

    Pie (thePane,rectDesc.left,rectDesc.top,rectDesc.right,rectDesc.bottom,
         sX,sY,fX,fY);


}

//__________________________________________________________________

void _HYPlatformGraphicPane::_EraseArc (_HYRect rectDesc, int s, int f)
{
    // TBI
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetColor  (_HYColor c)
{
    COLORREF theC = HYColor2ColorRef(c,thePane);
    HBRUSH newBrush = (HBRUSH)CreateSolidBrush (theC);

    if (oldBrush) {
        DeleteObject (SelectObject (thePane,newBrush));
    } else {
        oldBrush = (HBRUSH)SelectObject (thePane,newBrush);
    }

    if (fillColor) {
        DeleteObject (fillColor);
    }
    fillColor = newBrush;
    _HYRect pR = {0,0,0,0,0};
    pR.width = lastPenSize;
    CheckPen (pR,true);
    SetTextColor (thePane,theC);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetBColor  (_HYColor c)
{
    COLORREF theC = HYColor2ColorRef(c,thePane);
    HBRUSH newBrush = (HBRUSH)CreateSolidBrush (theC);
    if (backColor) {
        DeleteObject (fillColor);
    }
    backColor = newBrush;
    SetBkColor (thePane, theC);
}


//__________________________________________________________________

void _HYPlatformGraphicPane::_SetFont (_HYFont f)
{
    HFONT   newFont = CreateFont (f.size,0,0,0,(f.style&HY_FONT_BOLD)?FW_BOLD:FW_NORMAL,f.style&HY_FONT_ITALIC,FALSE,FALSE,ANSI_CHARSET,OUT_DEFAULT_PRECIS,
                                  CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,DEFAULT_PITCH|FF_DONTCARE,f.face.sData);
    if (oldFont) {
        DeleteObject (SelectObject (thePane,newFont));
    } else {
        oldFont = (HFONT)SelectObject (thePane, newFont);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetFontSize (long s)
{
    _SetFont (((_HYGraphicPane*)this)->font);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_StartDraw  (void)
{
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_EndDraw    (void)
{
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetPort    (Ptr nP)
{
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_SlidePane  (int dv, int dh)
{
    _HYGraphicPane* theParent = (_HYGraphicPane*)this;
    RECT r = {0,0,theParent->h,theParent->w};

    ScrollDC (thePane, dh, dv, &r, &r, NULL, NULL);
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_SlideRect (_HYRect& rct, int dv, int dh)
{
    _HYGraphicPane* theParent = (_HYGraphicPane*)this;

    RECT r = HYRect2Rect (rct);
    ScrollDC (thePane, dh, dv, &r, &r, NULL, NULL);
}

//__________________________________________________________________
void _HYPlatformGraphicPane::_InvertRect (_HYRect& rct)
{
    RECT r = HYRect2Rect (rct);
    InvertRect (thePane,&r);
}


//__________________________________________________________________
void _HYPlatformGraphicPane::_SavePicture   (_String prompt)
{

    _String         initFileName ("untitled.bmp"),
                    bmp          ("Bitmap\0*.bmp\0\0"),
                    fileName;


    if (SaveFileFunction (fileName, prompt, initFileName, bmp.sData, nil) == 0) {

        if (!fileName.endswith (".bmp")) {
            fileName = fileName & ".bmp";
        }

        BITMAP          bmp;
        PBITMAPINFO     pbmi;
        WORD            cClrBits;

        // Retrieve the bitmap color format, width, and height.
        if (GetObject(paneBitMap, sizeof(BITMAP), (LPSTR)&bmp)) {
            cClrBits = (WORD)(bmp.bmPlanes * bmp.bmBitsPixel);
            if (cClrBits == 1) {
                cClrBits = 1;
            } else if (cClrBits <= 4) {
                cClrBits = 4;
            } else if (cClrBits <= 8) {
                cClrBits = 8;
            } else if (cClrBits <= 16) {
                cClrBits = 16;
            } else if (cClrBits <= 24) {
                cClrBits = 24;
            } else {
                cClrBits = 32;
            }

            // Allocate memory for the BITMAPINFO structure. (This structure
            // contains a BITMAPINFOHEADER structure and an array of RGBQUAD
            // data structures.)

            if (cClrBits != 24)
                pbmi = (PBITMAPINFO) LocalAlloc(LPTR,
                                                sizeof(BITMAPINFOHEADER) +
                                                sizeof(RGBQUAD) * (1<< cClrBits));

            // There is no RGBQUAD array for the 24-bit-per-pixel format.

            else
                pbmi = (PBITMAPINFO) LocalAlloc(LPTR,
                                                sizeof(BITMAPINFOHEADER));

            checkPointer (pbmi);

            // Initialize the fields in the BITMAPINFO structure.

            pbmi->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
            pbmi->bmiHeader.biWidth = bmp.bmWidth;
            pbmi->bmiHeader.biHeight = bmp.bmHeight;
            pbmi->bmiHeader.biPlanes = bmp.bmPlanes;
            pbmi->bmiHeader.biBitCount = bmp.bmBitsPixel;
            if (cClrBits < 24) {
                pbmi->bmiHeader.biClrUsed = (1<<cClrBits);
            }

            // If the bitmap is not compressed, set the BI_RGB flag.
            pbmi->bmiHeader.biCompression = BI_RGB;

            // Compute the number of bytes in the array of color
            // indices and store the result in biSizeImage.
            // For Windows NT, the width must be DWORD aligned unless
            // the bitmap is RLE compressed. This example shows this.
            // For Windows 95/98/Me, the width must be WORD aligned unless the
            // bitmap is RLE compressed.
            pbmi->bmiHeader.biSizeImage = ((pbmi->bmiHeader.biWidth * cClrBits +31) & ~31) /8
                                          * pbmi->bmiHeader.biHeight;
            // Set biClrImportant to 0, indicating that all of the
            // device colors are important.
            pbmi->bmiHeader.biClrImportant = 0;

            HANDLE            hf;       // file handle
            BITMAPFILEHEADER  hdr;      // bitmap file-header
            PBITMAPINFOHEADER pbih;     // bitmap info-header
            LPBYTE            lpBits;   // memory pointer
            DWORD             dwTotal;  // total count of bytes
            DWORD             cb;       // incremental count of bytes
            BYTE              *hp;      // byte pointer
            DWORD             dwTmp;

            pbih   = (PBITMAPINFOHEADER) pbmi;
            lpBits = (LPBYTE) GlobalAlloc(GMEM_FIXED, pbih->biSizeImage);


            // Retrieve the color table (RGBQUAD array) and the bits
            // (array of palette indices) from the DIB.
            if (!GetDIBits(thePane, paneBitMap, 0, (WORD) pbih->biHeight, lpBits, pbmi,
                           DIB_RGB_COLORS)) {
                LocalFree (pbmi);
                return;
            }

            // Create the .BMP file.
            hf = CreateFile(fileName.sData,
                            GENERIC_READ | GENERIC_WRITE,
                            (DWORD) 0,
                            NULL,
                            CREATE_ALWAYS,
                            FILE_ATTRIBUTE_NORMAL,
                            (HANDLE) NULL);

            if (hf == INVALID_HANDLE_VALUE) {
                LocalFree (pbmi);
                return;
            }

            hdr.bfType = 0x4d42;        // 0x42 = "B" 0x4d = "M"
            // Compute the size of the entire file.
            hdr.bfSize = (DWORD) (sizeof(BITMAPFILEHEADER) +
                                  pbih->biSize + pbih->biClrUsed
                                  * sizeof(RGBQUAD) + pbih->biSizeImage);
            hdr.bfReserved1 = 0;
            hdr.bfReserved2 = 0;

            // Compute the offset to the array of color indices.
            hdr.bfOffBits = (DWORD) sizeof(BITMAPFILEHEADER) +
                            pbih->biSize + pbih->biClrUsed
                            * sizeof (RGBQUAD);

            // Copy the BITMAPFILEHEADER into the .BMP file.
            if (!WriteFile(hf, (LPVOID) &hdr, sizeof(BITMAPFILEHEADER),
                           (LPDWORD) &dwTmp,  NULL)) {
                LocalFree (pbmi);
                return;
            }

            // Copy the BITMAPINFOHEADER and RGBQUAD array into the file.
            if (!WriteFile(hf, (LPVOID) pbih, sizeof(BITMAPINFOHEADER)
                           + pbih->biClrUsed * sizeof (RGBQUAD),
                           (LPDWORD) &dwTmp, ( NULL))) {
                LocalFree (pbmi);
                return;
            }

            // Copy the array of color indices into the .BMP file.
            dwTotal = cb = pbih->biSizeImage;
            hp = lpBits;
            if (!WriteFile(hf, (LPSTR) hp, (int) cb, (LPDWORD) &dwTmp,NULL)) {
                LocalFree (pbmi);
                return;
            }

            // Close the .BMP file.
            if (!CloseHandle(hf)) {
                LocalFree (pbmi);
                return;
            }

            // Free memory.
            GlobalFree((HGLOBAL)lpBits);
            LocalFree (pbmi);
        }
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawPicRes (_HYRect& r, long id)
// id here refers to a bitmap resource
{
    HBITMAP aPic = LoadBitmap (ProgramInstance,MAKEINTRESOURCE(id));
    if (aPic) {
        /*BITMAP theBM;
        GetObject (aPic, sizeof (BITMAP), &theBM);
        if (otherDC)
        {
            if (r.right-r.left<=0)
                r.right = r.right = r.left + theBM.bmWidth;

            if (r.bottom-r.top<=0)
                r.bottom = r.bottom = r.top + theBM.bmHeight;

            SelectObject (otherDC, aPic);

            TransparentBlt   (thePane, r.left, r.top, r.right-r.left+1, r.bottom-r.top+1,
                          otherDC, 0, 0, theBM.bmWidth, theBM.bmHeight, RGB(255,255,255));

            SelectObject (otherDC, oDCBM);
        }
        else
        {
            _String errMsg = _String ("Failed to make CompatibleDC in _DrawPicRes");
            ReportWarning (errMsg);
        }*/

        POINT   bSize = {0,0};

        if (r.right-r.left>0) {
            bSize.x = r.right-r.left+1;
        }

        if (r.bottom-r.top>0) {
            bSize.y = r.bottom-r.top + 1;
        }

        bSize = DrawTransparentBitmap (thePane,aPic,r.left,r.right,bSize.x,bSize.y,RGB(255,255,255));

        if (r.right-r.left<=0) {
            r.right = r.left + bSize.x;
        }

        if (r.bottom-r.top<=0) {
            r.bottom = r.top + bSize.y;
        }

        DeleteObject (aPic);
    } else {
        _String errMsg = _String ("No bitmap resource with ID ") & id;
        ReportWarning (errMsg);
    }
}

//__________________________________________________________________

Ptr _HYPlatformGraphicPane::_DefinePolygon  (_SimpleList& points)
{
    if ((points.lLength>=6)&&(points.lLength%2==0)) {
        POINT* pts = new POINT [points.lLength/2];
        checkPointer (pts);

        for (long k=0; k<points.lLength; k+=2) {
            pts[k/2].x = points.lData[k];
            pts[k/2].y = points.lData[k+1];
        }

        HRGN    rgn = CreatePolygonRgn (pts,points.lLength/2,WINDING);
        checkPointer (rgn);
        delete (pts);
        return (Ptr)rgn;
    }
    return nil;
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_KillPolygon   (Ptr rgn)
{
    if (rgn) {
        DeleteObject (rgn);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_DrawPolygon (Ptr rgn, long width)
{
    if (rgn) {
        FrameRgn (thePane, (HRGN)rgn, fillColor, width, width);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_FillPolygon (Ptr rgn)
{
    if (rgn) {
        FillRgn (thePane, (HRGN)rgn, fillColor);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_ErasePolygon (Ptr rgn)
{
    if (rgn) {
        FillRgn (thePane, (HRGN)rgn, backColor);
    }
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_CopyToClipboard (void)
{
    PlaceBitmapInClipboard (paneBitMap,NULL);
}

//__________________________________________________________________

void _HYPlatformGraphicPane::_SetDialogBG (void)
{
    ((_HYGraphicPane*)this)->SetBColor         (GetDialogBackgroundColor());
}

//___________________________________________________________________

char        isWindows2000orXP = -1;

//__________________________________________________________________
POINT   DrawTransparentBitmap(HDC hdc, HBITMAP hBitmap, short xStart,short yStart,short xSize, short ySize, COLORREF cTransparentColor)
{
    POINT      ptSize;
    if (isWindows2000orXP == -1) {
        OSVERSIONINFO osinfo;
        osinfo.dwOSVersionInfoSize = sizeof (osinfo);
        GetVersionEx (&osinfo);
        isWindows2000orXP = (osinfo.dwMajorVersion > 4);
    }
    if (isWindows2000orXP) {
        BITMAP     bm;;
        GetObject (hBitmap, sizeof (BITMAP), &bm);
        ptSize.x = bm.bmWidth;            // Get width of bitmap
        ptSize.y = bm.bmHeight;           // Get height of bitmap
        HDC        hdcBM   = CreateCompatibleDC(hdc);
        if (hdcBM) {
            if (xSize<=0) {
                xSize = ptSize.x;
            }
            if (ySize<=0) {
                ySize = ptSize.y;
            }

            HBITMAP hdcBMOLD = (HBITMAP)SelectObject (hdcBM, hBitmap);

            TransparentBlt   (hdc, xStart, yStart, xSize, ySize,
                              hdcBM, 0, 0,bm.bmWidth, bm.bmHeight, RGB(255,255,255));

            SelectObject (hdcBM, hdcBMOLD);
            DeleteDC (hdcBM);
        } else {
            _String errMsg = _String ("Failed to make CompatibleDC in _DrawPicRes");
            ReportWarning (errMsg);
        }
    } else {
        BITMAP     bm;
        COLORREF   cColor;
        HBITMAP    bmAndBack, bmAndObject, bmAndMem, bmSave;
        HBITMAP    bmBackOld, bmObjectOld, bmMemOld, bmSaveOld;
        HDC        hdcMem, hdcBack, hdcObject, hdcTemp, hdcSave;

        hdcTemp = CreateCompatibleDC(hdc);
        SelectObject(hdcTemp, hBitmap);   // Select the bitmap

        GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
        ptSize.x = bm.bmWidth;            // Get width of bitmap
        ptSize.y = bm.bmHeight;           // Get height of bitmap
        DPtoLP(hdcTemp, &ptSize, 1);      // Convert from device

        // to logical points

        if (xSize<=0) {
            xSize = ptSize.x;
        }
        if (ySize<=0) {
            ySize = ptSize.y;
        }


        // Create some DCs to hold temporary data.
        hdcBack   = CreateCompatibleDC(hdc);
        hdcObject = CreateCompatibleDC(hdc);
        hdcMem    = CreateCompatibleDC(hdc);
        hdcSave   = CreateCompatibleDC(hdc);

        // Create a bitmap for each DC. DCs are required for a number of
        // GDI functions.

        // Monochrome DC
        bmAndBack   = CreateBitmap(ptSize.x, ptSize.y, 1, 1, NULL);

        // Monochrome DC
        bmAndObject = CreateBitmap(ptSize.x, ptSize.y, 1, 1, NULL);

        bmAndMem    = CreateCompatibleBitmap(hdc, ptSize.x, ptSize.y);
        bmSave      = CreateCompatibleBitmap(hdc, ptSize.x, ptSize.y);

        // Each DC must select a bitmap object to store pixel data.
        bmBackOld   = (HBITMAP)SelectObject(hdcBack, bmAndBack);
        bmObjectOld = (HBITMAP)SelectObject(hdcObject, bmAndObject);
        bmMemOld    = (HBITMAP)SelectObject(hdcMem, bmAndMem);
        bmSaveOld   = (HBITMAP)SelectObject(hdcSave, bmSave);

        // Set proper mapping mode.
        SetMapMode(hdcTemp, GetMapMode(hdc));

        // Save the bitmap sent here, because it will be overwritten.
        BitBlt(hdcSave, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0, SRCCOPY);

        // Set the background color of the source DC to the color.
        // contained in the parts of the bitmap that should be transparent
        cColor = SetBkColor(hdcTemp, cTransparentColor);

        // Create the object mask for the bitmap by performing a BitBlt
        // from the source bitmap to a monochrome bitmap.
        BitBlt(hdcObject, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0,
               SRCCOPY);

        // Set the background color of the source DC back to the original
        // color.
        SetBkColor(hdcTemp, cColor);

        // Create the inverse of the object mask.
        BitBlt(hdcBack, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0,
               NOTSRCCOPY);

        // Copy the background of the main DC to the destination.
        BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdc, xStart, yStart,
               SRCCOPY);

        // Mask out the places where the bitmap will be placed.
        BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0, SRCAND);

        // Mask out the transparent colored pixels on the bitmap.
        BitBlt(hdcTemp, 0, 0, ptSize.x, ptSize.y, hdcBack, 0, 0, SRCAND);

        // XOR the bitmap with the background on the destination DC.
        BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0, SRCPAINT);

        // Copy the destination to the screen.
        StretchBlt(hdc, xStart, yStart, xSize, ySize, hdcMem, 0, 0, ptSize.x, ptSize.y,
                   SRCCOPY);

        // Place the original bitmap back into the bitmap sent here.
        BitBlt(hdcTemp, 0, 0, ptSize.x, ptSize.y, hdcSave, 0, 0, SRCCOPY);

        // Delete the memory bitmaps.
        DeleteObject(SelectObject(hdcBack, bmBackOld));
        DeleteObject(SelectObject(hdcObject, bmObjectOld));
        DeleteObject(SelectObject(hdcMem, bmMemOld));
        DeleteObject(SelectObject(hdcSave, bmSaveOld));

        // Delete the memory DCs.
        DeleteDC(hdcMem);
        DeleteDC(hdcBack);
        DeleteDC(hdcObject);
        DeleteDC(hdcSave);
        DeleteDC(hdcTemp);
    }
    return ptSize;
}

//EOF