import logging
import pathlib
import typing

import docker
import docker.errors

from ideas import constants, exceptions
from ideas.commands import validate_tenant
from ideas.session import Session, get_default_session
from ideas.utils import display, http
from ideas.utils.docker_utils import get_docker_client

logger = logging.getLogger()


def get_containers(filters, session: typing.Optional[Session] = None):
    """Get containers published in IDEAS"""
    session = session or get_default_session()

    yield from http.handle_pagination(
        http.get,
        f"{session.base_url}/api/v0/tes/image/",
        session.headers,
        session.auth,
        filters,
    )


def get_repo_digest(name: str, session: Session):
    """
    Get the repo digest of a local image.
    A repo digest is a unique hash of an image that has been published to a docker registry.
    If the local image has been published to IDEAS registry, the repo digest for that registry will be returned.
    Otherwise, if the local image has never been published, or published to a different registry, None will be returned.
    """
    client = get_docker_client()
    image = client.images.get(name)
    repo_digests = image.attrs.get("RepoDigests", [])
    matched_repo_digest = None
    tenant = validate_tenant(session.tenant_id, session)
    assert tenant
    tenant_key = tenant["key"]
    for repo_digest in repo_digests:
        repository_url = f"{session.registry_url}/{tenant_key}"
        if repo_digest.startswith(repository_url):
            matched_repo_digest = repo_digest.rsplit("@sha256:", maxsplit=1)[-1]
            break
    logger.info(f"Matched repo digest: {matched_repo_digest}")
    return matched_repo_digest


def get_container_from_repo_digest(
    repo_digest: str, tenant_id: int, session: typing.Optional[Session] = None
):
    """Get IDEAS container info based on repo digest of local docker image"""
    result = list(
        get_containers(
            filters=(
                ("checksum", repo_digest),
                ("tenant", tenant_id),
            ),
            session=session,
        )
    )
    if len(result) == 1:
        return result[0]
    return None


def split_full_name(full_name: str):
    repository = full_name.split(":", maxsplit=1)
    if len(repository) == 1:
        repository, tag = repository[0], "latest"
    else:
        repository, tag = repository

    return repository, tag


def publish_container(
    full_name: str,
    description: str,
    tenant_id: int,
    label: typing.Optional[str] = None,
    namespace: typing.Optional[str] = None,
    tags: typing.Tuple[str] = (),
    readme: typing.Optional[pathlib.Path] = None,
    session: typing.Optional[Session] = None,
):
    logger.debug(f"Publishing container: {full_name}")

    session = session or get_default_session()
    tenant_id = tenant_id or session.tenant_id
    validate_tenant(tenant_id=tenant_id, session=session)

    try:
        client = get_docker_client()
        image = client.images.get(full_name)
    except docker.errors.ImageNotFound:
        raise exceptions.ContainerNotFoundException(
            f"Failed to find local container {full_name}"
        )
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            f"Failed to find local container {full_name} due to unexpected error with docker."
        )

    _, image_id = image.attrs["Id"].split(":", maxsplit=1)
    size = image.attrs["Size"]
    arch = image.attrs["Architecture"]

    if arch not in constants.SUPPORTED_ARCHITECTURES:
        raise exceptions.ContainerUnsupportedArchException(
            f"Container arch {arch} not supported in IDEAS", arch=arch
        )

    # examine image_id and determine if it exists in the be
    logger.debug("Checking existence of container image_id in IDEAS container registry")

    # if exists, logger.debug warning and exit
    repo_digest = get_repo_digest(name=full_name, session=session)
    if repo_digest:
        logger.debug(
            "Found repo digest for IDEAS registry, checking if checksum exists in IDEAS"
        )
        ideas_container = get_container_from_repo_digest(
            repo_digest=repo_digest, tenant_id=tenant_id, session=session
        )

        if ideas_container:
            logger.debug("Container exists in IDEAS registry, skipping publishing")
            raise exceptions.ContainerPublishExistsException(
                f"Local container {full_name} with checksum {repo_digest} has already been published to IDEAS container registry.",
                checksum=repo_digest,
            )

    # else run be command and return name of image
    logger.debug(
        "Container doesn't exist in IDEAS registry, proceeding with publish..."
    )

    # full name should be parsable since docker image is retrieved by full name earlier
    name, _ = split_full_name(full_name=full_name)

    # If user-friendly label provided, use that as the tag in IDEAS
    # otherwise default to the first 12 characters of the image id
    if label:
        tag = label
    else:
        tag = image_id[:12]

    # If user-friendly namespace provided, use that as the namespace in IDEAS
    # otherwise default to namespace of local image
    if namespace:
        name = namespace

    data = {
        "name": name,
        "label": tag,
        "description": description,
        "tags": tags,
        "size": size,
        "tenant": tenant_id,
    }

    if readme:
        try:
            with open(readme, "r") as f:
                data["readme"] = f.read()
        except FileNotFoundError:
            raise exceptions.ContainerException(
                f"Readme file for container not found: {readme}"
            )

    try:
        response = http.post(
            f"{session.base_url}/api/v0/tes/image/", session.headers, session.auth, data
        )
    except exceptions.ApiException:
        raise exceptions.ContainerPublishPostException()

    logger.debug(
        f"Pushing container to IDEAS container registry with name: {response['full_name']}"
    )

    # get repository name from BE response
    repository = response["repository"]
    token = session.get_access_token()
    registry_repository_name = f"{session.registry_url}/{repository}"
    try:
        with display.console.status("Publishing image to IDEAS container registry..."):
            client.login(username=tag, password=token, registry=session.registry_url)
            image = client.images.get(full_name)
            image.tag(registry_repository_name, tag=tag)

            # TODO: figure out how to stream progress here
            # when passing stream=True, the return value is a blocking generator
            # which can cause issues with the push operation where it freezes unexpectedly
            client.images.push(registry_repository_name, tag=tag, stream=False)
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            "Failed to push local container to IDEAS container registry due to unexpected error with docker."
        )

    # patch the pushed container image checksum with the repo digest
    # the repo digest is not saved to the docker image metadata until after the the image is fully pushed to registtry
    try:
        logger.debug("Patching published image with repo digest for checksum")
        repo_digest = get_repo_digest(name=full_name, session=session)
        http.patch(
            f"{session.base_url}/api/v0/tes/image/{response['id']}/",
            headers=session.headers,
            auth=session.auth,
            data={"checksum": repo_digest},
        )
    except exceptions.ApiException:
        raise exceptions.ContainerPublishPostException(
            "Failed to update container image checksum with repo digest after push"
        )

    return response


def download_container(full_name: str, session: typing.Optional[Session] = None):
    session = session or get_default_session()
    logger.debug(f"Downloading image {full_name} from IDEAS registry")
    repository, tag = split_full_name(full_name=full_name)

    result = list(
        get_containers(
            filters=(("repository", repository), ("label", tag)),
        )
    )

    if len(result) != 1:
        raise exceptions.ContainerNotFoundException(
            f"Failed to find container with name {full_name} in IDEAS container registry"
        )

    result = result[0]
    registry_repository_name = f"{session.registry_url}/{repository}"
    token = session.get_access_token()

    try:
        with display.console.status(
            "Downloading image from IDEAS container registry..."
        ):
            client = get_docker_client()
            client.login(username=tag, password=token, registry=session.registry_url)
            client.images.pull(
                registry_repository_name,
                tag=tag,
                platform=constants.CONTAINER_ARCHITECTURE,
            )

            image = client.images.get(f"{registry_repository_name}:{tag}")
            image.tag(repository, tag=tag)
    except docker.errors.ImageNotFound:
        raise exceptions.ContainerDownloadAccessException(
            f"Failed to find {full_name} in IDEAS container registry. Ensure you have download access to the container."
        )
    except docker.errors.DockerException:
        raise exceptions.DockerExecutionError(
            "Failed to download containe from IDEAS container registry due to unexpected error with docker."
        )
