import functools
import logging
import typing

import requests

from ideas.utils.display import EXIT_STATUS, abort

logger = logging.getLogger(__name__)

DEFAULT_ENVIRONMENTS_URL = "https://environment.inscopix.cloud/"


class Environment(typing.TypedDict):
    USER_POOL_ID: str
    CLIENT_ID: str
    REGION: str
    DJANGO_URL: str
    REGISTRY_URL: str
    production: bool


@functools.cache
def _get_environments(environments_url: str) -> dict[str, Environment]:
    """
    Retrieve and cache the environments from our CDN in memory.
    """
    try:
        return requests.get(environments_url).json()["environment"]
    except Exception:
        # Can't re-raise here because we are called when instantiating click
        # options for the --env flag
        abort(log="Failed to get environments", exit_code=EXIT_STATUS.UNHANDLED_ERROR)


def get_environments(
    environments_url: str | None = DEFAULT_ENVIRONMENTS_URL, include_dev=False
) -> dict[str, Environment]:
    """
    Retrieve a list of supported environments. Cached during the lifetime of a
    CLI command.
    """
    if environments_url is None:
        environments_url = DEFAULT_ENVIRONMENTS_URL

    environments = _get_environments(environments_url)

    if not include_dev:
        return {key: env for (key, env) in environments.items() if env["production"]}

    return environments
