class IdeasError(Exception):
    """
    Base class for all ideas exceptions.
    """


class UnhandledError(IdeasError):
    """
    Raise this (hopefully infrequently) if you have no idea how to process an error.
    """


class UnauthorizedError(IdeasError):
    """
    Raised if the Cognito authorization fails, or the API returns a 401/403 response.
    """


class InvalidProfileError(IdeasError):
    """
    Raised if the specified configuration profile doesn't exist.
    """


class ApiException(IdeasError):
    """
    Raised for any non-specific API errors; provides the JSON response from the API.
    """

    def __init__(self, message, json):
        super().__init__(message)
        self.json = json


class FileProcessedTimedOut(IdeasError):
    """
    Raised when creating a series, if the specified files have not been successfully processed in
    a reasonable amount of time.
    """


class QuotaExceededError(ApiException):
    """
    Raised if the API returns a 413 Payload Too Large error.
    """


class MismatchedChecksumError(IdeasError):
    """
    Raised if the checksum returned by AWS doesn't match the local checksum on download/upload of a
    file.
    """


class DockerExecutionError(IdeasError):
    """
    Raised if there is an unexpected error with the Docker API.
    """


class DockerUnavailableError(IdeasError):
    """
    Raised if there is an unexpected error with the Docker API.
    """

    @staticmethod
    def get_instructions():
        """Get instructions on how to resolve this error."""
        from ideas.utils import display

        return (
            "Failed to run docker. Please ensure the docker application is running.",
            "\nTo start docker, on Linux:",
            display.bash("sudo systemctl start docker"),
            "\nTo start docker, on  macOS:",
            display.bash("open -a Docker"),
            "\nTo start docker, on  Windows:",
            display.bash(r"'C:\Program Files\Docker\Docker\Docker Desktop.exe'"),
        )


class NvidiaRuntimeMissingError(IdeasError):
    """
    Raised if the user is trying to use gpus for tool execution but docker reports the nvidia
    runtime is missing.
    """

    @staticmethod
    def get_instructions():
        """Get instructions on how to resolve this error."""

        return (
            "Failed to run tool with the nvidia runtime.",
            "\nPlease follow the steps in the link below to install and configure the nvidia runtime for docker",
            "\nhttps://docs.nvidia.com/datacenter/cloud-native/container-toolkit/latest/install-guide.html",
        )


class UserNotTenantMemberError(IdeasError):
    """
    Raised if user is trying to perform an action in a tenant they are not a member of.
    """


# ======================================== Tool Errors ========================================


class ToolException(IdeasError):
    """
    Raised if there is an error with tools.
    """


class ToolConfigFileNotFoundError(ToolException):
    """
    Raised if the user specifies a tool file not found when configuring a tool.
    """


class ToolConfigFunctionNotFoundError(ToolException):
    """
    Raised if the user specifies a tool function not found when configuring a tool.
    """


class ToolConfigContainerNotFoundError(ToolException):
    """
    Raised if the user specifies a tool container image not found when configuring a tool.
    """

    def __init__(self, message=None, container=None):
        super().__init__(message)
        self.container = container


class ToolMisconfiguredError(ToolException):
    """
    Raised if a given tool is not configured correctly.
    """

    def __init__(self, message=None, key=None):
        super().__init__(message)
        self.key = key


class ToolInputsNotFoundError(ToolException):
    """
    Raised if a given tool inputs file is not found.
    """


class ToolInputsFormatError(ToolException):
    """
    Raised if tool inputs are not formatted correctly.
    """

    def __init__(self, message=None, key=None, value=None):
        super().__init__(message)
        self.message = message
        self.key = key
        self.value = value


class ToolInputsFileNotFoundError(ToolException):
    """
    Raised if an input file specified in tool inputs is not found in the user's local filesystem.
    """

    def __init__(self, message, file):
        super().__init__(message)
        self.file = file


class ToolExecutionError(ToolException):
    """
    Raised if there is an error executing the tool code inside the tool container.
    """


class ToolNotFoundError(ToolException):
    """
    Raised if the no tools are found in a user's codebase.
    """


class ToolBundleBuildError(ToolException):
    """
    Raised if there is an error building a code bundle.
    """


class ToolBundleExtractError(ToolException):
    """
    Raised if we are unable to extract the tool bundle file.
    """


class ToolIntrospectionError(ToolException):
    """
    Raised if we are unable to introspect the given tool code file.
    """


class ToolSchemaValidationError(ToolException):
    """
    Raised if we are unable to introspect the given tool code file.
    """


# ======================================== Container Errors ========================================


class ContainerException(IdeasError):
    """
    Raised if there is an error with containers.
    """


class ContainerNotPublishedError(ContainerException):
    """
    Raised if a container is not published to IDEAS container registry when generating code bundle that references the container.
    """

    def __init__(self, message=None, container=None):
        super().__init__(message)
        self.container = container


class ContainerUnsupportedArchException(ContainerException):
    """
    Raised if a container arch is not supported in IDEAS.
    """

    def __init__(self, message=None, arch=None):
        super().__init__(message)
        self.arch = arch


class ContainerNotFoundException(ContainerException):
    """
    Raised if a container is not found in IDEAS container registry.
    """


class ContainerDownloadAccessException(ContainerException):
    """
    Raised if a container could not be downloaded from IDEAS container registry.
    """


class ContainerPublishExistsException(ContainerException):
    """
    Raised if a container is not found in IDEAS container registry.
    """

    def __init__(self, message=None, checksum=None):
        super().__init__(message)
        self.checksum = checksum


class ContainerPublishPostException(ContainerException):
    """
    Raised if there was an error creating the container image in IDEAS.
    """


class ContainerPythonNotFound(ContainerException):
    """
    Raised if `python` is not an available command inside a chosen container
    """


# ======================================== Output Data Errors ========================================


class OutputDataException(Exception):
    """
    Raised if there is an exception related to output data
    """


class OutputDataFileDoesNotExist(OutputDataException):
    """
    Raised if an output file does not exist
    """


class OutputDataFileOutsideOutputDir(OutputDataException):
    """
    Raise if an output file exists outside of the output directory
    """
