import glob
import json
import logging
import os
import shlex
import shutil
import tempfile
import textwrap
import typing
from datetime import datetime

import docker
import docker.errors
from ideas_schemas import schemas
from ideas_schemas.exceptions import SchemaValidationError
from rich.table import Table

from ideas import constants, exceptions
from ideas.containers import split_full_name
from ideas.tools.introspection import params
from ideas.tools.introspection.parsers import ExecutorType, introspect_code
from ideas.utils import display
from ideas.utils.docker_utils import (
    copy_files_from_container_to_host,
    get_docker_client,
)
from ideas.utils.os_utils import is_windows

logger = logging.getLogger(__name__)


def _generate_tool_python_command(
    code_dir_name: str, module_name: str, function_name: str
):
    """Helper function for generating tool command in python"""
    tool_command = f"""
import os
import shutil
import sys
import json

os.makedirs('{constants.CONTAINER_OUTPUT_DIR_PATH}')
os.chdir('{constants.CONTAINER_OUTPUT_DIR_PATH}')
shutil.copy('/tmp/{constants.IDEAS_INPUTS_FILE_NAME}', '{constants.CONTAINER_OUTPUT_DIR_PATH}/{constants.IDEAS_INPUTS_FILE_NAME}')
sys.path.append('/tmp/{code_dir_name}')
from {module_name} import {function_name}

with open('{constants.IDEAS_INPUTS_FILE_NAME}', 'r') as f:
    inputs = json.load(f)

{function_name}(**inputs)
"""
    return tool_command


class ToolConfig:
    """
    The local configuration of a tool.
    This configuration enables local testing of tool execution
    as well as generating code bundles for publishing tools in IDEAS.

    :param key: The key of the tool.
        Configured in the cloud when a tool is created.
    :param code_dir: The directory of the user's local codebase
        containing analysis for tools.
    :param dir: The directory of the ideas configuration files
        for tools in the user's local codebase.
    :param file: The configuration file for this tool in the user's
        local codebase.
    :param code_file: The file in the user's codebase which contains
        the analysis function to execute.
    :param function: The analysis function to execute.
    :param container: The container image to use execute the analysis
        function within.
    :param data: The config data represented as a dict, for serialization
        to a JSON file that is saved to the user's local codebase.
    """

    key: str
    code_dir: str
    dir: str
    file: str
    code_file: typing.Optional[str] = None
    function: typing.Optional[str] = None
    container: typing.Optional[str] = None
    executor_type: typing.Optional[ExecutorType] = None
    ast: typing.Optional[typing.Dict[str, typing.List[params.Parameter]]] = None
    data: typing.Optional[typing.Dict] = None

    def __init__(self, key: str, code_dir: typing.Optional[str] = None):
        self.key = key
        self.code_dir = code_dir if code_dir else os.getcwd()
        self.dir = os.path.join(self.code_dir, constants.IDEAS_DIR_NAME, key)
        self.file = os.path.join(self.dir, constants.IDEAS_CONFIG_FILE_NAME)

    def _generate(self) -> None:
        """
        Generates tool configuration by introspecting user's code.

        :raises ToolConfigFileNotFoundError: If the code file could not
            be found in the user's local codebase.
        :raises ToolIntrospectionError: If the code file could not
            be introspected.
        """
        self.executor_type, self.ast = introspect_code(
            os.path.join(self.code_dir, self.code_file)
        )
        self.data = {
            "schema_version": constants.TOOL_CONFIG_SCHEMA_VERSION,
            "file": self.code_file,
            "container": self.container,
            "executor_type": self.executor_type,
        }
        if self.function:
            self.data["function"] = self.function

    def _validate(self) -> None:
        """
        Validate tool configuration properties.

        :raises ToolMisconfiguredError: If the tool is misconfigured.
        :raises ToolConfigFileNotFoundError: If the code file could not
            be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not
            be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not
            be found in the local docker images.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        # validate file exists
        code_file = os.path.join(self.code_dir, self.code_file)
        if not os.path.exists(code_file):
            raise exceptions.ToolConfigFileNotFoundError(
                f"Failed to find code file: {self.code_file}"
            )

        # validate function exists in file
        if self.function:
            # validate executor type
            if self.executor_type != ExecutorType.PYTHON:
                raise exceptions.ToolMisconfiguredError(
                    f"Function is only supported for python execution, but detected {self.executor_type} execution.",
                    key=self.key,
                )

            if self.function not in self.ast:
                raise exceptions.ToolConfigFunctionNotFoundError(
                    f"Failed to find function in code file: {self.function}"
                )

            self.introspected_parameters = self.ast[self.function]
        else:
            if self.executor_type == ExecutorType.PYTHON:
                raise exceptions.ToolMisconfiguredError(
                    "Function must be specified for python execution.", key=self.key
                )

        # validate container exists
        client = get_docker_client()

        try:
            client.images.get(self.container)
        except docker.errors.ImageNotFound:
            raise exceptions.ToolConfigContainerNotFoundError(
                f"Failed to find container: {self.container}", container=self.container
            )
        except docker.errors.DockerException:
            raise exceptions.DockerExecutionError(
                "Failed to validate docker container image exists due to unexpected error with docker."
            )

    def load(self) -> None:
        """
        Load the existing configuration from the user's local codebase

        :raises ToolMisconfiguredError: If the tool is not configured correctly.
        :raises ToolConfigFileNotFoundError: If the code file could not
            be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not
            be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not
            be found in the local docker images.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        if not os.path.exists(self.file):
            raise exceptions.ToolMisconfiguredError(
                f"Config file not found: {self.file}", key=self.key
            )
        with open(self.file, "r") as f:
            self.data = json.load(f)

        self.code_file = self.data.get("file")
        self.function = self.data.get("function", None)
        self.container = self.data.get("container")
        self.executor_type = ExecutorType(self.data["executor_type"])
        if not self.function and self.executor_type == ExecutorType.PYTHON:
            raise exceptions.ToolMisconfiguredError(
                "Expected PYTHON executor type for config with function", key=self.key
            )

        self.executor_type, self.ast = introspect_code(
            os.path.join(self.code_dir, self.code_file)
        )
        self._validate()

    def save(
        self,
        file: typing.Optional[str] = None,
        function: typing.Optional[str] = None,
        container: typing.Optional[str] = None,
    ) -> None:
        """
        Save the tool configuration to a JSON file in the user's local codebase.

        :param file: The file in the user's codebase which contains
            the analysis function to execute.
        :param function: The analysis function to execute.
        :param container: The container image to use execute the analysis
            function within.

        :raises ToolMisconfiguredError: If the tool is misconfigured.
        :raises ToolIntrospectionError: If the code file could not be introspected.
        :raises ToolConfigFileNotFoundError: If the code file could not be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not be found in the local docker images.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        if file:
            if is_windows():
                file = file.replace("\\", "/")
            self.code_file = file

        if function:
            self.function = function

        if container:
            self.container = container

        if not self.code_file or not self.container:
            raise exceptions.ToolMisconfiguredError(
                "Cannot save tool config with empty code file or container",
                key=self.key,
            )

        self._generate()
        self._validate()

        logger.debug(f"Writing tool config to: {self.file}")
        if not os.path.exists(self.dir):
            os.makedirs(self.dir)
        with open(self.file, "w") as f:
            json.dump(self.data, f, indent=4, sort_keys=True)


class ToolSpec:
    """
    The specification of a tool in IDEAS.
    This specification describes how to execute a tool,
    and how to render it's corresponding analysis table in IDEAS.

    :param tool_config: The local configuration of a tool.
        Used to generate the tool spec.
    :param code_dir: The directory of the user's local codebase
        containing analysis for tools.
    :param dir: The directory of the ideas configuration files
        for tools in the user's local codebase.
    :param file: The spec file for this tool in the user's
        local codebase.
    :param data: The spec data represented as a dict, for serialization
        to a JSON file that is saved to the user's local codebase.
    """

    tool_config: ToolConfig
    code_dir: str
    dir: str
    file: str
    data: typing.Optional[typing.Dict] = None

    def __init__(self, tool_config: ToolConfig, code_dir: typing.Optional[str] = None):
        self.tool_config = tool_config
        self.code_dir = code_dir if code_dir else os.getcwd()
        self.dir = os.path.join(
            self.code_dir, constants.IDEAS_DIR_NAME, self.tool_config.key
        )
        self.file = os.path.join(self.dir, constants.IDEAS_TOOL_SPEC_FILE_NAME)

    def _generate(self) -> None:
        """
        Generates tool spec from the introspected params in the tool config.
        """
        params = []
        for introspected in self.tool_config.introspected_parameters:
            params.append(
                {
                    "key": introspected.key,
                    "name": introspected.key,
                    "required": False,
                    # TODO: infer description from docstring if provided
                    "description": introspected.key,
                    "type": introspected.to_dict(),
                }
            )

        container_repository, container_label = split_full_name(
            self.tool_config.container
        )
        self.data = {
            "key": self.tool_config.key,
            "name": self.tool_config.key,
            "schema_version": constants.TOOL_SPEC_SCHEMA_VERSION,
            "container_image": {
                "repository": container_repository,
                "label": container_label,
            },
            "command": {
                "file_path": self.tool_config.code_file,
                "entry_point": self.tool_config.function,
                "program_type": self.tool_config.executor_type,
            },
            "params": params,
            "results": [
                {
                    "key": "tool_output_group",
                    "name": "Tool Outputs",
                    "description": "Tool outputs",
                    "file_patterns": ["**/*"],
                }
            ],
            "resources": {
                "cpu": 1,
                "gpu": 0,
                "memory": 1000,
            },
        }

    def _validate(self) -> None:
        """
        Validate tool spec properties.

        :raises ToolSchemaValidationError: If the tool spec failed to validate using the v3 tool spec IDEAS schema.
        """
        # validate the local tool spec with the IDEAS v3 tool spec schema
        try:
            schemas.validate_v3_tool_spec(self.data)
        except SchemaValidationError as e:
            raise exceptions.ToolSchemaValidationError(
                f"Failed to validate v3 tool spec\n{e.message}"
            )

    def load(self) -> None:
        """
        Load the existing configuration from the user's local codebase

        :raises ToolMisconfiguredError: If the tool is not configured correctly.
        :raises ToolSchemaValidationError: If the tool spec failed to validate using the v3 tool spec IDEAS schema.
        """
        if not os.path.exists(self.file):
            raise exceptions.ToolMisconfiguredError(
                f"Tool spec file not found: {self.file}", key=self.tool_config.key
            )
        with open(self.file, "r") as f:
            self.data = json.load(f)

        self._validate()

    def save(self) -> None:
        """
        Save the tool spec to a JSON file in the user's local codebase.

        :raises ToolMisconfiguredError: If the tool is misconfigured.
        :raises ToolSchemaValidationError: If the tool spec failed
            to validate using the v3 tool spec IDEAS schema.
        """
        self._generate()
        self._validate()
        with open(self.file, "w") as f:
            json.dump(self.data, f, indent=4)


class ToolInputs:
    """
    The inputs of a tool to use for local execution.

    :param tool_config: The local configuration of a tool.
        Used to generate default inputs for a tool,
        and validate parameter keys in existing input files.
    :param code_dir: The directory of the user's local codebase
        containing analysis for tools.
    :param dir: The directory of the ideas configuration files
        for tools in the user's local codebase.
    :param file: The spec file for this tool in the user's
        local codebase.
    :param data: The input data represented as a dict, for serialization
        to a JSON file that is saved to the user's local codebase.
    """

    tool_config: ToolConfig
    code_dir: str
    dir: str
    file: str
    data: typing.Optional[typing.Dict] = None

    def __init__(self, tool_config: ToolConfig, code_dir: typing.Optional[str] = None):
        self.tool_config = tool_config
        self.code_dir = code_dir if code_dir else os.getcwd()
        self.dir = os.path.join(
            self.code_dir, constants.IDEAS_DIR_NAME, self.tool_config.key
        )
        self.file = os.path.join(self.dir, constants.IDEAS_INPUTS_FILE_NAME)

    def load(self, file: typing.Optional[str] = None) -> None:
        """
        Load the existing inputs from the user's local codebase

        :raises ToolMisconfiguredError: If the tool is not configured correctly.

        """
        if file is None:
            if not os.path.exists(self.file):
                raise exceptions.ToolMisconfiguredError(
                    f"Default inputs file not found: {self.file}",
                    key=self.tool_config.key,
                )
            file = self.file
        else:
            if not os.path.exists(file):
                raise exceptions.ToolInputsNotFoundError(
                    f"Inputs file not found: {self.file}"
                )

        with open(file, "r") as f:
            self.data = json.load(f)

        self._validate()

    def _generate(self) -> None:
        """
        Generates tool inputs from the introspected params in the tool config.
        """
        # if there are existing inputs (only happens on tool run)
        # don't overwrite changes made by user. only update added or removed args
        if self.data is None:
            self.data = {}

        for argument in self.tool_config.introspected_parameters:
            if argument.key not in self.data:
                self.data[argument.key] = argument.default

        for arg_key in self.data.keys():
            if not any(
                [arg.key == arg_key for arg in self.tool_config.introspected_parameters]
            ):
                del self.data[arg_key]

    def _validate(self) -> None:
        """
        Validate tool inputs properties.

        :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
        """
        for input_key, input_value in self.data.items():
            matching_arguments = [
                a
                for a in self.tool_config.introspected_parameters
                if a.key == input_key
            ]
            if len(matching_arguments) != 1:
                raise exceptions.ToolInputsFormatError(
                    f"Input key {input_key} not found in tool config",
                    key=input_key,
                    value=input_value,
                )

            if input_value is None:
                continue

            argument = matching_arguments[0]
            try:
                argument.validate(input_value)

                # show a warning about type annotation missing if input arg fails to validate
                if argument.type_annotation is None:
                    display.show_warning(
                        f"Argument {argument.key} does not have a type annotation. For the best experience please add type annotations to your tool function."
                    )
            except exceptions.ToolInputsFormatError as error:
                display.show_warning(
                    f"Tool inputs not formatted correctly. [bold magenta]{error.message}[/bold magenta]",
                )

    def save(self) -> None:
        """
        Save the tool inputs to a JSON file in the user's local codebase.

        :raises ToolMisconfiguredError: If the tool is misconfigured.
        :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
        """
        self._generate()
        self._validate()
        with open(self.file, "w") as file:
            json.dump(self.data, file, indent=4)

    def write_tmp_inputs(self) -> typing.Tuple[str, typing.List[str]]:
        """
        Write tmp inputs file for local tool execution inside docker container.
        This function will detect any file path inputs, and map these file paths from the
        user's local filesystem, the docker container file system.
        This allows users to define file inputs without having to worry about
        mounting files to the running docker container.
        """
        input_files = {}
        for input_key, input_value in self.data.items():
            matching_arguments = [
                a
                for a in self.tool_config.introspected_parameters
                if a.key == input_key
            ]
            assert len(matching_arguments) == 1
            ideas_type = matching_arguments[0].ideas_type
            if ideas_type == params.IdeasParameterType.PATH:
                # if param type is a path param we need to mount local input files to docker container
                # the existance of these paths is validated in ToolInputs._validate()
                input_files[input_key] = input_value

        input_files_volume_paths = []
        new_input_files = {}
        for input_key, input_file in input_files.items():
            if isinstance(input_file, str):
                input_files_volume_paths.append(
                    f"{os.path.abspath(os.path.expanduser(input_file))}:{constants.CONTAINER_DATA_DIR_PATH}/{os.path.basename(input_file)}"
                )
                new_input_files[input_key] = (
                    f"{constants.CONTAINER_DATA_DIR_PATH}/{os.path.basename(input_file)}"
                )
            elif isinstance(input_file, list):
                new_input_files[input_key] = []
                for f in input_file:
                    input_files_volume_paths.append(
                        f"{os.path.abspath(os.path.expanduser(f))}:{constants.CONTAINER_DATA_DIR_PATH}/{os.path.basename(f)}"
                    )
                    new_input_files[input_key].append(
                        f"{constants.CONTAINER_DATA_DIR_PATH}/{os.path.basename(f)}"
                    )

        tmp_inputs = self.data | new_input_files
        tmp_inputs_file = f"{tempfile.gettempdir()}/inputs.json"
        with open(tmp_inputs_file, "w") as f:
            json.dump(tmp_inputs, f, indent=4)
        return tmp_inputs_file, input_files_volume_paths


class Tool:
    """
    A combination of local analysis code and a docker container image.
    This class enables running tools locally, mimicking the execution of tools in the cloud.

    :param key: The key of the tool.
        Configured in the cloud when a tool is created.
    :param code_dir: The directory of the user's local codebase
        containing analysis for tools.
    :param config: The tool config, specifying the local code and container to use for execution.
    :param spec: The tool spec derived from the tool config.
    :param inputs: The tool inputs derived from the tool config.
    :param platform: The platform to use for running docker containers.
    :param running_container_name: The name of the running docker container.
    """

    key: str
    code_dir: str
    config: ToolConfig
    spec: ToolSpec
    inputs: ToolInputs
    platform: str = constants.CONTAINER_ARCHITECTURE
    running_container_name: str = constants.RUNNING_CONTAINER_NAME

    def __init__(self, key: str, code_dir: typing.Optional[str] = None):
        self.key = key
        self.code_dir = code_dir if code_dir else os.getcwd()
        self.config = ToolConfig(key=key, code_dir=code_dir)
        self.spec = ToolSpec(tool_config=self.config, code_dir=code_dir)
        self.inputs = ToolInputs(tool_config=self.config, code_dir=code_dir)

    def configure(
        self,
        file: typing.Optional[str] = None,
        function: typing.Optional[str] = None,
        container: typing.Optional[str] = None,
    ) -> None:
        """
        Configure tool for local execution.

        :param key: The key of the tool in IDEAS. Configured in the cloud when a tool is created.
        :param file: The file containing the entry point of analysis code to run for the tool.
            For python execution, this is the python module containing the entry point function to execute.
        :param function: The entry point function within the specified file to run for the tool.
        :param container: The container to run the local code within.
        :param code_dir: The code directory to configure tools within. By default, the current directory is used.

        :raises ToolMisconfiguredError: If the tool is misconfigured.
        :raises ToolIntrospectionError: If the code file could not be introspected.
        :raises ToolConfigFileNotFoundError: If the code file could not be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not be found in the local docker images.
        :raises ToolSchemaValidationError: If the tool spec failed to validate using the v3 tool spec IDEAS schema.
        :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        self.config.save(file=file, function=function, container=container)
        self.spec.save()
        self.inputs.save()

    def _process_outputs(self, tool_output_dir: str) -> None:
        """Process outputs after running tool."""
        logger.debug("Finished running tool, detecting output data")
        output_files = glob.glob(f"{tool_output_dir}/**", recursive=True)
        output_files = [f[len(tool_output_dir) + 1 :] for f in output_files]
        output_files = [
            f
            for f in output_files
            if not f.endswith(constants.IDEAS_OUTPUT_DATA_FILE_NAME)
            and not f.endswith(constants.IDEAS_INPUTS_FILE_NAME)
            and os.path.isfile(os.path.join(tool_output_dir, f))
        ]
        output_files.sort()

        output_data = None
        output_data_file = os.path.join(
            tool_output_dir, constants.IDEAS_OUTPUT_DATA_FILE_NAME
        )
        if os.path.exists(output_data_file):
            with open(output_data_file, "r") as f:
                output_data = json.load(f)
        else:
            display.show_warning(
                f"No {constants.IDEAS_OUTPUT_DATA_FILE_NAME} found! If you want previews and metadata to appear on IDEAS, generate this file in your code."
            )
        output_files_summary = []
        all_preview_files = set()
        for output_file in output_files:
            previews = []
            metadata = []

            if output_data:
                for od in output_data["output_files"]:
                    if od["file"] == output_file:
                        previews = od["previews"]
                        metadata = od["metadata"]

                        preview_files = set([p["file"] for p in od["previews"]])
                        all_preview_files = all_preview_files | preview_files
                        break

            if previews:
                for p in previews:
                    p["caption"] = textwrap.fill(p["caption"], width=60)
                previews = [
                    f"File: {p['file']}\n- Caption: {p['caption']}" for p in previews
                ]
                previews = "\n\n".join(previews)
            else:
                previews = "No previews found!"

            if metadata:
                metadata = [
                    f"Name: {m['name']}\n- Key: {m['key']}\n- Value: {m['value']}"
                    for m in metadata
                ]
                metadata = "\n\n".join(metadata)
            else:
                metadata = "No metadata found!"

            output_files_summary.append([output_file, previews, metadata])

        output_files_summary = [
            f for f in output_files_summary if f[0] not in all_preview_files
        ]

        table = Table(show_lines=True)
        table.add_column("File")
        table.add_column("Previews")
        table.add_column("Metadata")
        for x in output_files_summary:
            table.add_row(x[0], x[1], x[2])
        display.console.print(table)

    def _run(self, clean: bool = False, gpus: typing.Optional[str] = None) -> str:
        """
        Internal run function.

        :raises ToolConfigContainerNotFoundError: If the container could not be found in the local docker images.
        :raises ToolExecutionError: If there was an error running the analysis code in the docker container.
        :raises ToolException: If there was an unexpected tool error.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        tmp_inputs_file, input_files_volume_paths = self.inputs.write_tmp_inputs()
        code_dir_name = os.path.basename(self.code_dir)
        if self.config.executor_type == ExecutorType.PYTHON:
            module_name = self.config.code_file
            if module_name.startswith("./"):
                module_name = module_name.split("./")[-1]
            module_name = module_name.replace(".py", "").replace("/", ".")
            tool_command = [
                "python",
                "-c",
                _generate_tool_python_command(
                    code_dir_name, module_name, self.config.function
                ),
            ]
            env_args = []
        elif self.config.executor_type == ExecutorType.BASH:
            tool_command = [
                "bash",
                "-c",
                f"/tmp/{code_dir_name}/{self.config.code_file}",
            ]

            # Pass inputs as environment variables in docker, hacky, but works for a quick POC.
            # We should instead bake these inputs into the command; use tree-sitter to find and replace
            # the variables
            with open(tmp_inputs_file, "r") as f:
                input_values = json.load(f)

            env_args = []
            for input_key, input_value in input_values.items():
                if isinstance(input_value, str):
                    input_value = shlex.quote(input_value)
                    env_args.append(f"{input_key}='{input_value}'")
                else:
                    env_args.append(f"{input_key}={input_value}")
        else:
            raise exceptions.ToolException(
                f"Executor type {self.config.executor_type} not supported"
            )

        # Create local output dir and bind mount to container output dir
        tool_all_outputs_dir = os.path.join(self.code_dir, "outputs", self.config.key)
        if clean and os.path.exists(tool_all_outputs_dir):
            shutil.rmtree(tool_all_outputs_dir)

        timestamp = datetime.now().strftime("%Y-%m-%dT%H.%M.%S")
        tool_output_dir = os.path.join(tool_all_outputs_dir, timestamp)
        os.makedirs(tool_output_dir)

        client = get_docker_client()
        try:
            running_container = client.containers.get(self.running_container_name)
            running_container.remove()
        except docker.errors.NotFound:
            pass
        except docker.errors.DockerException:
            raise exceptions.DockerExecutionError(
                "Failed to remove previous tool run container due to unexpected error with docker."
            )

        # validate that the python command exists in the image
        # in the future, we should support user-defined python commands
        running_container = None
        try:
            running_container = client.containers.run(
                image=self.config.container,
                command="python --version",
                platform=self.platform,
                detach=True,
                name=self.running_container_name,
            )
            result = running_container.wait()
            exit_code = result["StatusCode"]
            if exit_code:
                raise exceptions.ToolExecutionError(
                    f"Checking if python command exists resulted in an error, with exit status: {exit_code}"
                )
        except docker.errors.DockerException:
            raise exceptions.ContainerPythonNotFound(
                "Failed to validate python command exists in running container."
            )
        finally:
            if running_container is not None:
                try:
                    running_container.remove()
                except docker.errors.DockerException:
                    raise exceptions.DockerExecutionError(
                        "Failed to remove tool run container due to unexpected error with docker."
                    )

        try:
            logger.debug(
                f"Starting container {self.config.container}, running file: {self.config.code_file}, function: {self.config.function}"
            )

            ## additional kwargs to pass to docker run
            kwargs = {}
            if gpus:
                if gpus == "all":
                    kwargs["device_requests"] = [
                        docker.types.DeviceRequest(capabilities=[["gpu"]], count=-1)
                    ]
                else:
                    try:
                        gpu_count = int(gpus)
                    except ValueError:
                        raise exceptions.ToolExecutionError(
                            "Failed to convert gpu count to number"
                        )

                    kwargs["device_requests"] = [
                        docker.types.DeviceRequest(
                            capabilities=[["gpu"]], count=gpu_count
                        )
                    ]
                kwargs["runtime"] = (
                    "nvidia"  # currently only nvidia runtime is supported for gpus
                )

            running_container = client.containers.run(
                image=self.config.container,
                command=tool_command,
                platform=self.platform,
                volumes=[
                    f"{self.code_dir}:/tmp/{code_dir_name}",
                    f"{tmp_inputs_file}:/tmp/{constants.IDEAS_INPUTS_FILE_NAME}",
                ]
                + input_files_volume_paths,
                environment=env_args,
                name=self.running_container_name,
                detach=True,  # detach and get logs from container output
                **kwargs,
            )
            for line in running_container.logs(stream=True):
                print(line.decode("utf-8"), end="")

            result = running_container.wait()
            exit_code = result["StatusCode"]
            if exit_code:
                raise exceptions.ToolExecutionError(
                    f"Tool execution resulted in an error, with exit status: {exit_code}"
                )

            copy_files_from_container_to_host(
                container=running_container,
                container_path=f"{constants.CONTAINER_OUTPUT_DIR_PATH}/.",
                host_path=tool_output_dir,
            )

        except docker.errors.APIError as e:
            if e.explanation and "nvidia" in e.explanation:
                raise exceptions.NvidiaRuntimeMissingError(
                    "Failed to run tool container due to missing nvidia runtime."
                ) from e
            else:
                raise exceptions.DockerExecutionError(
                    "Failed to run tool container due to unexpected error with docker."
                ) from e
        except docker.errors.DockerException:
            raise exceptions.DockerExecutionError(
                "Failed to run tool container due to unexpected error with docker."
            )
        finally:
            try:
                running_container.remove()
            except docker.errors.NotFound:
                # Nothing to do, already removed
                pass
            except docker.errors.DockerException:
                raise exceptions.DockerExecutionError(
                    "Failed to remove tool run container due to unexpected error with docker."
                )

        return tool_output_dir

    def load(self, inputs_file: typing.Optional[str] = None) -> None:
        """
        Load the existing configuration, spec, and inputs from the user's local codebase

        :param inputs_file: The file containing inputs to run the tool with, formatted in JSON.
            By default, .ideas/<key>/inputs.json is used.

        :raises ToolMisconfiguredError: If the tool is not configured correctly.
        :raises ToolConfigFileNotFoundError: If the code file could not
            be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not
            be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not
            be found in the local docker images.
        :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        self.config.load()
        self.spec.load()
        self.inputs.load(file=inputs_file)

    def run(
        self,
        inputs_file: typing.Optional[str] = None,
        clean: bool = False,
        skip_introspect: bool = False,
        gpus: typing.Optional[str] = None,
    ) -> None:
        """
        Run tool locally by executing local analysis code in container image and process output files.

        :param key: The key of the tool in IDEAS. Configured in the cloud when a tool is created.
        :param inputs_file: The file containing inputs to run the tool with, formatted in JSON.
            By default, .ideas/<key>/inputs.json is used.
        :param clean: Remove previous tool output directories.
        :param code_dir: The code directory to configure tools within. By default, the current directory is used.
        :param skip_introspect: Skips re-introspecting function to ensure all arguments are up to date.
        :param gpus: Enable gpu access to docker container. Value can either be `all` indicating to use all gpus, or a number indicating the amount of gpus required. Uses nvidia gpu docker runtime.

        :raises ToolMisconfiguredError: If the tool is not configured correctly.
        :raises ToolConfigFileNotFoundError: If the code file could not
            be found in the user's local codebase.
        :raises ToolConfigFunctionNotFoundError: If the function could not
            be found in the code file.
        :raises ToolConfigContainerNotFoundError: If the container could not
            be found in the local docker images.
        :raises ToolInputsFormatError: If the tool inputs are formatted incorrectly.
        :raises ToolExecutionError: If there was an error running the analysis code in the docker container.
        :raises ToolException: If there was an unexpected tool error.
        :raises DockerExecutionError: If there was an unexpected error with docker.
        """
        # load current configuration and re-run introspection
        # to update tool spec based off updates to function args
        if not skip_introspect:
            self.load()
            self.configure()

        # load user-specified inputs and run tool
        self.load(inputs_file=inputs_file)
        tool_output_dir = self._run(clean=clean, gpus=gpus)
        self._process_outputs(tool_output_dir=tool_output_dir)
        return tool_output_dir
