import enum
import os
import typing
from abc import ABC, abstractmethod

from ideas import exceptions


class IdeasParameterType(enum.StrEnum):
    """
    The types of parameters supported in IDEAS
    Each parameter type determines different behavior
    for the corresponding analysis table columns in IDEAS.
    """

    PATH = "ToolPathParam"  # File path parameter
    INT = "ToolIntRangeParam"  # Int parameter with optional range
    FLOAT = "ToolFloatRangeParam"  # Float parameter with optional range
    BOOL = "ToolBooleanParam"  # Bool parameter
    STR = "ToolStringParam"  # String parameter


class Parameter(ABC):
    """
    Abstract class to represent a parameter introspected from a function signature.

    :param key: The name of the parameter in the function signature.
    :param type_annotation: The type annotation specified for the
        parameter in the function signature.
    :param default: The default value assigned to the parameter
        in the function signature.
    :param ideas_type: The corresponding parameter type in IDEAS.
    :param validation_class: The python type class to use for validating
        input instances of the parameter.
    """

    def __init__(
        self,
        key: str,
        type_annotation: str,
        default: typing.Any,
    ):
        self.key = key
        self.type_annotation = type_annotation
        self.default = default

    def __eq__(self, other):
        return (
            self.key == other.key
            and self.type_annotation == other.type_annotation
            and self.default == other.default
            and self.to_dict() == other.to_dict()
        )

    def __repr__(self):
        return f"Parameter(key={self.key}, type_annotation={self.type_annotation}, default={self.default})"

    @property
    @abstractmethod
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        pass

    @property
    @abstractmethod
    def data_type(self) -> str:
        pass

    @property
    @abstractmethod
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        pass

    def to_dict(self) -> dict:
        """Convert to dict for IDEAS tool spec."""
        data = {"param_type": str(self.ideas_type)}
        if self.default is not None and isinstance(self.default, self.validation_class):
            data["default"] = self.default
        return data

    def validate(self, instance: typing.Any) -> None:
        """
        Validate an input value for this parameter.

        :raises ToolInputsFormatError: If the input value failed to validate for this parameter.
        """
        if not isinstance(instance, self.validation_class):
            raise exceptions.ToolInputsFormatError(
                f"Input value ({instance}) for parameter {self.key} does not match expected type {self.validation_class}",
                key=self.key,
                value=instance,
            )


class PathParameter(Parameter):
    """
    Parameter for an input file.
    This parameter type only supports a single input file.
    See `PathMultipleParameter` for multiple file support.
    """

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.PATH

    @property
    def data_type(self) -> str:
        return "string"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return str

    def to_dict(self) -> dict:
        """Convert to dict for IDEAS tool spec."""
        data = super().to_dict()
        data["multiple"] = False
        if "default" in data:
            del data["default"]
        return data

    def validate(self, instance: typing.Any):
        """
        Validate an input value for this parameter.

        :raises ToolInputsFormatError: If the input value failed to validate for this parameter.
        """
        if instance is None:
            # support none for input paths
            # TODO: distinguish between optional and non-optional type annotation
            return

        super().validate(instance)
        if not os.path.exists(os.path.expanduser(instance)):
            # if the type is a path param and the file doesn't exist, hard fail
            raise exceptions.ToolInputsFileNotFoundError(
                f"Failed to find file for path param: {os.path.expanduser(instance)}",
                file=instance,
            )


class PathMultipleParameter(Parameter):
    """Parameter for one or more input files."""

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.PATH

    @property
    def data_type(self) -> str:
        return "array"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return list

    def to_dict(self) -> dict:
        """Convert to dict for IDEAS tool spec."""
        data = super().to_dict()
        data["multiple"] = True
        if "default" in data:
            del data["default"]
        return data

    def validate(self, instance: typing.Any):
        """
        Validate an input value for this parameter.

        :raises ToolInputsFormatError: If the input value failed to validate for this parameter.
        """
        if instance is None:
            # support none for input paths
            # TODO: distinguish between optional and non-optional type annotation
            return

        if not isinstance(instance, list) or not all(
            [isinstance(v, str) for v in instance]
        ):
            raise exceptions.ToolInputsFormatError(
                f"Input value ({instance}) for parameter {self.key} does not match expected type list with all string values",
                key=self.key,
                value=instance,
            )

        for v in instance:
            if not os.path.exists(os.path.expanduser(v)):
                # if the type is a path param and the file doesn't exist, hard fail
                raise exceptions.ToolInputsFileNotFoundError(
                    f"Failed to find file for path param: {os.path.expanduser(v)}",
                    file=v,
                )


class IntParameter(Parameter):
    """Parameter for integer inputs."""

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.INT

    @property
    def data_type(self) -> str:
        return "integer"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return int


class FloatParameter(Parameter):
    """Parameter for floating point inputs."""

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.FLOAT

    @property
    def data_type(self) -> str:
        return "float"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return float

    def validate(self, instance: typing.Any) -> None:
        """
        Validate an input value for this parameter.

        :raises ToolInputsFormatError: If the input value failed to validate for this parameter.
        """
        try:
            super().validate(instance)
        except exceptions.ToolInputsFormatError as error:
            if not isinstance(instance, int):
                raise error


class BoolParameter(Parameter):
    """Parameter for boolean inputs."""

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.BOOL

    @property
    def data_type(self) -> str:
        return "boolean"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return bool


class StringParameter(Parameter):
    """Parameter for string inputs."""

    def __init__(self, key: str, type_annotation: str, default: typing.Any):
        super().__init__(
            key=key,
            type_annotation=type_annotation,
            default=default,
        )

    @property
    def ideas_type(self) -> IdeasParameterType:
        """The corresponding type of parameter type in IDEAS"""
        return IdeasParameterType.STR

    @property
    def data_type(self) -> str:
        return "string"

    @property
    def validation_class(self) -> type:
        """The python type class to use for validating input instances of the parameter."""
        return str

    def validate(self, instance: typing.Any) -> None:
        """
        Validate an input value for this parameter.

        :raises ToolInputsFormatError: If the input value failed to validate for this parameter.
        """
        try:
            super().validate(instance)
        except exceptions.ToolInputsFormatError:
            # If the type annotation is str, then raise validation error.
            if self.type_annotation == "str":
                raise exceptions.ToolInputsFormatError(
                    f"Input value ({instance}) for parameter {self.key} does not match expected type {self.validation_class}."
                    f"\nIf this parameter is meant to be a file input, ensure you use the type annotation `pathlib.Path` or `List\\[pathlib.Path]` in your function.",
                    key=self.key,
                    value=instance,
                )


# Map of python type annotations to parameter types.
TYPE_ANNOTATION_TO_IDEAS_PARAMETER_TYPE = {
    # pathlib.Path is a hacky way to indicate an input file for the tool runner using just the standard lib
    # this not publicly documented but still included for internal support
    "pathlib.Path": PathParameter,
    "Optional[pathlib.Path]": PathParameter,
    "List[pathlib.Path]": PathMultipleParameter,
    "Optional[List[pathlib.Path]]": PathMultipleParameter,
    # IdeasFile is the recommended approach for indicating input files
    "IdeasFile": PathParameter,
    "Optional[IdeasFile]": PathParameter,
    "List[IdeasFile]": PathMultipleParameter,
    "Optional[List[IdeasFile]]": PathMultipleParameter,
    # Python primitive types
    "int": IntParameter,
    "Optional[int]": IntParameter,
    "float": FloatParameter,
    "Optional[float]": FloatParameter,
    "bool": BoolParameter,
    "Optional[bool]": BoolParameter,
    "str": StringParameter,
    "Optional[str]": StringParameter,
}


def create_parameter(key: str, type_annotation: str, default: typing.Any):
    """
    Factory function to create a parameter from a function signature.
    This method tries to map the type annotation in the function signautre
    to a compatible IDEAS parameter type.
    If not compatible IDEAS parameter type exists, the function defaults
    to mapping to a `StringParameter` type.
    """
    param_class = TYPE_ANNOTATION_TO_IDEAS_PARAMETER_TYPE.get(
        type_annotation, StringParameter
    )

    return param_class(key=key, type_annotation=type_annotation, default=default)
