"""
Helpers for parsing AST from tool code.
"""

import ast
import enum
import mimetypes
import pathlib
import typing as t

import tree_sitter_bash as tsbash
import tree_sitter_python as tspython
from tree_sitter import Language, Node, Parser, Query

from ideas.exceptions import (
    ToolConfigFileNotFoundError,
    ToolIntrospectionError,
)
from ideas.tools.introspection import params

BASH_LANG = Language(tsbash.language())
PYTHON_LANG = Language(tspython.language())

HERE = pathlib.Path(__file__).parent
PYTHON_QUERY_PATH = HERE / "queries" / "python.scm"
BASH_QUERY_PATH = HERE / "queries" / "bash.scm"
NOTSET = object()


class ExecutorType(enum.StrEnum):
    """Type of executors supported for tools in IDEAS"""

    PYTHON = "PYTHON"
    BASH = "BASH"


def parse_ast(source_bytes, query, tree) -> list:
    def get_text(node: Node) -> str:
        # Helper to return the text for a specific node
        return node.text.decode("utf-8")

    def get_key(captures, key: str, default=NOTSET) -> t.Optional[str]:
        try:
            capture: t.List[Node] = captures[key]
        except KeyError:
            if default == NOTSET:
                raise ToolIntrospectionError(
                    f"Key {key} missing from introspection results"
                )

            # No value set, doesn't apply
            return None

        if len(capture) != 1:
            raise ToolIntrospectionError(
                f"Captured {len(captures)} when we expected one"
            )

        return get_text(capture[0])

    # Build results
    functions = {}
    for _, captures in query.matches(tree.root_node):
        function_name = get_key(captures, "func.name")
        functions.setdefault(function_name, [])

        arg_name = get_key(captures, "arg.name", default=None)
        if arg_name is not None:
            arg_type = get_key(captures, "arg.type", default=None)
            arg_default = get_key(captures, "arg.default", default=None)
            if arg_default is not None:
                try:
                    arg_default = ast.literal_eval(arg_default)
                except ValueError:
                    arg_default = None
            # If there's not type annotation, try to infer from default
            if arg_type is None and arg_default is not None:
                arg_type = type(arg_default).__name__

            functions[function_name].append(
                params.create_parameter(
                    key=arg_name, type_annotation=arg_type, default=arg_default
                )
            )
    return functions


def parse_python_ast(source_bytes: bytes) -> dict:
    parser = Parser(PYTHON_LANG)
    tree = parser.parse(source_bytes)

    PYTHON_QUERY = PYTHON_QUERY_PATH.read_text()

    # Load the query
    query = Query(PYTHON_LANG, PYTHON_QUERY)

    return parse_ast(
        source_bytes,
        query,
        tree,
    )


def parse_shell_ast(source_bytes: bytes) -> dict:
    parser = Parser(BASH_LANG)
    tree = parser.parse(source_bytes)

    BASH_QUERY = BASH_QUERY_PATH.read_text()

    # Load the query
    query = Query(BASH_LANG, BASH_QUERY)

    return parse_ast(
        source_bytes,
        query,
        tree,
    )


def read_code(file: str) -> t.Tuple[ExecutorType, t.Dict]:
    """
    Read the code file into memory.

    :returns: A tuple containing the guessed mimetype of the code file
        (used to determine the executor type), and the contents of
        the code file.
    :raises ToolConfigFileNotFoundError: If the code file could not
        be found in the user's local codebase.
    """
    try:
        with open(file, "rb") as f:
            code = f.read()
    except FileNotFoundError:
        raise ToolConfigFileNotFoundError(f"Failed to find code file: {file}")

    mimetype, _ = mimetypes.guess_type(file)
    return mimetype, code


def introspect_code(code_file: str) -> None:
    """
    Introspects user's code file, detecting all functions defined

    :raises ToolConfigFileNotFoundError: If the code file could not
        be found in the user's local codebase.
    :raises ToolIntrospectionError: If the code file could not
        be introspected.
    """
    mimetype, code = read_code(code_file)
    match mimetype:
        case "text/x-python":
            return ExecutorType.PYTHON, parse_python_ast(code)
        case "text/x-sh":
            return ExecutorType.BASH, parse_shell_ast(code)
        case "application/x-sh":
            return ExecutorType.BASH, parse_shell_ast(code)
        case "text/plain":
            if code_file.endswith(".py"):
                return ExecutorType.PYTHON, parse_python_ast(code)
            else:
                # default to shell execution for plain text file
                return ExecutorType.BASH, parse_shell_ast(code)
        case _:
            raise ToolIntrospectionError(
                f"Unsupported or unguessable script type: {mimetype}"
            )
