import typing

import click

KeyValueOptionValue: typing.TypeAlias = tuple[str, str]


class KeyValueOption(click.ParamType):
    """
    Process arguments expected to be in the form `foo=bar`, raising an error
    if they do not conform to this specification.
    """

    name = "key-value option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        try:
            key, value = value.split("=", maxsplit=1)
            return (key, value)
        except ValueError:
            if param is not None:
                param_msg = f" for {param.name}"
            else:
                param_msg = ""

            self.fail(
                f"{value} is not a valid value{param_msg}. Values must be separated as key=value",
                param,
                ctx,
            )


class SearchOption(click.ParamType):
    """
    Returns a single `KeyValueOptionValue` suitable to mix into the filters to
    search objects in IDEAS by partial match.
    """

    name = "search option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        if not value:
            return tuple()
        return (("search", str(value)),)


class TenantIdOption(click.ParamType):
    """
    Acts like an `int` but returns a str that can be used in a header.
    """

    name = "tenant-id"

    def convert(self, value, param, ctx):
        try:
            int(value)
            return str(value)
        except (TypeError, ValueError):
            self.fail(f"{value} is not a valid tenant-id", param, ctx)


class ListOption(click.ParamType):
    """
    Process arguments expected to be in the form `foo,bar`, raising an error
    if they do not conform to this specification.
    """

    name = "list option"

    def convert(
        self,
        value: str,
        param: typing.Optional[click.core.Option],
        ctx: click.core.Context,
    ) -> KeyValueOptionValue:
        try:
            str(value)
            return str(value)
        except ValueError:
            self.fail(f"{value} is not a valid list option", param, ctx)
