from __future__ import annotations

import io
import logging
import tarfile

import docker
import docker.errors

from ideas import exceptions

CHUNK_SIZE = 10 * 1024 * 1024

logger = logging.getLogger(__name__)


def get_docker_client():
    """Get the docker client"""
    try:
        client = docker.from_env()
    except docker.errors.DockerException:
        raise exceptions.DockerUnavailableError(
            "Failed to get docker client, ensure docker is running"
        )
    return client


### ------------------------------------------------------------------------------
### Copied from ideas-docker-agent/src/ideas_docker_agent/container/filesystem.py
### Helpers for dealing with container filesystem operations, through docker sdk.


class StreamWrapper(io.RawIOBase):
    """
    Convert a tar stream from docker into a file-like object to be used with tarfile.

    Acts like an unbuffered raw file object, wrapping io.RawIOBase.

    References:
    - https://docs.python.org/3/glossary.html#term-file-object
    - https://docs.python.org/3/library/io.html#io.RawIOBase
    """

    def __init__(self, generator):
        self.generator = generator
        self.buffer = b""

    def readable(self):
        """
        Determines if stream can be read from. Our stream isn't a file object, it's a generator
        from docker-sdk; thus, it's always readable.
        """
        return True

    def readinto(self, b):
        """
        Read bytes from docker archived stream into the pre-allocated buffer from io.RawIOBase.

        Returns the number of bytes remaining to be read, or zero if the stream is complete.
        """
        try:
            # Use our buffer first, then consume from docker stream
            if not self.buffer:
                self.buffer = next(self.generator)
        except StopIteration:
            # End of stream from docker
            return 0

        # We copy data from our local buffer into the io.RawIOBase buffer
        bytes_to_read = min(len(b), len(self.buffer))
        b[:bytes_to_read] = self.buffer[:bytes_to_read]
        # If we read more into our local buffer than fits into the RawIOBase buffer, we truncate our
        # local buffer and will send it on the next read
        self.buffer = self.buffer[bytes_to_read:]

        return bytes_to_read


def copy_files_from_container_to_host(
    container, container_path, host_path, chunk_size: int = CHUNK_SIZE
):
    """Copy files from container to host"""
    try:
        strm, _ = container.get_archive(container_path, chunk_size)
    except docker.errors.NotFound as e:
        raise FileNotFoundError from e

    with tarfile.open(fileobj=StreamWrapper(strm), mode="r|*") as tar:
        tar.extractall(path=host_path)


### Copied from ideas-docker-agent/src/ideas_docker_agent/container/filesystem.py
### ------------------------------------------------------------------------------
