# -*- coding: utf-8 -*-
"""
@author:XuMing(xuming624@qq.com)
@description:
RunPythonCode MCP Server implementation.
This module provides Python code execution tools via MCP (Model Context Protocol).
"""
from typing import Optional
import tempfile
import asyncio
from mcp.server import Server
from mcp.types import Tool, TextContent
import mcp.server.stdio
from .code import RunPythonCode

# Create MCP server instance
server = Server("RunPythonCodeServer")

# Initialize the RunPythonCode tool
python_runner = RunPythonCode(base_dir=tempfile.mkdtemp())


@server.list_tools()
async def list_tools() -> list[Tool]:
    """List available tools."""
    return [
        Tool(
            name="run_python_code",
            description="Run Python code in the current environment",
            inputSchema={
                "type": "object",
                "properties": {
                    "code": {
                        "type": "string",
                        "description": "The Python code to execute"
                    }
                },
                "required": ["code"]
            }
        ),
        Tool(
            name="save_to_file_and_run",
            description="Save Python code to a file and run it",
            inputSchema={
                "type": "object",
                "properties": {
                    "file_name": {
                        "type": "string",
                        "description": "Name of the file to save (e.g., 'script.py')"
                    },
                    "code": {
                        "type": "string",
                        "description": "Python code to save and execute"
                    },
                    "variable_to_return": {
                        "type": "string",
                        "description": "Optional variable name to return its value"
                    },
                    "overwrite": {
                        "type": "boolean",
                        "description": "Whether to overwrite existing file",
                        "default": True
                    }
                },
                "required": ["file_name", "code"]
            }
        ),
        Tool(
            name="pip_install_package",
            description="Install a Python package using pip",
            inputSchema={
                "type": "object",
                "properties": {
                    "package_name": {
                        "type": "string",
                        "description": "Name of the package to install"
                    }
                },
                "required": ["package_name"]
            }
        ),
        Tool(
            name="run_python_file",
            description="Run an existing Python file",
            inputSchema={
                "type": "object",
                "properties": {
                    "file_name": {
                        "type": "string",
                        "description": "Name of the Python file to run"
                    },
                    "variable_to_return": {
                        "type": "string",
                        "description": "Optional variable name to return its value"
                    }
                },
                "required": ["file_name"]
            }
        )
    ]


@server.call_tool()
async def call_tool(name: str, arguments: dict) -> list[TextContent]:
    """Call a tool with the given arguments."""
    if name == "run_python_code":
        code = arguments.get("code", "")
        result = python_runner.run_python_code(code)
        return [TextContent(type="text", text=str(result))]
    
    elif name == "save_to_file_and_run":
        file_name = arguments.get("file_name", "")
        code = arguments.get("code", "")
        variable_to_return = arguments.get("variable_to_return")
        overwrite = arguments.get("overwrite", True)
        result = python_runner.save_to_file_and_run(file_name, code, variable_to_return, overwrite)
        return [TextContent(type="text", text=str(result))]
    
    elif name == "pip_install_package":
        package_name = arguments.get("package_name", "")
        result = python_runner.pip_install_package(package_name)
        return [TextContent(type="text", text=str(result))]
    
    elif name == "run_python_file":
        file_name = arguments.get("file_name", "")
        variable_to_return = arguments.get("variable_to_return")
        result = python_runner.run_python_file_return_variable(file_name, variable_to_return)
        return [TextContent(type="text", text=str(result))]
    
    else:
        raise ValueError(f"Unknown tool: {name}")


def run_mcp_server():
    """Run the MCP server with stdio transport."""
    async def main():
        async with mcp.server.stdio.stdio_server() as (read_stream, write_stream):
            await server.run(
                read_stream,
                write_stream,
                server.create_initialization_options()
            )
    
    asyncio.run(main())