import unittest
from typing import Dict, List, Optional, Union

import ddt

from iker.common.utils.typeutils import is_identical_type, is_optional_type


class DummyClass(object):
    pass


@ddt.ddt
class TypeUtilsTest(unittest.TestCase):
    data_is_optional_type = [
        (int, False),
        (float, False),
        (str, False),
        (bool, False),
        (DummyClass, False),
        (int | None, True),
        (float | None, True),
        (str | None, True),
        (bool | None, True),
        (DummyClass | None, True),
        (None | int, True),
        (None | float, True),
        (None | str, True),
        (None | bool, True),
        (None | DummyClass, True),
        (Optional[int], True),
        (Optional[float], True),
        (Optional[str], True),
        (Optional[bool], True),
        (Optional[DummyClass], True),
        (int | object | None, True),
        (float | object | None, True),
        (str | object | None, True),
        (bool | object | None, True),
        (DummyClass | object | None, True),
        (list[int], False),
        (list[float], False),
        (list[str], False),
        (list[bool], False),
        (list[DummyClass], False),
        (list[int | None], False),
        (list[float | None], False),
        (list[str | None], False),
        (list[bool | None], False),
        (list[DummyClass | None], False),
        (list[int] | None, True),
        (list[float] | None, True),
        (list[str] | None, True),
        (list[bool] | None, True),
        (list[DummyClass] | None, True),
        (dict[int, int], False),
        (dict[float, float], False),
        (dict[str, str], False),
        (dict[bool, bool], False),
        (dict[DummyClass, DummyClass], False),
        (dict[int, int | None], False),
        (dict[float, float | None], False),
        (dict[str, str | None], False),
        (dict[bool, bool | None], False),
        (dict[DummyClass, DummyClass | None], False),
        (dict[int, int] | None, True),
        (dict[float, float] | None, True),
        (dict[str, str] | None, True),
        (dict[bool, bool] | None, True),
        (dict[DummyClass, DummyClass] | None, True),
    ]

    @ddt.idata(data_is_optional_type)
    @ddt.unpack
    def test_is_optional_type(self, t, expect):
        self.assertEqual(is_optional_type(t), expect)

    data_is_identical_type = [
        (int, int, True),
        (float, float, True),
        (str, str, True),
        (bool, bool, True),
        (DummyClass, DummyClass, True),
        (int | None, int | None, True),
        (float | None, float | None, True),
        (str | None, str | None, True),
        (bool | None, bool | None, True),
        (DummyClass | None, DummyClass | None, True),
        (None | int, int | None, True),
        (None | float, float | None, True),
        (None | str, str | None, True),
        (None | bool, bool | None, True),
        (None | DummyClass, DummyClass | None, True),
        (Optional[int], int | None, True),
        (Optional[float], float | None, True),
        (Optional[str], str | None, True),
        (Optional[bool], bool | None, True),
        (Optional[DummyClass], DummyClass | None, True),
        (Union[int, None], int | None, True),
        (Union[float, None], float | None, True),
        (Union[str, None], str | None, True),
        (Union[bool, None], bool | None, True),
        (Union[DummyClass, None], DummyClass | None, True),
        (Union[None, int], int | None, True),
        (Union[None, float], float | None, True),
        (Union[None, str], str | None, True),
        (Union[None, bool], bool | None, True),
        (Union[None, DummyClass], DummyClass | None, True),
        (list[int], list[int], True),
        (list[float], list[float], True),
        (list[str], list[str], True),
        (list[bool], list[bool], True),
        (list[DummyClass], list[DummyClass], True),
        (List[int], list[int], True),
        (List[float], list[float], True),
        (List[str], list[str], True),
        (List[bool], list[bool], True),
        (List[DummyClass], list[DummyClass], True),
        (List[Optional[int]], list[int | None], True),
        (List[Optional[float]], list[float | None], True),
        (List[Optional[str]], list[str | None], True),
        (List[Optional[bool]], list[bool | None], True),
        (List[Optional[DummyClass]], list[DummyClass | None], True),
        (dict[int, int], dict[int, int], True),
        (dict[float, float], dict[float, float], True),
        (dict[str, str], dict[str, str], True),
        (dict[bool, bool], dict[bool, bool], True),
        (dict[DummyClass, DummyClass], dict[DummyClass, DummyClass], True),
        (Dict[int, int], dict[int, int], True),
        (Dict[float, float], dict[float, float], True),
        (Dict[str, str], dict[str, str], True),
        (Dict[bool, bool], dict[bool, bool], True),
        (Dict[DummyClass, DummyClass], dict[DummyClass, DummyClass], True),
        (Dict[int, List[Optional[int]]], dict[int, list[int | None]], True),
        (Dict[float, List[Optional[float]]], dict[float, list[float | None]], True),
        (Dict[str, List[Optional[str]]], dict[str, list[str | None]], True),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool | None]], True),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass | None]], True),

        (int, int | None, False),
        (float, float | None, False),
        (str, str | None, False),
        (bool, bool | None, False),
        (DummyClass, DummyClass | None, False),
        (int, Optional[int], False),
        (float, Optional[float], False),
        (str, Optional[str], False),
        (bool, Optional[bool], False),
        (DummyClass, Optional[DummyClass], False),

        (List[int], List[DummyClass], False),
        (List[float], List[DummyClass], False),
        (List[str], List[DummyClass], False),
        (List[bool], List[DummyClass], False),
        (List[int], list[DummyClass], False),
        (List[float], list[DummyClass], False),
        (List[str], list[DummyClass], False),
        (List[bool], list[DummyClass], False),

        (List[Optional[int]], list[int], False),
        (List[Optional[float]], list[float], False),
        (List[Optional[str]], list[str], False),
        (List[Optional[bool]], list[bool], False),
        (List[Optional[DummyClass]], list[DummyClass], False),
        (List[Optional[int]], list[int] | None, False),
        (List[Optional[float]], list[float] | None, False),
        (List[Optional[str]], list[str] | None, False),
        (List[Optional[bool]], list[bool] | None, False),
        (List[Optional[DummyClass]], list[DummyClass] | None, False),

        (Dict[int, int], Dict[int, DummyClass], False),
        (Dict[float, float], Dict[float, DummyClass], False),
        (Dict[str, str], Dict[str, DummyClass], False),
        (Dict[bool, bool], Dict[bool, DummyClass], False),
        (Dict[int, int], Dict[DummyClass, int], False),
        (Dict[float, float], Dict[DummyClass, float], False),
        (Dict[str, str], Dict[DummyClass, str], False),
        (Dict[bool, bool], Dict[DummyClass, bool], False),
        (Dict[int, int], dict[int, DummyClass], False),
        (Dict[float, float], dict[float, DummyClass], False),
        (Dict[str, str], dict[str, DummyClass], False),
        (Dict[bool, bool], dict[bool, DummyClass], False),
        (Dict[int, int], dict[DummyClass, int], False),
        (Dict[float, float], dict[DummyClass, float], False),
        (Dict[str, str], dict[DummyClass, str], False),
        (Dict[bool, bool], dict[DummyClass, bool], False),

        (Dict[int, List[Optional[int]]], dict[int, list[int]], False),
        (Dict[float, List[Optional[float]]], dict[float, list[float]], False),
        (Dict[str, List[Optional[str]]], dict[str, list[str]], False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]], False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]], False),
        (Dict[int, List[Optional[int]]], dict[int, list[int] | None], False),
        (Dict[float, List[Optional[float]]], dict[float, list[float] | None], False),
        (Dict[str, List[Optional[str]]], dict[str, list[str] | None], False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool] | None], False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass] | None], False),
        (Dict[int, List[Optional[int]]], dict[int, list[int]] | None, False),
        (Dict[float, List[Optional[float]]], dict[float, list[float]] | None, False),
        (Dict[str, List[Optional[str]]], dict[str, list[str]] | None, False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]] | None, False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]] | None, False),
    ]

    @ddt.idata(data_is_identical_type)
    @ddt.unpack
    def test_is_identical_type(self, a, b, expect):
        self.assertEqual(is_identical_type(a, b), expect)
        self.assertEqual(is_identical_type(b, a), expect)
