import unittest
from collections.abc import Callable, Mapping, Sequence
from typing import Dict, List, Optional, SupportsFloat, SupportsInt, Union

import ddt

from iker.common.utils.typeutils import is_identical_type, is_optional_type


class DummyClass(object):
    pass


class DummySubClass(DummyClass):
    pass


@ddt.ddt
class TypeUtilsTest(unittest.TestCase):
    data_is_optional_type = [
        (int, False),
        (float, False),
        (str, False),
        (bool, False),
        (DummyClass, False),
        (int | None, True),
        (float | None, True),
        (str | None, True),
        (bool | None, True),
        (DummyClass | None, True),
        (None | int, True),
        (None | float, True),
        (None | str, True),
        (None | bool, True),
        (None | DummyClass, True),
        (Optional[int], True),
        (Optional[float], True),
        (Optional[str], True),
        (Optional[bool], True),
        (Optional[DummyClass], True),
        (int | object | None, True),
        (float | object | None, True),
        (str | object | None, True),
        (bool | object | None, True),
        (DummyClass | object | None, True),
        (list[int], False),
        (list[float], False),
        (list[str], False),
        (list[bool], False),
        (list[DummyClass], False),
        (list[int | None], False),
        (list[float | None], False),
        (list[str | None], False),
        (list[bool | None], False),
        (list[DummyClass | None], False),
        (list[int] | None, True),
        (list[float] | None, True),
        (list[str] | None, True),
        (list[bool] | None, True),
        (list[DummyClass] | None, True),
        (dict[int, int], False),
        (dict[float, float], False),
        (dict[str, str], False),
        (dict[bool, bool], False),
        (dict[DummyClass, DummyClass], False),
        (dict[int, int | None], False),
        (dict[float, float | None], False),
        (dict[str, str | None], False),
        (dict[bool, bool | None], False),
        (dict[DummyClass, DummyClass | None], False),
        (dict[int, int] | None, True),
        (dict[float, float] | None, True),
        (dict[str, str] | None, True),
        (dict[bool, bool] | None, True),
        (dict[DummyClass, DummyClass] | None, True),
    ]

    @ddt.idata(data_is_optional_type)
    @ddt.unpack
    def test_is_optional_type(self, t, expect):
        self.assertEqual(is_optional_type(t), expect)

    data_is_identical_type = [
        (int, int, True, True),
        (float, float, True, True),
        (str, str, True, True),
        (bool, bool, True, True),
        (DummyClass, DummyClass, True, True),
        (int | None, int | None, True, True),
        (float | None, float | None, True, True),
        (str | None, str | None, True, True),
        (bool | None, bool | None, True, True),
        (DummyClass | None, DummyClass | None, True, True),
        (None | int, int | None, True, True),
        (None | float, float | None, True, True),
        (None | str, str | None, True, True),
        (None | bool, bool | None, True, True),
        (None | DummyClass, DummyClass | None, True, True),
        (Optional[int], int | None, True, True),
        (Optional[float], float | None, True, True),
        (Optional[str], str | None, True, True),
        (Optional[bool], bool | None, True, True),
        (Optional[DummyClass], DummyClass | None, True, True),
        (Union[int, None], int | None, True, True),
        (Union[float, None], float | None, True, True),
        (Union[str, None], str | None, True, True),
        (Union[bool, None], bool | None, True, True),
        (Union[DummyClass, None], DummyClass | None, True, True),
        (Union[None, int], int | None, True, True),
        (Union[None, float], float | None, True, True),
        (Union[None, str], str | None, True, True),
        (Union[None, bool], bool | None, True, True),
        (Union[None, DummyClass], DummyClass | None, True, True),
        (list[int], list[int], True, True),
        (list[float], list[float], True, True),
        (list[str], list[str], True, True),
        (list[bool], list[bool], True, True),
        (list[DummyClass], list[DummyClass], True, True),
        (List[int], list[int], True, True),
        (List[float], list[float], True, True),
        (List[str], list[str], True, True),
        (List[bool], list[bool], True, True),
        (List[DummyClass], list[DummyClass], True, True),
        (List[Optional[int]], list[int | None], True, True),
        (List[Optional[float]], list[float | None], True, True),
        (List[Optional[str]], list[str | None], True, True),
        (List[Optional[bool]], list[bool | None], True, True),
        (List[Optional[DummyClass]], list[DummyClass | None], True, True),
        (dict[int, int], dict[int, int], True, True),
        (dict[float, float], dict[float, float], True, True),
        (dict[str, str], dict[str, str], True, True),
        (dict[bool, bool], dict[bool, bool], True, True),
        (dict[DummyClass, DummyClass], dict[DummyClass, DummyClass], True, True),
        (Dict[int, int], dict[int, int], True, True),
        (Dict[float, float], dict[float, float], True, True),
        (Dict[str, str], dict[str, str], True, True),
        (Dict[bool, bool], dict[bool, bool], True, True),
        (Dict[DummyClass, DummyClass], dict[DummyClass, DummyClass], True, True),
        (Dict[int, List[Optional[int]]], dict[int, list[int | None]], True, True),
        (Dict[float, List[Optional[float]]], dict[float, list[float | None]], True, True),
        (Dict[str, List[Optional[str]]], dict[str, list[str | None]], True, True),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool | None]], True, True),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass | None]], True, True),

        (int, int | None, True, False),
        (float, float | None, True, False),
        (str, str | None, True, False),
        (bool, bool | None, True, False),
        (DummyClass, DummyClass | None, True, False),
        (int, Optional[int], True, False),
        (float, Optional[float], True, False),
        (str, Optional[str], True, False),
        (bool, Optional[bool], True, False),
        (DummyClass, Optional[DummyClass], True, False),

        (int, int | None, False, True),
        (float, float | None, False, True),
        (str, str | None, False, True),
        (bool, bool | None, False, True),
        (DummyClass, DummyClass | None, False, True),
        (int, Optional[int], False, True),
        (float, Optional[float], False, True),
        (str, Optional[str], False, True),
        (bool, Optional[bool], False, True),
        (DummyClass, Optional[DummyClass], False, True),

        (List[int], List[DummyClass], True, False),
        (List[float], List[DummyClass], True, False),
        (List[str], List[DummyClass], True, False),
        (List[bool], List[DummyClass], True, False),
        (List[int], list[DummyClass], True, False),
        (List[float], list[DummyClass], True, False),
        (List[str], list[DummyClass], True, False),
        (List[bool], list[DummyClass], True, False),

        (List[Optional[int]], list[int], True, False),
        (List[Optional[float]], list[float], True, False),
        (List[Optional[str]], list[str], True, False),
        (List[Optional[bool]], list[bool], True, False),
        (List[Optional[DummyClass]], list[DummyClass], True, False),
        (List[Optional[int]], list[int] | None, True, False),
        (List[Optional[float]], list[float] | None, True, False),
        (List[Optional[str]], list[str] | None, True, False),
        (List[Optional[bool]], list[bool] | None, True, False),
        (List[Optional[DummyClass]], list[DummyClass] | None, True, False),

        (List[Optional[int]], list[int], False, True),
        (List[Optional[float]], list[float], False, True),
        (List[Optional[str]], list[str], False, True),
        (List[Optional[bool]], list[bool], False, True),
        (List[Optional[DummyClass]], list[DummyClass], False, True),
        (List[Optional[int]], list[int] | None, False, True),
        (List[Optional[float]], list[float] | None, False, True),
        (List[Optional[str]], list[str] | None, False, True),
        (List[Optional[bool]], list[bool] | None, False, True),
        (List[Optional[DummyClass]], list[DummyClass] | None, False, True),

        (Dict[int, int], Dict[int, DummyClass], True, False),
        (Dict[float, float], Dict[float, DummyClass], True, False),
        (Dict[str, str], Dict[str, DummyClass], True, False),
        (Dict[bool, bool], Dict[bool, DummyClass], True, False),
        (Dict[int, int], Dict[DummyClass, int], True, False),
        (Dict[float, float], Dict[DummyClass, float], True, False),
        (Dict[str, str], Dict[DummyClass, str], True, False),
        (Dict[bool, bool], Dict[DummyClass, bool], True, False),
        (Dict[int, int], dict[int, DummyClass], True, False),
        (Dict[float, float], dict[float, DummyClass], True, False),
        (Dict[str, str], dict[str, DummyClass], True, False),
        (Dict[bool, bool], dict[bool, DummyClass], True, False),
        (Dict[int, int], dict[DummyClass, int], True, False),
        (Dict[float, float], dict[DummyClass, float], True, False),
        (Dict[str, str], dict[DummyClass, str], True, False),
        (Dict[bool, bool], dict[DummyClass, bool], True, False),

        (Dict[int, List[Optional[int]]], dict[int, list[int]], True, False),
        (Dict[float, List[Optional[float]]], dict[float, list[float]], True, False),
        (Dict[str, List[Optional[str]]], dict[str, list[str]], True, False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]], True, False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]], True, False),
        (Dict[int, List[Optional[int]]], dict[int, list[int] | None], True, False),
        (Dict[float, List[Optional[float]]], dict[float, list[float] | None], True, False),
        (Dict[str, List[Optional[str]]], dict[str, list[str] | None], True, False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool] | None], True, False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass] | None], True, False),
        (Dict[int, List[Optional[int]]], dict[int, list[int]] | None, True, False),
        (Dict[float, List[Optional[float]]], dict[float, list[float]] | None, True, False),
        (Dict[str, List[Optional[str]]], dict[str, list[str]] | None, True, False),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]] | None, True, False),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]] | None, True, False),

        (Dict[int, List[Optional[int]]], dict[int, list[int]], False, True),
        (Dict[float, List[Optional[float]]], dict[float, list[float]], False, True),
        (Dict[str, List[Optional[str]]], dict[str, list[str]], False, True),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]], False, True),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]], False, True),
        (Dict[int, List[Optional[int]]], dict[int, list[int] | None], False, True),
        (Dict[float, List[Optional[float]]], dict[float, list[float] | None], False, True),
        (Dict[str, List[Optional[str]]], dict[str, list[str] | None], False, True),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool] | None], False, True),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass] | None], False, True),
        (Dict[int, List[Optional[int]]], dict[int, list[int]] | None, False, True),
        (Dict[float, List[Optional[float]]], dict[float, list[float]] | None, False, True),
        (Dict[str, List[Optional[str]]], dict[str, list[str]] | None, False, True),
        (Dict[bool, List[Optional[bool]]], dict[bool, list[bool]] | None, False, True),
        (Dict[DummyClass, List[Optional[DummyClass]]], dict[DummyClass, list[DummyClass]] | None, False, True),

        (int | str | list[int | str], list[str | int] | str | int, False, True),
        (int | str | list[int | str], list[str | int] | str | int, True, True),
        (None | int | str | list[None | int | str], list[str | int | None] | str | int | None, False, True),
        (None | int | str | list[None | int | str], list[str | int | None] | str | int | None, True, True),
        (None | int | str | list[None | int | str], list[str | int] | str | int, False, True),
        (None | int | str | list[None | int | str], list[str | int] | str | int, True, False),
        (
            Callable[[int | str | list[bool | float]], bool | list[bool | list[float]]],
            Callable[[list[float | bool] | str | int], list[list[float] | bool] | bool],
            False,
            True,
        ),
        (
            Callable[[int | str | list[bool | float]], bool | list[bool | list[float]]],
            Callable[[list[float | bool] | str | int], list[list[float] | bool] | bool],
            True,
            True,
        ),
        (
            Callable[
                [None | int | str | list[None | bool | float]],
                None | bool | list[None | bool | list[None | float]],
            ],
            Callable[
                [list[float | bool | None] | str | int | None],
                list[list[float | None] | bool | None] | bool | None,
            ],
            False,
            True,
        ),
        (
            Callable[
                [None | int | str | list[None | bool | float]],
                None | bool | list[None | bool | list[None | float]],
            ],
            Callable[
                [list[float | bool | None] | str | int | None],
                list[list[float | None] | bool | None] | bool | None,
            ],
            True,
            True,
        ),
        (
            Callable[
                [None | int | str | list[None | bool | float]],
                None | bool | list[None | bool | list[None | float]],
            ],
            Callable[[list[float | bool] | str | int], list[list[float] | bool] | bool],
            False,
            True,
        ),
        (
            Callable[
                [None | int | str | list[None | bool | float]],
                None | bool | list[None | bool | list[None | float]],
            ],
            Callable[[list[float | bool] | str | int], list[list[float] | bool] | bool],
            True,
            False,
        ),
    ]

    @ddt.idata(data_is_identical_type)
    @ddt.unpack
    def test_is_identical_type(self, a, b, strict_optional, expect):
        self.assertEqual(is_identical_type(a, b, strict_optional=strict_optional), expect)
        self.assertEqual(is_identical_type(b, a, strict_optional=strict_optional), expect)

    data_is_identical_type__variant = [
        (int, int, False, False, True),
        (int, int, True, False, True),
        (int, int, False, True, True),
        (int, int, True, True, True),

        (float, float, False, False, True),
        (float, float, True, False, True),
        (float, float, False, True, True),
        (float, float, True, True, True),

        (int, SupportsInt, False, False, False),
        (int, SupportsInt, True, False, True),
        (int, SupportsInt, False, True, False),
        (int, SupportsInt, True, True, True),

        (float, SupportsFloat, False, False, False),
        (float, SupportsFloat, True, False, True),
        (float, SupportsFloat, False, True, False),
        (float, SupportsFloat, True, True, True),

        (list[int], list[int], False, False, True),
        (list[int], list[int], True, False, True),
        (list[int], list[int], False, True, True),
        (list[int], list[int], True, True, True),
        (List[int], list[int], False, False, True),
        (List[int], list[int], True, False, True),
        (List[int], list[int], False, True, True),
        (List[int], list[int], True, True, True),

        (dict[int, float], dict[int, float], False, False, True),
        (dict[int, float], dict[int, float], True, False, True),
        (dict[int, float], dict[int, float], False, True, True),
        (dict[int, float], dict[int, float], True, True, True),
        (Dict[int, float], dict[int, float], False, False, True),
        (Dict[int, float], dict[int, float], True, False, True),
        (Dict[int, float], dict[int, float], False, True, True),
        (Dict[int, float], dict[int, float], True, True, True),

        (list[int], Sequence[int], False, False, False),
        (list[int], Sequence[int], True, False, True),
        (list[int], Sequence[int], False, True, False),
        (list[int], Sequence[int], True, True, True),
        (list[int], Sequence[SupportsInt], False, False, False),
        (list[int], Sequence[SupportsInt], True, False, True),
        (list[int], Sequence[SupportsInt], False, True, False),
        (list[int], Sequence[SupportsInt], True, True, True),
        (list[SupportsInt], Sequence[int], False, False, False),
        (list[SupportsInt], Sequence[int], True, False, False),
        (list[SupportsInt], Sequence[int], False, True, False),
        (list[SupportsInt], Sequence[int], True, True, True),

        (dict[int, float], Mapping[int, float], False, False, False),
        (dict[int, float], Mapping[int, float], True, False, True),
        (dict[int, float], Mapping[int, float], False, True, False),
        (dict[int, float], Mapping[int, float], True, True, True),
        (dict[int, float], Mapping[SupportsInt, SupportsFloat], False, False, False),
        (dict[int, float], Mapping[SupportsInt, SupportsFloat], True, False, True),
        (dict[int, float], Mapping[SupportsInt, SupportsFloat], False, True, False),
        (dict[int, float], Mapping[SupportsInt, SupportsFloat], True, True, True),
        (dict[SupportsInt, SupportsFloat], Mapping[int, float], False, False, False),
        (dict[SupportsInt, SupportsFloat], Mapping[int, float], True, False, False),
        (dict[SupportsInt, SupportsFloat], Mapping[int, float], False, True, False),
        (dict[SupportsInt, SupportsFloat], Mapping[int, float], True, True, True),

        (list[DummySubClass], Sequence[DummySubClass], False, False, False),
        (list[DummySubClass], Sequence[DummySubClass], True, False, True),
        (list[DummySubClass], Sequence[DummySubClass], False, True, False),
        (list[DummySubClass], Sequence[DummySubClass], True, True, True),
        (list[DummySubClass], Sequence[DummyClass], False, False, False),
        (list[DummySubClass], Sequence[DummyClass], True, False, True),
        (list[DummySubClass], Sequence[DummyClass], False, True, False),
        (list[DummySubClass], Sequence[DummyClass], True, True, True),
        (list[DummyClass], Sequence[DummySubClass], False, False, False),
        (list[DummyClass], Sequence[DummySubClass], True, False, False),
        (list[DummyClass], Sequence[DummySubClass], False, True, False),
        (list[DummyClass], Sequence[DummySubClass], True, True, True),

        (dict[int, DummySubClass], Mapping[int, DummySubClass], False, False, False),
        (dict[int, DummySubClass], Mapping[int, DummySubClass], True, False, True),
        (dict[int, DummySubClass], Mapping[int, DummySubClass], False, True, False),
        (dict[int, DummySubClass], Mapping[int, DummySubClass], True, True, True),
        (dict[int, DummySubClass], Mapping[SupportsInt, DummyClass], False, False, False),
        (dict[int, DummySubClass], Mapping[SupportsInt, DummyClass], True, False, True),
        (dict[int, DummySubClass], Mapping[SupportsInt, DummyClass], False, True, False),
        (dict[int, DummySubClass], Mapping[SupportsInt, DummyClass], True, True, True),
        (dict[SupportsInt, DummyClass], Mapping[int, DummySubClass], False, False, False),
        (dict[SupportsInt, DummyClass], Mapping[int, DummySubClass], True, False, False),
        (dict[SupportsInt, DummyClass], Mapping[int, DummySubClass], False, True, False),
        (dict[SupportsInt, DummyClass], Mapping[int, DummySubClass], True, True, True),
    ]

    @ddt.idata(data_is_identical_type__variant)
    @ddt.unpack
    def test_is_identical_type__variant(self, a, b, covariant, contravariant, expect):
        self.assertEqual(is_identical_type(a,
                                           b,
                                           strict_optional=False,
                                           covariant=covariant,
                                           contravariant=contravariant),
                         expect)
        self.assertEqual(is_identical_type(a,
                                           b,
                                           strict_optional=True,
                                           covariant=covariant,
                                           contravariant=contravariant),
                         expect)

        self.assertEqual(is_identical_type(b,
                                           a,
                                           strict_optional=False,
                                           covariant=contravariant,
                                           contravariant=covariant),
                         is_identical_type(a,
                                           b,
                                           strict_optional=True,
                                           covariant=covariant,
                                           contravariant=contravariant))
