"""
🎯 FOIL QUINLAN 1990 IMPLEMENTATION
============================================================

FOIL implementation following Quinlan (1990) specifications.

🧠 Inductive Logic Programming Library - Made possible by Benedict Chen
   benedict@benedictchen.com
   Support his work: 🍺 Buy him a beer: https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=WXQKYYKPHWXHS
   💖 Sponsor: https://github.com/sponsors/benedictchen

📚 Research Foundation:
- Quinlan, J.R. (1990). "Learning logical definitions from relations." 
  Machine Learning, 5(3), 239-266.
- Follows Section 3.2 information gain formula and variable binding framework

🎯 ELI5 Explanation:
This is like having a "reference implementation" of FOIL - a version that
follows the original 1990 research paper exactly, with no shortcuts or approximations.

Think of it as the "textbook solution" that researchers can use to verify
their results match Quinlan's original algorithm specifications.

🔧 Key Features:
• Exact Quinlan (1990) formula implementation
• Variable binding generation from Quinlan's θ-substitution framework
• Proper SLD resolution for coverage testing
• Mathematical correctness over performance optimization

⚙️ Use Cases:
• Research validation and benchmarking
• Reference implementation for comparison
• Teaching exact algorithm specifications
• Academic paper reproduction

🙏 Support This Work:
If this research-accurate FOIL helped validate your academic work, please consider:
🍺 Buy Benedict a beer: https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=WXQKYYKPHWXHS
💖 GitHub Sponsor: https://github.com/sponsors/benedictchen

Your support enables continued development of research-grade ILP implementations!
"""

import numpy as np
from typing import Dict, List, Tuple, Optional, Set, Any, Union
from dataclasses import dataclass
import logging
from itertools import product, combinations
import math
from collections import defaultdict
import time

from .foil_comprehensive_config import (
    FOILComprehensiveConfig, 
    InformationGainMethod,
    VariableBindingStrategy,
    CoverageTestingMethod,
    create_research_accurate_config
)
from .ilp_core import (
    LogicalTerm, LogicalAtom, LogicalClause, Example,
    InductiveLogicProgrammer
)


@dataclass
class VariableBinding:
    """
    Represents a variable binding (instantiation) as used in Quinlan (1990).
    
    FOIL operates on variable bindings θ = {X₁/a₁, X₂/a₂, ...}, not just examples.
    This is the key correction from the simplified implementation.
    """
    substitution: Dict[str, str]  # {variable_name: constant_value}
    is_positive: bool  # Whether this binding satisfies positive target
    satisfies_clause: bool = False  # Whether binding satisfies current partial clause
    example_source: Optional[Example] = None  # Source example for this binding
    
    def apply_to_atom(self, atom: LogicalAtom) -> LogicalAtom:
        """Apply this binding to an atom."""
        new_terms = []
        for term in atom.terms:
            if term.term_type == 'variable' and term.name in self.substitution:
                new_terms.append(LogicalTerm(name=self.substitution[term.name], term_type='constant'))
            else:
                new_terms.append(term)
        return LogicalAtom(predicate=atom.predicate, terms=new_terms, negated=atom.negated)


class ResearchAccurateFOILLearner:
    """
    Research-accurate FOIL implementation based on Quinlan (1990).
    
    This class provides comprehensive configuration options allowing users to 
    choose between different research-based approaches for each component.
    """
    
    def __init__(self, config: Optional[FOILComprehensiveConfig] = None):
        """
        Initialize research-accurate FOIL learner.
        
        Args:
            config: Comprehensive configuration for all FOIL components
        """
        self.config = config or create_research_accurate_config()
        
        # Learning state
        self.background_knowledge = []
        self.positive_examples = []
        self.negative_examples = []
        self.learned_clauses = []
        self.mode_declarations = {}  # predicate -> List[mode_spec]
        
        # Vocabulary
        self.predicates = set()
        self.constants = set()
        self.functions = set()
        
        # Performance tracking
        self.learning_statistics = {
            'bindings_generated': 0,
            'coverage_tests': 0,
            'information_gain_calculations': 0,
            'sld_resolution_steps': 0,
            'clauses_pruned': 0
        }
        
        # Configure logging based on settings
        logging.basicConfig(
            level=logging.DEBUG if self.config.verbose_output else logging.INFO
        )
        self.logger = logging.getLogger(__name__)
        
        print(f"🧠 Research-Accurate FOIL Learner Initialized")
        print(f"   Information Gain: {self.config.information_gain_method.value}")
        print(f"   Binding Generation: {self.config.binding_generation_method.value}")
        print(f"   Coverage Testing: {self.config.coverage_testing_method.value}")
    
    def add_mode_declaration(self, predicate: str, mode_spec: str):
        """
        Add mode declaration as in original Quinlan (1990) FOIL.
        
        Args:
            predicate: Predicate name
            mode_spec: Mode specification (e.g., "+person, -person" for parent/2)
        """
        if predicate not in self.mode_declarations:
            self.mode_declarations[predicate] = []
        self.mode_declarations[predicate].append(mode_spec)
        
        self.logger.info(f"Added mode declaration: {predicate}({mode_spec})")
    
    def learn_rules(self, target_predicate: str) -> List[LogicalClause]:
        """
        Learn rules using research-accurate FOIL algorithm.
        
        Args:
            target_predicate: Target predicate to learn rules for
            
        Returns:
            List of learned clauses
        """
        self.logger.info(f"🧠 Learning rules for {target_predicate} using research-accurate FOIL")
        
        # Get examples for target predicate
        pos_examples = [ex for ex in self.positive_examples 
                       if ex.atom.predicate == target_predicate]
        neg_examples = [ex for ex in self.negative_examples 
                       if ex.atom.predicate == target_predicate]
        
        if not pos_examples:
            self.logger.warning(f"No positive examples for {target_predicate}")
            return []
        
        learned_rules = []
        uncovered_positive = pos_examples.copy()
        
        # FOIL covering loop
        while uncovered_positive:
            self.logger.info(f"Learning clause to cover {len(uncovered_positive)} remaining examples")
            
            clause = self._learn_single_clause_research_accurate(
                target_predicate, uncovered_positive, neg_examples
            )
            
            if clause is None:
                self.logger.info("No more useful clauses can be learned")
                break
            
            learned_rules.append(clause)
            
            # Remove covered examples using research-accurate coverage testing
            newly_covered = self._get_covered_examples_research_accurate(clause, uncovered_positive)
            uncovered_positive = [ex for ex in uncovered_positive if ex not in newly_covered]
            
            self.logger.info(f"Clause covers {len(newly_covered)} positive examples")
        
        self.logger.info(f"✅ Learned {len(learned_rules)} research-accurate rules")
        return learned_rules
    
    def _learn_single_clause_research_accurate(self, 
                                             target_predicate: str,
                                             pos_examples: List[Example],
                                             neg_examples: List[Example]) -> Optional[LogicalClause]:
        """
        Learn single clause using research-accurate methods.
        
        This implements research-based methods for proper clause learning.
        """
        # Initialize clause with proper variable structure
        head_arity = self._get_predicate_arity(target_predicate)
        head_vars = [LogicalTerm(name=f"X{i}", term_type='variable') for i in range(head_arity)]
        head = LogicalAtom(predicate=target_predicate, terms=head_vars)
        
        current_clause = LogicalClause(head=head, body=[])
        
        # Generate initial variable bindings (SOLUTION 1: Variable Bindings)
        current_bindings = self._generate_variable_bindings(
            current_clause, pos_examples + neg_examples
        )
        
        self.logger.info(f"Generated {len(current_bindings)} initial variable bindings")
        
        # Greedy literal selection with research-accurate information gain
        for depth in range(self.config.max_variables_per_clause):
            # Generate candidate literals using selected strategy
            candidate_literals = self._generate_candidate_literals_research_accurate(
                current_clause, current_bindings
            )
            
            if not candidate_literals:
                break
            
            # Select best literal using chosen information gain method
            best_literal, best_gain = self._select_best_literal_research_accurate(
                candidate_literals, current_clause, current_bindings
            )
            
            if best_literal is None:
                break
            
            # Add literal and update bindings
            current_clause.body.append(best_literal)
            new_bindings = self._filter_bindings_by_literal(current_bindings, best_literal)
            
            self.logger.info(f"Added literal: {best_literal}, Gain: {best_gain:.4f}")
            self.logger.info(f"Bindings: {len(current_bindings)} -> {len(new_bindings)}")
            
            current_bindings = new_bindings
            
            # Stop if no negative bindings remain
            negative_bindings = [b for b in current_bindings if not b.is_positive]
            if not negative_bindings:
                break
        
        # Apply pruning based on selected strategy
        if self._should_prune_clause(current_clause, current_bindings):
            self.logger.info("Clause pruned by complexity control")
            return None
        
        return current_clause
    
    def _generate_variable_bindings(self, 
                                  clause: LogicalClause,
                                  examples: List[Example]) -> List[VariableBinding]:
        """
        SOLUTION 1: Generate proper variable bindings as in Quinlan (1990).
        
        This corrects the simplified implementation that confused examples with bindings.
        """
        self.learning_statistics['bindings_generated'] += 1
        
        if self.config.binding_strategy == VariableBindingStrategy.EXHAUSTIVE_ENUMERATION:
            return self._exhaustive_binding_enumeration(clause, examples)
        elif self.config.binding_strategy == VariableBindingStrategy.CONSTRAINT_GUIDED:
            return self._constraint_guided_binding_generation(clause, examples)
        elif self.config.binding_strategy == VariableBindingStrategy.HEURISTIC_PRUNING:
            return self._sampling_based_binding_generation(clause, examples)
        else:  # HYBRID
            return self._hybrid_binding_generation(clause, examples)
    
    def _exhaustive_binding_enumeration(self, 
                                      clause: LogicalClause, 
                                      examples: List[Example]) -> List[VariableBinding]:
        """Exhaustive enumeration of all possible variable bindings."""
        bindings = []
        
        # Extract variables from clause
        variables = set()
        for atom in [clause.head] + clause.body:
            for term in atom.terms:
                if term.term_type == 'variable':
                    variables.add(term.name)
        
        variables = list(variables)
        
        # For each example, generate all possible bindings
        for example in examples:
            # Create binding from example constants
            example_constants = [term.name for term in example.atom.terms 
                               if term.term_type == 'constant']
            
            # Generate all combinations of constants for variables
            if len(variables) <= len(example_constants):
                for const_assignment in combinations(example_constants, len(variables)):
                    substitution = dict(zip(variables, const_assignment))
                    
                    binding = VariableBinding(
                        substitution=substitution,
                        is_positive=example.is_positive,
                        example_source=example
                    )
                    
                    # Check if binding is consistent with clause
                    if self._is_binding_consistent(binding, clause):
                        bindings.append(binding)
                    
                    # Limit explosion
                    if len(bindings) >= self.config.binding_enumeration_limit:
                        self.logger.warning(f"Hit binding enumeration limit: {self.config.binding_enumeration_limit}")
                        return bindings
        
        return bindings
    
    def _constraint_guided_binding_generation(self, 
                                            clause: LogicalClause,
                                            examples: List[Example]) -> List[VariableBinding]:
        """Generate bindings using mode declarations and constraints."""
        bindings = []
        
        # Use mode declarations to guide binding generation
        if not self.mode_declarations:
            self.logger.warning("No mode declarations - falling back to exhaustive enumeration")
            return self._exhaustive_binding_enumeration(clause, examples)
        
        # Extract head variables and their modes
        head_predicate = clause.head.predicate
        if head_predicate not in self.mode_declarations:
            return self._exhaustive_binding_enumeration(clause, examples)
        
        for example in examples:
            if example.atom.predicate != head_predicate:
                continue
            
            # Create binding from example using mode information
            for mode_spec in self.mode_declarations[head_predicate]:
                modes = mode_spec.split(', ')
                if len(modes) != len(example.atom.terms):
                    continue
                
                substitution = {}
                for i, (mode, term) in enumerate(zip(modes, example.atom.terms)):
                    var_name = f"X{i}"
                    if mode.startswith('+'):  # Input mode - use constant from example
                        if term.term_type == 'constant':
                            substitution[var_name] = term.name
                    elif mode.startswith('-'):  # Output mode - can be any constant
                        substitution[var_name] = term.name if term.term_type == 'constant' else f"_G{i}"
                    # Constant mode (#) requires exact match - handle separately
                
                if substitution:
                    binding = VariableBinding(
                        substitution=substitution,
                        is_positive=example.is_positive,
                        example_source=example
                    )
                    bindings.append(binding)
        
        return bindings[:self.config.binding_enumeration_limit]
    
    def _calculate_information_gain_quinlan_original(self,
                                                   literal: LogicalAtom,
                                                   current_clause: LogicalClause,
                                                   bindings_before: List[VariableBinding],
                                                   bindings_after: List[VariableBinding]) -> float:
        """
        SOLUTION 2: Quinlan's exact information gain formula from 1990 paper.
        
        Gain = t × (log(p₁/(p₁+n₁)) - log(p₀/(p₀+n₀)))
        
        Where:
        - t = number of positive bindings that extend the rule
        - p₀,n₀ = positive/negative bindings before adding literal
        - p₁,n₁ = positive/negative bindings after adding literal
        
        CRITICAL: Uses variable bindings, not examples!
        """
        self.learning_statistics['information_gain_calculations'] += 1
        
        # Count bindings (not examples!)
        p0 = len([b for b in bindings_before if b.is_positive])
        n0 = len([b for b in bindings_before if not b.is_positive])
        p1 = len([b for b in bindings_after if b.is_positive])
        n1 = len([b for b in bindings_after if not b.is_positive])
        
        # t parameter: positive bindings that extend the rule
        if self.config.t_parameter_calculation == "positive_bindings":
            t = p1
        elif self.config.t_parameter_calculation == "all_bindings":
            t = p1 + n1
        else:  # "weighted"
            t = p1 * math.log(max(1, len(bindings_after) / max(1, len(bindings_before))))
        
        # Handle edge cases
        if p0 == 0 or p1 == 0 or (p0 + n0) == 0 or (p1 + n1) == 0:
            return 0.0
        
        # Apply Laplace correction if enabled
        if self.config.use_laplace_correction:
            alpha = self.config.laplace_alpha
            p0_corrected = p0 + alpha
            n0_corrected = n0 + alpha  
            p1_corrected = p1 + alpha
            n1_corrected = n1 + alpha
        else:
            p0_corrected, n0_corrected = p0, n0
            p1_corrected, n1_corrected = p1, n1
        
        # Compute information gain with selected logarithm base
        if self.config.logarithm_base == "natural":
            log_func = math.log
        elif self.config.logarithm_base == "base2": 
            log_func = math.log2
        else:  # base10
            log_func = math.log10
        
        try:
            gain = t * (
                log_func(p1_corrected / (p1_corrected + n1_corrected)) - 
                log_func(p0_corrected / (p0_corrected + n0_corrected))
            )
        except (ValueError, ZeroDivisionError):
            if self.config.handle_zero_probabilities:
                gain = 0.0
            else:
                raise
        
        if self.config.log_information_gain_details:
            self.logger.debug(f"Information gain calculation:")
            self.logger.debug(f"  Before: p0={p0}, n0={n0}")
            self.logger.debug(f"  After: p1={p1}, n1={n1}")
            self.logger.debug(f"  t parameter: {t}")
            self.logger.debug(f"  Gain: {gain:.6f}")
        
        return gain
    
    def _coverage_test_sld_resolution(self, 
                                    clause: LogicalClause, 
                                    example: Example) -> bool:
        """
        SOLUTION 3: Proper SLD resolution for coverage testing.
        
        This replaces the simplified implementation with proper theorem proving.
        """
        self.learning_statistics['coverage_tests'] += 1
        self.learning_statistics['sld_resolution_steps'] += 1
        
        # Initialize SLD resolution
        goals = [example.atom]
        substitution = {}
        
        for step in range(self.config.sld_resolution_max_steps):
            if not goals:
                return True  # Success - all goals resolved
            
            # Select goal based on selection rule
            if self.config.sld_selection_rule == "leftmost":
                current_goal = goals.pop(0)
            else:  # random or heuristic
                current_goal = goals.pop(np.random.randint(len(goals)))
            
            # Try to unify with clause head
            unification = {}
            if self._unify_atoms(current_goal, clause.head, unification):
                # Apply substitution and add body goals
                new_goals = []
                for body_literal in clause.body:
                    new_goal = self._apply_substitution(body_literal, unification)
                    new_goals.append(new_goal)
                
                goals = new_goals + goals
                substitution.update(unification)
                continue
            
            # Try background knowledge
            resolved = False
            for bg_clause in self.background_knowledge:
                unification = {}
                if self._unify_atoms(current_goal, bg_clause.head, unification):
                    # Apply substitution and add body goals
                    new_goals = []
                    for body_literal in bg_clause.body:
                        new_goal = self._apply_substitution(body_literal, unification)
                        new_goals.append(new_goal)
                    
                    goals = new_goals + goals
                    substitution.update(unification)
                    resolved = True
                    break
            
            if not resolved:
                return False  # Failure - no clause can resolve current goal
        
        return False  # Timeout
    
    def _unify_atoms(self, 
                    atom1: LogicalAtom, 
                    atom2: LogicalAtom, 
                    substitution: Dict[str, str]) -> bool:
        """Unification algorithm for atoms."""
        if atom1.predicate != atom2.predicate or len(atom1.terms) != len(atom2.terms):
            return False
        
        local_subst = substitution.copy()
        
        for term1, term2 in zip(atom1.terms, atom2.terms):
            if not self._unify_terms(term1, term2, local_subst):
                return False
        
        substitution.update(local_subst)
        return True
    
    def _unify_terms(self, 
                    term1: LogicalTerm, 
                    term2: LogicalTerm, 
                    substitution: Dict[str, str]) -> bool:
        """Unify two logical terms."""
        # Dereference variables
        while term1.term_type == 'variable' and term1.name in substitution:
            term1 = LogicalTerm(name=substitution[term1.name], term_type='constant')
        
        while term2.term_type == 'variable' and term2.name in substitution:
            term2 = LogicalTerm(name=substitution[term2.name], term_type='constant')
        
        # Unification cases
        if term1.term_type == 'constant' and term2.term_type == 'constant':
            return term1.name == term2.name
        elif term1.term_type == 'variable':
            if self.config.enable_occurs_check and self._occurs_check(term1.name, term2, substitution):
                return False
            substitution[term1.name] = term2.name
            return True
        elif term2.term_type == 'variable':
            if self.config.enable_occurs_check and self._occurs_check(term2.name, term1, substitution):
                return False
            substitution[term2.name] = term1.name
            return True
        
        return False
    
    def _occurs_check(self, var: str, term: LogicalTerm, substitution: Dict[str, str]) -> bool:
        """Occurs check to prevent infinite structures."""
        if term.term_type == 'variable':
            if var == term.name:
                return True
            if term.name in substitution:
                return self._occurs_check(var, LogicalTerm(substitution[term.name], 'constant'), substitution)
        return False
    
    def _apply_substitution(self, atom: LogicalAtom, substitution: Dict[str, str]) -> LogicalAtom:
        """Apply variable substitution to atom."""
        new_terms = []
        for term in atom.terms:
            if term.term_type == 'variable' and term.name in substitution:
                new_terms.append(LogicalTerm(name=substitution[term.name], term_type='constant'))
            else:
                new_terms.append(term)
        return LogicalAtom(predicate=atom.predicate, terms=new_terms, negated=atom.negated)
    
    # Additional helper methods...
    def _get_predicate_arity(self, predicate: str) -> int:
        """Get arity of predicate from examples."""
        for example in self.positive_examples + self.negative_examples:
            if example.atom.predicate == predicate:
                return len(example.atom.terms)
        return 2  # Default arity
    
    def print_learning_statistics(self):
        """Print comprehensive learning statistics."""
        print("\n📊 Research-Accurate FOIL Learning Statistics:")
        for key, value in self.learning_statistics.items():
            print(f"   {key.replace('_', ' ').title()}: {value}")


# Factory function for easy usage
def create_research_accurate_foil(config: Optional[FOILComprehensiveConfig] = None) -> ResearchAccurateFOILLearner:
    """
    Create a research-accurate FOIL learner with comprehensive configuration.
    
    Args:
        config: Optional configuration (defaults to research-accurate)
        
    Returns:
        ResearchAccurateFOILLearner: Configured learner
    """
    return ResearchAccurateFOILLearner(config or create_research_accurate_config())