"""
🧠 HYPOTHESIS GENERATION MODULE - Core ILP Hypothesis Space Search
================================================================

This module implements the fundamental hypothesis generation mechanisms for
Inductive Logic Programming, following Muggleton & De Raedt's (1994) theoretical
framework for systematic hypothesis space exploration.

Based on: Muggleton & De Raedt (1994) "Inductive Logic Programming: Theory and Methods"

🔬 Theoretical Foundation:
The core insight of ILP is that learning logical rules can be framed as a search
through the space of possible logical hypotheses, guided by coverage of positive
examples and avoidance of negative examples. This module implements:

1. **Initial Hypothesis Generation**: Creating candidate hypotheses from background knowledge
2. **Pattern Extraction**: Converting examples into generalizable patterns
3. **Unification**: Robinson's algorithm for logical term matching
4. **Candidate Generation**: Systematic exploration of rule combinations

🎯 Key Innovation:
Rather than learning from raw statistical patterns, ILP generates interpretable
logical rules by combining symbolic reasoning with empirical evidence. This bridges
the gap between symbolic AI and machine learning.

🏗️ Architecture:
- Pattern-based hypothesis seeding from positive examples
- Background knowledge integration through unification
- Systematic candidate clause generation
- Robinson's unification algorithm for term matching

Mathematical Framework:
- Hypothesis Space: H (set of possible logical clauses)  
- Background Knowledge: B (known facts and rules)
- Examples: E+ (positive) and E- (negative)
- Goal: Find H such that B ∧ H ⊨ E+ and B ∧ H ∧ E- ⊭ ⊥

Author: Benedict Chen
"""

from typing import Dict, List, Optional, Set, Tuple, Union
from .logical_structures import (
    LogicalTerm, LogicalAtom, LogicalClause, Example,
    create_variable, create_constant
)


class HypothesisGenerationMixin:
    """
    🧠 Mixin for ILP Hypothesis Generation Operations
    
    This mixin provides the core hypothesis generation functionality for Inductive
    Logic Programming systems. It implements the fundamental mechanisms for:
    
    1. **Initial Hypothesis Generation**: Systematic creation of candidate hypotheses
    2. **Pattern Extraction and Instantiation**: Converting examples to patterns
    3. **Unification**: Robinson's algorithm for logical term matching
    4. **Candidate Generation**: Background knowledge integration
    
    🔬 Theoretical Background:
    Based on Muggleton & De Raedt (1994), this implements the hypothesis generation
    phase of the ILP learning cycle. The key insight is that good hypotheses can be
    systematically generated by:
    
    - Extracting patterns from positive examples
    - Connecting these patterns to background knowledge through unification
    - Generating candidate clauses that respect logical constraints
    
    The hypothesis space is structured by logical relationships, allowing efficient
    search through semantically meaningful rule combinations rather than brute-force
    enumeration of all possible logical formulas.
    
    🎯 Usage Pattern:
    This mixin is designed to be used with an ILP system class that provides:
    - self.background_knowledge: List of background clauses
    - self.max_variables: Maximum variables per clause  
    - self.max_clause_length: Maximum clause complexity
    - self.learning_stats: Dictionary for tracking statistics
    - Various predicate compatibility methods
    
    Example Integration:
        class MyILPSystem(HypothesisGenerationMixin):
            def __init__(self):
                self.background_knowledge = []
                self.max_variables = 4
                # ... other initialization
                
            def learn_rules(self, target_predicate):
                hypotheses = self._generate_initial_hypotheses(
                    target_predicate, positive_examples
                )
                # ... continue with refinement
    """
    
    def _generate_initial_hypotheses(
        self, 
        target_predicate: str, 
        positive_examples: List[Example]
    ) -> List[LogicalClause]:
        """
        🔬 Generate Initial Hypothesis Clauses from Examples and Background Knowledge
        
        This is the first phase of ILP hypothesis generation, implementing Muggleton
        & De Raedt's approach to systematic hypothesis space seeding. Rather than
        random generation, this creates structured hypotheses by:
        
        1. **Pattern Extraction**: Converting positive examples into generalizable patterns
        2. **Background Integration**: Connecting patterns to background knowledge via unification  
        3. **Unit Clause Generation**: Creating simple facts as baseline hypotheses
        
        🎯 ELI5: Think of this as a detective looking at clues (positive examples) and
        connecting them to what they already know (background knowledge) to form
        initial theories about what rule might explain the evidence!
        
        🔬 Algorithm Details:
        The method implements a structured approach to hypothesis generation:
        
        **Phase 1: Pattern Extraction**
        - Extract variable patterns from each positive example
        - Convert constants to variables to enable generalization
        - Map specific instances to abstract templates
        
        **Phase 2: Background Knowledge Integration** 
        - For each background clause, attempt unification with extracted patterns
        - Generate candidate clauses by connecting target predicate to background
        - Use Robinson's unification algorithm for sound logical matching
        
        **Phase 3: Unit Clause Generation**
        - Create simple facts directly from patterns
        - Provide baseline hypotheses for cases with minimal background knowledge
        
        Args:
            target_predicate (str): The predicate to learn rules for
                                  Example: "father", "grandparent", "likes"
                                  Must appear in positive examples as head predicate
            positive_examples (List[Example]): Training examples where is_positive=True
                                             Used to extract patterns and guide generation
                                             Should contain diverse instances of target predicate
        
        Returns:
            List[LogicalClause]: Generated initial hypothesis clauses
                               Each clause attempts to explain some positive examples
                               Sorted by potential quality (simpler clauses first)
                               May include both facts and rules
        
        💡 Key Insight: Good initial hypotheses come from systematic combination of
        observed patterns with existing knowledge, not random generation!
        
        🔧 Implementation Notes:
        - Updates self.learning_stats['clauses_generated'] for tracking
        - Respects self.max_variables constraint during pattern processing
        - Uses background knowledge compatibility for sound hypothesis generation
        
        Example:
            >>> # Positive examples: father(john, mary), father(bob, alice)
            >>> # Background: parent(john, mary), male(john), parent(bob, alice), male(bob)
            >>> hypotheses = self._generate_initial_hypotheses("father", examples)
            >>> # Might generate: father(X,Y) :- parent(X,Y), male(X)
            
        ⚡ Complexity: O(|E+| × |B| × V^k) where |E+| = positive examples,
        |B| = background clauses, V = vocabulary size, k = max variables
        """
        hypotheses = []
        
        # Phase 1: Extract unique term patterns from positive examples
        example_patterns = []
        for example in positive_examples:
            pattern = self._extract_pattern(example.atom)
            if pattern not in example_patterns:  # Avoid duplicates
                example_patterns.append(pattern)
        
        # Phase 2: Generate clauses based on background knowledge integration
        for bg_clause in self.background_knowledge:
            # Try to connect background predicate to target through unification
            candidate_clauses = self._generate_candidate_clauses(
                target_predicate, bg_clause, example_patterns
            )
            hypotheses.extend(candidate_clauses)
        
        # Phase 3: Generate simple unit clauses (facts) as baseline hypotheses
        for pattern in example_patterns:
            unit_clause = LogicalClause(
                head=self._instantiate_pattern(target_predicate, pattern),
                body=[],
                confidence=1.0
            )
            hypotheses.append(unit_clause)
        
        # Update learning statistics
        if hasattr(self, 'learning_stats'):
            self.learning_stats['clauses_generated'] = len(hypotheses)
        
        return hypotheses
    
    def _extract_pattern(self, atom: LogicalAtom) -> List[str]:
        """
        🔍 Extract Generalizable Variable Pattern from Logical Atom
        
        This method converts a specific logical atom into an abstract pattern that
        can be used for generalization. It's a key step in ILP that enables learning
        general rules from specific examples.
        
        🎯 ELI5: Think of this as taking a specific sentence like "John loves Mary" 
        and turning it into a template like "Person1 loves Person2" so you can 
        recognize the same pattern in other examples!
        
        🔬 Algorithm Details:
        The pattern extraction process follows these steps:
        
        1. **Constant Preservation**: Constants remain as-is to preserve specific constraints
        2. **Variable Standardization**: Variables are mapped to standard form (V0, V1, V2...)
        3. **Consistent Mapping**: Same original variable always maps to same standard variable
        
        This creates patterns that can be:
        - Unified with background knowledge
        - Instantiated with different terms
        - Used for hypothesis generation
        
        Args:
            atom (LogicalAtom): The logical atom to extract pattern from
                              Example: father(john, mary) or loves(X, Y)
                              Can contain mix of constants and variables
        
        Returns:
            List[str]: Abstract pattern with standardized variables
                      Constants preserved as-is, variables as V0, V1, V2...
                      Order matches original atom's term order
        
        💡 Key Insight: Abstraction enables generalization while preserving
        essential structural information for unification!
        
        Examples:
            >>> atom = LogicalAtom("father", [LogicalTerm("john", "constant"), 
            ...                               LogicalTerm("mary", "constant")])
            >>> pattern = self._extract_pattern(atom)
            >>> # Result: ["john", "mary"] (constants preserved)
            
            >>> atom = LogicalAtom("loves", [LogicalTerm("X", "variable"),
            ...                              LogicalTerm("Y", "variable")])  
            >>> pattern = self._extract_pattern(atom)
            >>> # Result: ["V0", "V1"] (variables standardized)
            
            >>> atom = LogicalAtom("parent", [LogicalTerm("john", "constant"),
            ...                               LogicalTerm("X", "variable")])
            >>> pattern = self._extract_pattern(atom)  
            >>> # Result: ["john", "V0"] (mixed case)
        
        🔧 Technical Notes:
        - Variable mapping is consistent within single atom
        - Does not preserve variable names across different atoms
        - Pattern can be used with _instantiate_pattern() to recreate atoms
        """
        pattern = []
        var_map = {}  # Maps original variable names to standardized names
        var_counter = 0
        
        for term in atom.terms:
            if term.term_type == 'constant':
                # Constants remain as-is to preserve specific constraints
                pattern.append(term.name)
            elif term.term_type == 'variable':
                # Variables get mapped to standard form V0, V1, V2...
                if term.name not in var_map:
                    var_map[term.name] = f"V{var_counter}"
                    var_counter += 1
                pattern.append(var_map[term.name])
            else:  # function terms
                # For now, treat function terms as constants
                # More sophisticated implementation would handle function structure
                pattern.append(str(term))
        
        return pattern
    
    def _instantiate_pattern(self, predicate: str, pattern: List[str]) -> LogicalAtom:
        """
        🏗️ Create Logical Atom from Abstract Pattern
        
        This method reverses the pattern extraction process, creating a concrete
        logical atom from an abstract pattern. It's used to instantiate general
        patterns into specific hypothesis clauses.
        
        🎯 ELI5: This takes a template like "Person1 loves Person2" and turns it
        back into a concrete statement like "X loves Y" that can be used in rules!
        
        🔬 Algorithm Details:
        The instantiation process converts pattern elements back to logical terms:
        
        1. **Variable Detection**: Elements starting with 'V' become variables
        2. **Constant Creation**: Other elements become constants  
        3. **Atom Construction**: Combine predicate with terms to form atom
        
        This enables:
        - Converting extracted patterns back to usable logical forms
        - Creating hypothesis clauses with appropriate term structure
        - Maintaining logical validity for unification operations
        
        Args:
            predicate (str): The predicate name for the new atom
                           Example: "father", "loves", "grandparent"
                           Will become the functor of the created atom
            pattern (List[str]): Abstract pattern from _extract_pattern()
                               Elements are either constants or standardized variables
                               Order determines term positions in created atom
        
        Returns:
            LogicalAtom: Concrete logical atom created from pattern
                        Variables as LogicalTerm(name="V0", term_type='variable')
                        Constants as LogicalTerm(name="john", term_type='constant')
                        Ready for use in clauses and unification
        
        💡 Key Insight: Pattern instantiation preserves logical structure while
        enabling flexible predicate assignment for hypothesis construction!
        
        Examples:
            >>> pattern = ["john", "V0"]
            >>> atom = self._instantiate_pattern("father", pattern)
            >>> # Result: father(john, V0) - mixed constant/variable
            
            >>> pattern = ["V0", "V1"] 
            >>> atom = self._instantiate_pattern("loves", pattern)
            >>> # Result: loves(V0, V1) - pure variables
            
            >>> pattern = ["alice", "bob"]
            >>> atom = self._instantiate_pattern("parent", pattern) 
            >>> # Result: parent(alice, bob) - pure constants
        
        🔧 Technical Notes:
        - Automatically determines term types based on pattern format
        - V-prefixed strings become variables, others become constants
        - Created atoms are ready for unification and clause construction
        """
        terms = []
        
        for pattern_element in pattern:
            if pattern_element.startswith('V'):
                # Standardized variable - create variable term
                terms.append(LogicalTerm(name=pattern_element, term_type='variable'))
            else:
                # Everything else treated as constant
                terms.append(LogicalTerm(name=pattern_element, term_type='constant'))
        
        return LogicalAtom(predicate=predicate, terms=terms, negated=False)
    
    def _generate_candidate_clauses(
        self, 
        target_predicate: str, 
        bg_clause: LogicalClause, 
        patterns: List[List[str]]
    ) -> List[LogicalClause]:
        """
        🔧 Generate Candidate Clauses by Connecting Background Knowledge to Target
        
        This method implements the core mechanism for connecting background knowledge
        to target predicates through logical unification. It's where the "inductive
        leap" happens - connecting observed patterns to existing knowledge to form
        meaningful hypotheses.
        
        🎯 ELI5: This is like a detective connecting evidence (patterns from examples)
        to what they already know (background knowledge) to form theories about
        what rule might explain the observations!
        
        🔬 Algorithm Details:
        The candidate generation process works by:
        
        1. **Pattern Integration**: For each extracted pattern from examples
        2. **Complexity Filtering**: Only consider patterns within variable limits
        3. **Head Creation**: Instantiate target predicate with pattern
        4. **Unification Attempt**: Try to unify background clause with pattern
        5. **Clause Construction**: Build candidate clauses from successful unifications
        
        The key insight is that meaningful hypotheses arise from successful
        unification between target patterns and background knowledge structure.
        
        Args:
            target_predicate (str): The predicate we're learning rules for
                                  Example: "grandmother", "ancestor", "friend"
                                  Will become the head predicate of generated clauses
            bg_clause (LogicalClause): Background knowledge clause to connect
                                     Example: parent(X,Y) :- ... or male(john).
                                     Provides logical structure for rule body
            patterns (List[List[str]]): Abstract patterns from positive examples
                                      Each pattern represents a way to instantiate target
                                      Used to create clause heads and guide unification
        
        Returns:
            List[LogicalClause]: Candidate clauses connecting background to target
                               Each clause has target_predicate as head
                               Body comes from unified background clause
                               May be empty if no unifications succeed
        
        💡 Key Insight: Systematic unification between patterns and background
        knowledge generates semantically meaningful hypothesis candidates!
        
        Example:
            >>> # target_predicate = "grandmother"  
            >>> # bg_clause = parent(X,Y) :- ...
            >>> # patterns = [["sue", "ann"], ["V0", "V1"]]
            >>> candidates = self._generate_candidate_clauses(target, bg_clause, patterns)
            >>> # Might generate: grandmother(X,Y) :- parent(X,Z), parent(Z,Y), female(X)
        
        🔧 Implementation Details:
        - Respects self.max_variables constraint for complexity control
        - Uses _attempt_unification() for sound logical connections
        - Returns empty list if no valid unifications possible
        - Each candidate is a potential rule worth further evaluation
        """
        candidates = []
        
        # Try connecting each pattern to the background clause
        for pattern in patterns:
            # Only consider patterns within variable complexity limits
            if len(pattern) <= self.max_variables:
                # Create head atom from target predicate and pattern
                head_atom = self._instantiate_pattern(target_predicate, pattern)
                
                # Attempt unification with background clause
                unified_body = self._attempt_unification(bg_clause, pattern)
                
                if unified_body:
                    # Successful unification - create candidate clause
                    candidate = LogicalClause(
                        head=head_atom, 
                        body=unified_body,
                        confidence=1.0  # Initial confidence, refined later
                    )
                    candidates.append(candidate)
        
        return candidates
    
    def _attempt_unification(
        self, 
        bg_clause: LogicalClause, 
        target_pattern: List[str]
    ) -> Optional[List[LogicalAtom]]:
        """
        🔗 Attempt Logical Unification Between Background Clause and Target Pattern
        
        This method implements the core unification mechanism that enables ILP to
        connect background knowledge with target predicates. It uses Robinson's
        unification algorithm to find logically sound connections.
        
        🎯 ELI5: This is like trying to fit puzzle pieces together - seeing if
        the shape of what we know (background clause) matches the pattern of
        what we're trying to learn (target pattern)!
        
        🔬 Theoretical Foundation:
        Unification is fundamental to automated theorem proving and logic programming.
        In ILP context, it enables:
        
        - **Sound Hypothesis Formation**: Only generate logically valid connections
        - **Background Integration**: Systematically incorporate existing knowledge  
        - **Variable Binding**: Create proper term relationships across clauses
        
        The algorithm follows Robinson (1965) with ILP-specific adaptations for
        hypothesis generation rather than query resolution.
        
        Algorithm Steps:
        1. **Pattern Term Creation**: Convert target pattern to logical terms
        2. **Dummy Head Creation**: Create temporary atom for unification target
        3. **Robinson Unification**: Attempt to unify background head with target
        4. **Substitution Application**: Apply found substitutions to background body
        5. **Body Construction**: Include background head in unified body
        
        Args:
            bg_clause (LogicalClause): Background knowledge clause
                                     Example: parent(X,Y) :- biological_parent(X,Y)
                                     Provides structure for potential rule body  
            target_pattern (List[str]): Abstract pattern from examples
                                      Example: ["V0", "V1"] or ["john", "mary"]
                                      Defines structure for unification attempt
        
        Returns:
            Optional[List[LogicalAtom]]: Unified body atoms if unification succeeds
                                       None if unification fails
                                       Body includes background clause components
                                       Ready for hypothesis clause construction
        
        💡 Key Insight: Successful unification indicates logical compatibility
        between background knowledge and target predicate structure!
        
        Examples:
            >>> # bg_clause: parent(X,Y) :- biological_parent(X,Y)
            >>> # target_pattern: ["V0", "V1"] 
            >>> result = self._attempt_unification(bg_clause, target_pattern)
            >>> # Might return: [biological_parent(V0,V1), parent(V0,V1)]
            
            >>> # bg_clause: male(john). (fact)
            >>> # target_pattern: ["john"]
            >>> result = self._attempt_unification(bg_clause, target_pattern)
            >>> # Might return: [male(john)]
        
        🔧 Technical Details:
        - Uses "target_pred" as dummy predicate for unification
        - Applies found substitutions to all background clause components
        - Includes background head as body atom (closed-world assumption)
        - Returns None for failed unifications (type/arity mismatches)
        """
        # Convert target pattern to logical terms
        target_terms = []
        for pattern_element in target_pattern:
            if pattern_element.startswith('V'):
                target_terms.append(LogicalTerm(name=pattern_element, term_type='variable'))
            else:
                target_terms.append(LogicalTerm(name=pattern_element, term_type='constant'))
        
        # Create temporary target atom for unification
        target_head = LogicalAtom(predicate="target_pred", terms=target_terms)
        
        # Attempt Robinson unification between background head and target
        substitution = self._robinson_unification(bg_clause.head, target_head)
        
        if substitution is not None:
            # Unification succeeded - apply substitution to background clause
            unified_body = []
            
            # Apply substitution to all body atoms
            for body_atom in bg_clause.body:
                unified_atom = self._apply_substitution(body_atom, substitution)
                unified_body.append(unified_atom)
            
            # Include the background head as a body atom (closed-world assumption)
            # This ensures the background predicate appears in learned rules
            unified_head = self._apply_substitution(bg_clause.head, substitution)
            unified_body.append(unified_head)
            
            return unified_body
        
        return None
    
    def _robinson_unification(
        self, 
        atom1: LogicalAtom, 
        atom2: LogicalAtom
    ) -> Optional[Dict[str, LogicalTerm]]:
        """
        🧮 Robinson's Unification Algorithm for Logical Atoms
        
        This implements the classic Robinson (1965) unification algorithm, adapted
        for ILP hypothesis generation. It's the mathematical foundation that enables
        sound logical reasoning in automated theorem proving and logic programming.
        
        🎯 ELI5: This is like finding if two sentence patterns can be made to match
        by filling in variables - like seeing if "X loves Y" and "john loves mary"
        can be unified by setting X=john and Y=mary!
        
        🔬 Theoretical Foundation:
        Robinson's algorithm is the cornerstone of automated reasoning, providing:
        
        - **Completeness**: Finds unifier if one exists
        - **Soundness**: Only produces valid logical unifications
        - **Efficiency**: Polynomial-time algorithm for most practical cases
        
        In ILP, unification enables systematic connection of background knowledge
        with target predicates, ensuring learned rules are logically valid.
        
        Algorithm Steps:
        1. **Predicate Compatibility**: Check if predicates can unify
        2. **Arity Matching**: Verify same number of arguments
        3. **Term-by-Term Unification**: Recursively unify corresponding terms
        4. **Substitution Building**: Construct variable binding dictionary
        5. **Consistency Checking**: Ensure no conflicting bindings
        
        Args:
            atom1 (LogicalAtom): First atom for unification attempt
                               Example: parent(X, Y) or male(john)
                               Can contain variables, constants, or functions
            atom2 (LogicalAtom): Second atom for unification attempt  
                               Example: parent(john, mary) or male(Z)
                               Must have compatible predicate and arity
        
        Returns:
            Optional[Dict[str, LogicalTerm]]: Variable substitution if unification succeeds
                                            Maps variable names to terms
                                            None if unification impossible
                                            Ready for _apply_substitution()
        
        💡 Key Insight: Unification captures the essence of logical matching -
        finding the most general way to make two expressions identical!
        
        Examples:
            >>> atom1 = LogicalAtom("parent", [LogicalTerm("X", "variable"),
            ...                                LogicalTerm("Y", "variable")])
            >>> atom2 = LogicalAtom("parent", [LogicalTerm("john", "constant"),
            ...                                LogicalTerm("mary", "constant")])
            >>> result = self._robinson_unification(atom1, atom2)
            >>> # Result: {"X": LogicalTerm("john", "constant"), 
            >>> #          "Y": LogicalTerm("mary", "constant")}
            
            >>> atom1 = LogicalAtom("loves", [LogicalTerm("X", "variable")])
            >>> atom2 = LogicalAtom("hates", [LogicalTerm("john", "constant")])
            >>> result = self._robinson_unification(atom1, atom2)
            >>> # Result: None (predicate mismatch)
        
        🔧 Implementation Notes:
        - Uses _predicates_compatible() for enhanced predicate matching
        - Calls _unify_terms() for recursive term unification
        - Handles occurs check through _unify_terms()
        - Returns consistent substitution dictionary or None
        """
        # Enhanced predicate compatibility checking
        if not self._predicates_compatible(atom1.predicate, atom2.predicate):
            return None
        
        # Arity must match for unification
        if len(atom1.terms) != len(atom2.terms):
            return None
        
        # Initialize empty substitution dictionary
        substitution = {}
        
        # Unify terms pairwise using recursive term unification
        for term1, term2 in zip(atom1.terms, atom2.terms):
            if not self._unify_terms(term1, term2, substitution):
                return None  # Unification failed
        
        return substitution
    
    def _unify_terms(
        self, 
        term1: LogicalTerm, 
        term2: LogicalTerm, 
        substitution: Dict[str, LogicalTerm]
    ) -> bool:
        """
        🔗 Unify Two Logical Terms with Occurs Check
        
        This method handles the detailed term-level unification that forms the
        foundation of Robinson's algorithm. It manages variable binding, constant
        matching, and function unification with proper occurs checking.
        
        🎯 ELI5: This is like checking if two puzzle pieces can fit together,
        where some pieces are flexible (variables) and others are fixed (constants).
        The occurs check prevents impossible infinite loops!
        
        🔬 Algorithm Details:
        The unification follows these rules:
        
        1. **Identical Terms**: Already unified terms succeed immediately
        2. **Variable Unification**: Variables can bind to any term (with occurs check)
        3. **Constant Matching**: Constants must be identical to unify
        4. **Function Unification**: Same functor with recursively unified arguments
        5. **Occurs Check**: Prevents infinite structures (X = f(X))
        
        The occurs check is crucial for soundness - it prevents creating infinite
        structures that would make the logic system inconsistent.
        
        Args:
            term1 (LogicalTerm): First term to unify
                               Can be variable, constant, or function
                               Variables may already have substitutions
            term2 (LogicalTerm): Second term to unify
                               Can be variable, constant, or function  
                               Variables may already have substitutions
            substitution (Dict[str, LogicalTerm]): Current variable bindings
                                                 Modified in-place during unification
                                                 Maps variable names to terms
        
        Returns:
            bool: True if terms can be unified, False otherwise
                 Modifies substitution dictionary as side effect
                 Failure leaves substitution in partial state
        
        💡 Key Insight: Term unification handles the detailed mechanics of
        variable binding while maintaining logical soundness!
        
        Examples:
            >>> # Variable-constant unification
            >>> term1 = LogicalTerm("X", "variable")
            >>> term2 = LogicalTerm("john", "constant")
            >>> result = self._unify_terms(term1, term2, {})
            >>> # Result: True, substitution becomes {"X": LogicalTerm("john", "constant")}
            
            >>> # Constant-constant matching
            >>> term1 = LogicalTerm("john", "constant")
            >>> term2 = LogicalTerm("mary", "constant")
            >>> result = self._unify_terms(term1, term2, {})
            >>> # Result: False (different constants)
            
            >>> # Occurs check failure
            >>> term1 = LogicalTerm("X", "variable")
            >>> term2 = LogicalTerm("f", "function", [LogicalTerm("X", "variable")])
            >>> result = self._unify_terms(term1, term2, {})
            >>> # Result: False (X cannot equal f(X))
        
        🔧 Technical Details:
        - Applies existing substitutions before unification attempt
        - Uses occurs check to prevent infinite structures
        - Modifies substitution dictionary in-place for efficiency
        - Handles nested function terms through recursion
        """
        # Apply current substitutions to both terms before unification
        term1 = self._apply_substitution_to_term(term1, substitution)
        term2 = self._apply_substitution_to_term(term2, substitution)
        
        # Same term - unification succeeds trivially
        if term1.name == term2.name and term1.term_type == term2.term_type:
            return True
        
        # Variable unification cases
        if term1.term_type == 'variable':
            # Occurs check to prevent infinite structures
            if self._occurs_check(term1.name, term2, substitution):
                return False
            substitution[term1.name] = term2
            return True
        
        elif term2.term_type == 'variable':
            # Occurs check to prevent infinite structures  
            if self._occurs_check(term2.name, term1, substitution):
                return False
            substitution[term2.name] = term1
            return True
        
        # Constant unification - must be identical
        elif term1.term_type == 'constant' and term2.term_type == 'constant':
            return term1.name == term2.name
        
        # Function unification - same functor, unified arguments
        elif term1.term_type == 'function' and term2.term_type == 'function':
            if term1.name != term2.name:
                return False
            
            # Check arity
            args1 = term1.arguments or []
            args2 = term2.arguments or []
            if len(args1) != len(args2):
                return False
            
            # Recursively unify arguments
            for arg1, arg2 in zip(args1, args2):
                if not self._unify_terms(arg1, arg2, substitution):
                    return False
            
            return True
        
        # Different types that can't be unified
        return False
    
    def _occurs_check(
        self, 
        var_name: str, 
        term: LogicalTerm, 
        substitution: Dict[str, LogicalTerm]
    ) -> bool:
        """
        🔄 Occurs Check to Prevent Infinite Structures
        
        This method implements the occurs check, a crucial component of sound
        unification that prevents the creation of infinite or cyclic structures
        that would break the logical system.
        
        🎯 ELI5: This is like checking if you're trying to define something in
        terms of itself in a way that creates an infinite loop - like saying
        "X equals the box containing X" which would create an endless recursion!
        
        🔬 Theoretical Foundation:
        The occurs check is essential for:
        
        - **Logical Soundness**: Prevents inconsistent term structures
        - **Termination**: Ensures unification algorithm terminates
        - **Decidability**: Maintains decidable unification procedure
        
        Without occurs check, unification might create terms like X = f(X),
        leading to infinite expansion: X = f(f(f(f(...))))
        
        Algorithm:
        1. **Direct Variable Check**: Is the term just the variable itself?
        2. **Substitution Resolution**: Apply current substitutions to term
        3. **Function Recursion**: Check if variable occurs in function arguments
        4. **Transitive Checking**: Follow substitution chains completely
        
        Args:
            var_name (str): Name of variable to check for occurrence
                          Example: "X", "Y", "Z"
                          Variable that might create infinite structure
            term (LogicalTerm): Term to check for variable occurrence
                              May be constant, variable, or function
                              Could contain var_name directly or indirectly
            substitution (Dict[str, LogicalTerm]): Current variable bindings
                                                 Used to resolve variables in term
                                                 May contain chains of substitutions
        
        Returns:
            bool: True if var_name occurs in term (infinite structure would result)
                 False if safe to bind var_name to term
                 Conservative - returns True for borderline cases
        
        💡 Key Insight: Preventing infinite structures preserves logical
        consistency while allowing maximum safe unification!
        
        Examples:
            >>> # Direct occurrence
            >>> occurs = self._occurs_check("X", LogicalTerm("X", "variable"), {})
            >>> # Result: True (X occurs in X)
            
            >>> # Safe constant binding
            >>> occurs = self._occurs_check("X", LogicalTerm("john", "constant"), {})
            >>> # Result: False (safe to bind X to john)
            
            >>> # Function with occurrence
            >>> func_term = LogicalTerm("f", "function", [LogicalTerm("X", "variable")])
            >>> occurs = self._occurs_check("X", func_term, {})
            >>> # Result: True (X occurs in f(X))
            
            >>> # Indirect occurrence through substitution
            >>> sub = {"Y": LogicalTerm("X", "variable")}
            >>> occurs = self._occurs_check("X", LogicalTerm("Y", "variable"), sub)
            >>> # Result: True (Y resolves to X)
        
        🔧 Implementation Notes:
        - Applies substitutions before checking to handle indirect occurrence
        - Recursively checks function arguments for deep occurrence
        - Conservative approach - prefers safety over maximum unification
        - Essential for maintaining logical system integrity
        """
        # Apply current substitutions to resolve the term
        term = self._apply_substitution_to_term(term, substitution)
        
        # Direct occurrence check
        if term.term_type == 'variable' and term.name == var_name:
            return True
        
        # Function occurrence check - recursively check arguments
        elif term.term_type == 'function' and term.arguments:
            return any(
                self._occurs_check(var_name, arg, substitution) 
                for arg in term.arguments
            )
        
        # Constants and unrelated variables are safe
        return False
    
    def _apply_substitution_to_term(
        self, 
        term: LogicalTerm, 
        substitution: Dict[str, LogicalTerm]
    ) -> LogicalTerm:
        """
        🔄 Apply Variable Substitution to Logical Term
        
        This method applies a substitution dictionary to a logical term, replacing
        variables with their bound values. It's fundamental to the unification
        process and hypothesis generation.
        
        🎯 ELI5: This is like find-and-replace for logical expressions - wherever
        you see a variable like X, replace it with whatever X is bound to,
        like "john" or another expression!
        
        🔬 Algorithm Details:
        The substitution process handles different term types:
        
        1. **Variable Terms**: Replace with bound value if exists in substitution
        2. **Constant Terms**: Return unchanged (constants don't get substituted)
        3. **Function Terms**: Recursively apply to all arguments
        
        The process may need to be applied multiple times for transitive
        substitutions (X->Y, Y->john leads to X->john).
        
        Args:
            term (LogicalTerm): Term to apply substitution to
                              Can be variable, constant, or function
                              Variables may or may not appear in substitution
            substitution (Dict[str, LogicalTerm]): Variable binding dictionary
                                                 Maps variable names to replacement terms
                                                 May contain nested variable chains
        
        Returns:
            LogicalTerm: Term with substitutions applied
                        Variables replaced with bound values
                        Functions have arguments recursively substituted
                        Constants returned unchanged
        
        💡 Key Insight: Systematic substitution application enables the
        propagation of variable bindings throughout logical structures!
        
        Examples:
            >>> # Variable substitution
            >>> term = LogicalTerm("X", "variable")
            >>> sub = {"X": LogicalTerm("john", "constant")}
            >>> result = self._apply_substitution_to_term(term, sub)
            >>> # Result: LogicalTerm("john", "constant")
            
            >>> # Function term substitution
            >>> term = LogicalTerm("f", "function", [LogicalTerm("X", "variable")])
            >>> sub = {"X": LogicalTerm("mary", "constant")}
            >>> result = self._apply_substitution_to_term(term, sub)
            >>> # Result: LogicalTerm("f", "function", [LogicalTerm("mary", "constant")])
            
            >>> # Unbound variable
            >>> term = LogicalTerm("Y", "variable")
            >>> sub = {"X": LogicalTerm("john", "constant")}
            >>> result = self._apply_substitution_to_term(term, sub)
            >>> # Result: LogicalTerm("Y", "variable") (unchanged)
        
        🔧 Technical Details:
        - Only substitutes variables that appear in substitution dictionary
        - Preserves term structure for constants and unbound variables
        - Recursively processes function arguments
        - Does not perform transitive substitution (single-step only)
        """
        if term.term_type == 'variable' and term.name in substitution:
            # Replace variable with its bound value
            return substitution[term.name]
        
        elif term.term_type == 'function' and term.arguments:
            # Recursively apply substitution to function arguments
            new_args = [
                self._apply_substitution_to_term(arg, substitution) 
                for arg in term.arguments
            ]
            return LogicalTerm(
                name=term.name, 
                term_type='function', 
                arguments=new_args
            )
        
        else:
            # Constants and unbound variables remain unchanged
            return term
    
    def _apply_substitution(
        self, 
        atom: LogicalAtom, 
        substitution: Dict[str, LogicalTerm]
    ) -> LogicalAtom:
        """
        🔄 Apply Variable Substitution to Logical Atom
        
        This method applies variable substitutions to all terms within a logical
        atom, creating a new atom with variables replaced by their bound values.
        
        🎯 ELI5: This takes a sentence like "loves(X, Y)" and a dictionary saying
        "X=john, Y=mary" and creates "loves(john, mary)" - filling in all the
        blanks with their actual values!
        
        Args:
            atom (LogicalAtom): Atom to apply substitution to
                              Contains terms that may include variables
                              Predicate name remains unchanged
            substitution (Dict[str, LogicalTerm]): Variable binding dictionary
                                                 Maps variable names to replacement terms
                                                 Applied to all terms in atom
        
        Returns:
            LogicalAtom: New atom with substitutions applied
                        Same predicate and negation status
                        All terms have variables substituted
                        Ready for use in unified clauses
        
        Example:
            >>> atom = LogicalAtom("parent", [LogicalTerm("X", "variable"),
            ...                               LogicalTerm("Y", "variable")])
            >>> sub = {"X": LogicalTerm("john", "constant"),
            ...        "Y": LogicalTerm("mary", "constant")}
            >>> result = self._apply_substitution(atom, sub)
            >>> # Result: LogicalAtom("parent", [LogicalTerm("john", "constant"),
            >>> #                                LogicalTerm("mary", "constant")])
        """
        new_terms = [
            self._apply_substitution_to_term(term, substitution) 
            for term in atom.terms
        ]
        return LogicalAtom(
            predicate=atom.predicate, 
            terms=new_terms, 
            negated=atom.negated
        )