"""Public API: Loop class + convenience helpers.

Surface:
- Loop: run/arun, session/asession
- Functional wrappers: run/arun
- Helpers: allow (permissions), mcp (servers)
"""

from __future__ import annotations

import asyncio
from collections.abc import AsyncIterator, Awaitable, Iterator
from contextlib import asynccontextmanager, contextmanager
from typing import Any, Protocol, runtime_checkable

from pydantic import BaseModel, TypeAdapter

from .events import OpenCodeEvent, parse_event
from .helper import safe_debug_log, to_session_id
from .invoke import async_invoke, run_opencode_jsonl
from .mcp import LocalMcpServer, RemoteMcpServer
from .permissions import Permission, PermissionLevel
from .providers import ProviderConfig
from .request import Request
from .response import Response
from .structured import invoke_structured as async_invoke_structured


def allow(
    *tools: str,
    read: bool = True,
    write: bool = True,
    bash: bool | dict[str, PermissionLevel] = False,
    webfetch: bool = False,
) -> Permission:
    """Convenience builder for Permission.

    Semantics:
    - tools may contain "bash" or "webfetch" to allow them quickly
    - edit is ALLOW only when both read=True and write=True; otherwise DENY
    - passing a dict for `bash` enables fine‑grained tool policies
    """
    edit_level = Permission.ALLOW if write and read else Permission.DENY
    bash_level = (
        Permission.ALLOW
        if (bash is True or "bash" in tools)
        else Permission.DENY
    )
    web_level = (
        Permission.ALLOW
        if (webfetch or "webfetch" in tools)
        else Permission.DENY
    )
    return Permission(
        edit=edit_level,
        bash=bash if isinstance(bash, dict) else bash_level,
        webfetch=web_level,
    )


def mcp(**servers: str) -> dict[str, LocalMcpServer | RemoteMcpServer]:
    """Build MCP server definitions.

    - Remote: pass a URL string (http/https)
    - Local: pass a command string; optional "ENV=VALUE" tokens allowed before cmd

    Examples:
      mcp(context7="https://mcp.context7.com/mcp")
      mcp(biomcp="uvx --from biomcp-python biomcp run")
    """
    out: dict[str, LocalMcpServer | RemoteMcpServer] = {}
    for name, spec in servers.items():
        s = spec.strip()
        if s.startswith(("http://", "https://")):
            out[name] = RemoteMcpServer(name=name, url=s)  # type: ignore[arg-type]
            continue
        env: dict[str, str] = {}
        cmd: list[str] = []
        for tok in s.split():
            if "=" in tok and not cmd:
                k, _, v = tok.partition("=")
                env[k] = v
            else:
                cmd.append(tok)
        out[name] = LocalMcpServer(
            name=name, command=cmd, environment=env or None
        )
    return out


# providers(...) helper removed; prefer innerloop.providers.provider(name, **options)


class Loop:
    """Reusable loop that hides config and exposes simple methods."""

    def __init__(
        self,
        *,
        model: str,
        perms: Permission | None = None,
        providers: dict[str, ProviderConfig] | None = None,
        mcp: dict[str, LocalMcpServer | RemoteMcpServer] | None = None,
        timeout: float | None = None,
    ) -> None:
        self.model = model
        self.perms = perms or Permission()
        self.providers = providers
        self.mcp = mcp
        self.default_response_format: (
            type[BaseModel] | TypeAdapter[Any] | None
        ) = None
        self._last_session_id: str | None = None
        self.default_timeout: float | None = timeout

    def run(
        self,
        prompt: str,
        *,
        response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        session: str | Response[Any] | None = None,
        max_retries: int = 3,
        # Single timeout (idle inactivity)
        timeout: float | None = None,
    ) -> Response[Any]:
        # Avoid creating coroutine objects when running inside an event loop;
        # raising early prevents un-awaited coroutine warnings in async tests.
        try:
            asyncio.get_running_loop()
        except RuntimeError:
            # No running loop → safe to proceed with asyncio.run
            pass
        else:
            # Avoid creating coroutine objects that would trigger "never awaited"
            # warnings when raising from within an active event loop.
            raise RuntimeError(
                "asyncio.run() cannot be called from a running event loop"
            )
        sid = to_session_id(session)
        if sid is None:
            sid = self._last_session_id
        eff_format = (
            response_format
            if response_format is not None
            else self.default_response_format
        )
        req = Request(
            model=self.model,
            prompt=prompt,
            permission=self.perms,
            providers=self.providers,
            mcp=self.mcp,
            response_format=eff_format,
            session=sid,
        )
        eff_timeout = timeout if timeout is not None else self.default_timeout
        if eff_format is not None:
            resp = asyncio.run(
                async_invoke_structured(
                    req, max_retries=max_retries, timeout=eff_timeout
                )
            )
        else:
            resp = asyncio.run(async_invoke(req, timeout=eff_timeout))
        if resp.session_id:
            self._last_session_id = resp.session_id
        return resp

    async def arun(
        self,
        prompt: str,
        *,
        response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        session: str | Response[Any] | None = None,
        max_retries: int = 3,
        timeout: float | None = None,
    ) -> Response[Any]:
        sid = to_session_id(session)
        if sid is None:
            sid = self._last_session_id
        eff_format = (
            response_format
            if response_format is not None
            else self.default_response_format
        )
        req = Request(
            model=self.model,
            prompt=prompt,
            permission=self.perms,
            providers=self.providers,
            mcp=self.mcp,
            response_format=eff_format,
            session=sid,
        )
        eff_timeout = timeout if timeout is not None else self.default_timeout
        if eff_format is not None:
            resp = await async_invoke_structured(
                req, max_retries=max_retries, timeout=eff_timeout
            )
        else:
            resp = await async_invoke(req, timeout=eff_timeout)
        if resp.session_id:
            self._last_session_id = resp.session_id
        return resp

    async def iterate_events(
        self,
        prompt: str,
        *,
        session: str | Response[Any] | None = None,
        timeout: float | None = None,
    ) -> AsyncIterator[OpenCodeEvent]:
        """Stream events from the LLM in real-time.

        This is a low-level interface that yields parsed event objects as they arrive.
        Use this when you need fine-grained visibility into what the LLM is doing,
        such as for building UIs, monitoring progress, or custom event processing.

        Note: This method does NOT support structured outputs (response_format).
        For structured outputs with retries, use run() or arun() which handle
        validation and retries automatically.

        Args:
            prompt: The user prompt to send to the LLM
            session: Optional session ID or Response object for multi-turn conversations
            timeout: Idle timeout in seconds (how long to wait for next event)

        Yields:
            OpenCodeEvent: Parsed event objects (StepStartEvent, TextEvent, ToolUseEvent,
                          StepFinishEvent, or ErrorEvent)

        Example:
            >>> loop = Loop(model="anthropic/claude-3-5-sonnet-20241022")
            >>> async for event in loop.iterate_events("What is 2 + 2?"):
            ...     if event.type == "text":
            ...         print(event.text, end="", flush=True)
            ...     elif event.type == "step_finish":
            ...         print(f"\\nCost: ${event.cost:.6f}")
        """
        sid = to_session_id(session)
        if sid is None:
            sid = self._last_session_id
        eff_timeout = timeout if timeout is not None else self.default_timeout

        # Stream raw events and parse them
        async for raw_event in run_opencode_jsonl(
            prompt,
            model=self.model,
            permission=self.perms,
            providers=self.providers,
            mcp_servers=self.mcp,  # type: ignore[arg-type]
            session_id=sid,
            timeout=eff_timeout,
        ):
            # Parse and yield typed event
            event = parse_event(raw_event)
            yield event

            # Update session ID from first event if not set
            if self._last_session_id is None:
                self._last_session_id = (
                    getattr(event, "sessionID", None) or None
                )

    @runtime_checkable
    class AskSync(Protocol):
        def __call__(
            self,
            prompt: str,
            response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        ) -> Response[Any]: ...

    @runtime_checkable
    class AskAsync(Protocol):
        def __call__(
            self,
            prompt: str,
            response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        ) -> Awaitable[Response[Any]]: ...

    @contextmanager
    def session(self) -> Iterator[AskSync]:
        sid: str | None = None

        def ask(
            prompt: str,
            response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        ) -> Response[Any]:
            nonlocal sid
            if sid is None:
                # Empty string sentinel means: force new session (no reuse),
                # let CLI allocate a real session ID on first call.
                sid = ""
            eff_format = (
                response_format
                if response_format is not None
                else self.default_response_format
            )
            resp = self.run(
                prompt,
                response_format=eff_format,
                session=sid,
            )
            sid = resp.session_id or sid
            return resp

        yield ask

    @asynccontextmanager
    async def asession(self) -> AsyncIterator[AskAsync]:
        sid: str | None = None

        async def ask(
            prompt: str,
            response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
        ) -> Response[Any]:
            nonlocal sid
            if sid is None:
                # Empty string sentinel means: force new session (no reuse),
                # let CLI allocate a real session ID on first call.
                sid = ""
            eff_format = (
                response_format
                if response_format is not None
                else self.default_response_format
            )
            resp = await self.arun(
                prompt,
                response_format=eff_format,
                session=sid,
            )
            sid = resp.session_id or sid
            return resp

        yield ask


def run(
    prompt: str,
    *,
    model: str,
    response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
    session: str | Response[Any] | None = None,
    perms: Permission | None = None,
    providers: dict[str, ProviderConfig] | None = None,
    mcp: dict[str, LocalMcpServer | RemoteMcpServer] | None = None,
    timeout: float | None = None,
) -> Response[Any]:
    safe_debug_log("API", f"run() START: model={model}, timeout={timeout}")
    try:
        result = Loop(
            model=model, perms=perms, providers=providers, mcp=mcp
        ).run(
            prompt,
            response_format=response_format,
            session=session,
            timeout=timeout,
        )
        safe_debug_log("API", "run() SUCCESS")
        return result
    except Exception as e:
        safe_debug_log("API", f"run() EXCEPTION: {type(e).__name__}: {e}")
        raise


async def arun(
    prompt: str,
    *,
    model: str,
    response_format: type[BaseModel] | TypeAdapter[Any] | None = None,
    session: str | Response[Any] | None = None,
    perms: Permission | None = None,
    providers: dict[str, ProviderConfig] | None = None,
    mcp: dict[str, LocalMcpServer | RemoteMcpServer] | None = None,
    timeout: float | None = None,
) -> Response[Any]:
    return await Loop(
        model=model, perms=perms, providers=providers, mcp=mcp
    ).arun(
        prompt,
        response_format=response_format,
        session=session,
        timeout=timeout,
    )


async def iterate_events(
    prompt: str,
    *,
    model: str,
    session: str | Response[Any] | None = None,
    perms: Permission | None = None,
    providers: dict[str, ProviderConfig] | None = None,
    mcp: dict[str, LocalMcpServer | RemoteMcpServer] | None = None,
    timeout: float | None = None,
) -> AsyncIterator[OpenCodeEvent]:
    """Stream events from the LLM in real-time.

    This is a low-level interface that yields parsed event objects as they arrive.
    Use this when you need fine-grained visibility into what the LLM is doing,
    such as for building UIs, monitoring progress, or custom event processing.

    Note: This function does NOT support structured outputs (response_format).
    For structured outputs with retries, use run() or arun() which handle
    validation and retries automatically.

    Args:
        prompt: The user prompt to send to the LLM
        model: Model ID (e.g., "anthropic/claude-3-5-sonnet-20241022")
        session: Optional session ID or Response object for multi-turn conversations
        perms: Permission settings controlling what tools the LLM can use
        providers: Optional provider configurations
        mcp: Optional MCP (Model Context Protocol) servers
        timeout: Idle timeout in seconds (how long to wait for next event)

    Yields:
        OpenCodeEvent: Parsed event objects (StepStartEvent, TextEvent, ToolUseEvent,
                      StepFinishEvent, or ErrorEvent)

    Example:
        >>> async for event in iterate_events("What is 2 + 2?", model="anthropic/claude-3-5-sonnet-20241022"):
        ...     if event.type == "text":
        ...         print(event.text, end="", flush=True)
        ...     elif event.type == "step_finish":
        ...         print(f"\\nCost: ${event.cost:.6f}")
    """
    loop = Loop(model=model, perms=perms, providers=providers, mcp=mcp)
    async for event in loop.iterate_events(
        prompt, session=session, timeout=timeout
    ):
        yield event


__all__ = [
    "Loop",
    "run",
    "arun",
    "iterate_events",
    "allow",
    "mcp",
]
