"""Agent management commands."""

from __future__ import annotations

import json

import typer

from ..agent import build_opencode_config
from ..permissions import Permission
from ..server.client import OpenCodeClient

agent_app = typer.Typer(name="agent", help="Manage OpenCode agents")


@agent_app.command()
def create(
    name: str,
    model: str,
    provider_config: str | None = typer.Option(
        None, "--provider", help="Provider config JSON file"
    ),
    no_write: bool = typer.Option(
        False, "--no-write", help="Disable write/edit tools"
    ),
    allow_bash: bool = typer.Option(
        False, "--allow-bash", help="Enable bash tool"
    ),
    allow_web: bool = typer.Option(
        False, "--allow-web", help="Enable webfetch tool"
    ),
    server_url: str = typer.Option(
        "http://127.0.0.1:4096", "--server", help="OpenCode server URL"
    ),
) -> None:
    """Create and upsert an agent to OpenCode server.

    Example:
        innerloop agent create lmstudio-agent lmstudio/google/gemma-3n-e4b \\
            --provider lmstudio-config.json --no-write --allow-bash
    """
    # Load provider config if provided
    providers = None
    if provider_config:
        with open(provider_config) as f:
            providers = json.load(f)

    # Build permissions
    perm = Permission(
        edit=Permission.DENY if no_write else Permission.ALLOW,
        bash=Permission.ALLOW if allow_bash else Permission.DENY,
        webfetch=Permission.ALLOW if allow_web else Permission.DENY,
    )

    # Build full OpenCode config
    config = build_opencode_config(
        model=model,
        permission=perm,
        providers=providers,
    )

    # Extract agent name
    agent_name = list(config["agent"].keys())[0]

    # Write to BotAssembly OpenCode config
    from ..home import get_home

    home = get_home()
    config_file = home / "opencode" / "opencode.json"
    config_file.parent.mkdir(parents=True, exist_ok=True)

    # Load existing config or create new
    if config_file.exists():
        existing = json.loads(config_file.read_text())
    else:
        existing = {"$schema": "https://opencode.ai/config.json"}

    # Merge: add our agent and provider
    existing.setdefault("agent", {}).update(config.get("agent", {}))
    if providers:
        existing.setdefault("provider", {}).update(config.get("provider", {}))

    # Write back
    config_file.write_text(json.dumps(existing, indent=2) + "\n")

    typer.echo(f"✓ Created agent: {agent_name}")
    typer.echo(f"  Model: {model}")
    typer.echo(f"  File: {config_file}")
    if providers:
        typer.echo(f"  Providers: {list(providers.keys())}")

    # Restart server if running (to reload config)
    client = OpenCodeClient(server_url)
    if client.is_available():
        typer.echo("  Server: Restart needed (run: innerloop server restart)")


@agent_app.command(name="list")
def list_agents(
    server_url: str = typer.Option(
        "http://127.0.0.1:4096", "--server", help="OpenCode server URL"
    ),
) -> None:
    """List all agents on OpenCode server."""
    client = OpenCodeClient(server_url)

    if not client.is_available():
        typer.echo(f"✗ Server not running at {server_url}", err=True)
        raise typer.Exit(1)

    agents = client.list_agents()

    if not agents:
        typer.echo("No agents found")
        return

    typer.echo(f"Found {len(agents)} agent(s):")
    for agent in agents:
        name = agent.get("name", "unknown")
        model = agent.get("model", "unknown")
        typer.echo(f"  {name}")
        typer.echo(f"    Model: {model}")


__all__ = ["agent_app"]
