"""Run command implementation."""

from __future__ import annotations

import json
import os
import sys

from ..api import Loop
from ..permissions import Permission


def run_command(
    prompt: str,
    model: str,
    struct: str | None,
    session: str | None,
    timeout: float | None,
    stream: bool,
    format: str,
    server_url: str | None,
) -> None:
    """Execute prompt via OpenCode.

    Args:
        prompt: Prompt to execute (or "-" for stdin)
        model: Model string
        struct: Schema name for structured output
        session: Session ID to continue
        timeout: Idle timeout
        stream: Stream events
        format: Output format (text, json, events)
        server_url: OpenCode server URL
    """
    try:
        _run_command_impl(
            prompt, model, struct, session, timeout, stream, format, server_url
        )
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def _run_command_impl(
    prompt: str,
    model: str,
    struct: str | None,
    session: str | None,
    timeout: float | None,
    stream: bool,
    format: str,
    server_url: str | None,
) -> None:
    """Implementation of run command."""
    # Read from stdin if prompt is "-"
    if prompt == "-":
        prompt = sys.stdin.read()

    # Load schema if specified
    response_format: str | None = None
    if struct:
        from ..schemas import load_schema

        try:
            response_format = load_schema(struct)
        except FileNotFoundError:
            print(f"✗ Schema not found: {struct}", file=sys.stderr)
            print(
                f"  Save with: innerloop schema save {struct} module:Class",
                file=sys.stderr,
            )
            sys.exit(1)

    # Build Loop with server_url if provided or auto-detect
    effective_server_url = server_url or os.getenv("INNERLOOP_SERVER_URL")

    loop = Loop(
        model=model,
        perms=Permission(),
        timeout=timeout,
        server_url=effective_server_url,
    )

    # Run (stream or blocking)
    if stream:
        # Stream events
        for event in loop.stream(
            prompt, response_format=response_format, session=session
        ):
            if format == "events":
                # JSONL output
                print(json.dumps(event.model_dump(by_alias=True)))
            else:
                # Text streaming (if TextEvent)
                from ..events import TextEvent

                if isinstance(event, TextEvent):
                    print(event.part.text or "", end="", flush=True)
        if format != "events":
            print()  # Final newline
    else:
        # Blocking run
        resp = loop.run(
            prompt, response_format=response_format, session=session
        )

        # Output formatting
        if format == "json":
            print(json.dumps(resp.model_dump(by_alias=True), indent=2))
        else:
            print(resp.output)


__all__ = ["run_command"]
