"""Core home directory functions."""

from __future__ import annotations

import os
from pathlib import Path

# Module-level home directory state (set once, used everywhere)
_home: Path | None = None


def set_home(path: Path | str) -> None:
    """Set home directory explicitly (from CLI --home flag).

    Args:
        path: Path to use as home directory
    """
    global _home
    _home = Path(path).resolve()


def get_home() -> Path:
    """Get home directory using priority resolution.

    Priority:
    1. Explicitly set via set_home()
    2. BOTASSEMBLY_HOME environment variable
    3. .botassembly/ in current working directory
    4. ~/.local/botassembly/ (default, created if needed)

    Returns:
        Resolved home directory path
    """
    # 1. Explicitly set
    if _home is not None:
        return _home

    # 2. Environment variable
    if env_home := os.getenv("BOTASSEMBLY_HOME"):
        return Path(env_home).resolve()

    # 3. Current working directory
    cwd_home = Path.cwd() / ".botassembly"
    if cwd_home.exists():
        return cwd_home.resolve()

    # 4. Global default (create if needed)
    global_home = Path.home() / ".local" / "botassembly"
    global_home.mkdir(parents=True, exist_ok=True)
    return global_home.resolve()
