"""OpenCode HTTP client for server API."""

from __future__ import annotations

from collections.abc import AsyncIterator
from typing import Any, cast

import httpx


class OpenCodeClient:
    """HTTP client for OpenCode server API.

    Based on opencode-server-config.md spec.
    """

    def __init__(self, base_url: str = "http://127.0.0.1:4096"):
        """Initialize client.

        Args:
            base_url: OpenCode server base URL
        """
        self.base_url = base_url.rstrip("/")

    def get_paths(self) -> dict[str, str]:
        """GET /path - discover config directories.

        Returns:
            Dict with keys: state, config, worktree, directory
        """
        resp = httpx.get(f"{self.base_url}/path", timeout=5.0)
        resp.raise_for_status()
        return cast(dict[str, str], resp.json())

    def list_agents(self) -> list[dict[str, Any]]:
        """GET /agent - list all agents.

        Returns:
            List of agent info dicts (name, description, mode, etc.)
        """
        resp = httpx.get(f"{self.base_url}/agent", timeout=5.0)
        resp.raise_for_status()
        return cast(list[dict[str, Any]], resp.json())

    def upsert_agent(self, name: str, config: dict[str, Any]) -> None:
        """PATCH /config?directory=config - upsert agent.

        Uses get_paths() to find config directory,
        then patches { "agent": { name: config } }.

        Args:
            name: Agent name
            config: Agent configuration dict
        """
        paths = self.get_paths()
        config_dir = paths["config"]

        patch = {"agent": {name: config}}

        resp = httpx.patch(
            f"{self.base_url}/config",
            params={"directory": config_dir},
            json=patch,
            timeout=10.0,
        )
        resp.raise_for_status()

    def create_session(self) -> str:
        """POST /session - create new session.

        Returns:
            Session ID
        """
        resp = httpx.post(f"{self.base_url}/session", json={}, timeout=5.0)
        resp.raise_for_status()
        data = cast(dict[str, Any], resp.json())
        return cast(str, data["id"])

    async def send_message(
        self,
        session_id: str,
        parts: list[dict[str, Any]],
        agent: str | None = None,
    ) -> AsyncIterator[dict[str, Any]]:
        """POST /session/:id/message - send prompt, get response.

        Args:
            session_id: Session ID
            parts: Message parts (text, etc.)
            agent: Optional agent name

        Yields:
            Event dicts from response.parts array
        """
        body: dict[str, Any] = {"parts": parts}
        if agent:
            body["agent"] = agent

        async with httpx.AsyncClient(timeout=120.0) as client:
            resp = await client.post(
                f"{self.base_url}/session/{session_id}/message",
                json=body,
            )
            resp.raise_for_status()
            data = cast(dict[str, Any], resp.json())

            # Yield parts from response
            response_parts = data.get("parts", [])
            for part in response_parts:
                yield part

    def is_available(self) -> bool:
        """Check if server is available.

        Returns:
            True if server responds to /agent
        """
        try:
            resp = httpx.get(f"{self.base_url}/agent", timeout=2.0)
            return resp.status_code == 200
        except httpx.HTTPError:
            return False


__all__ = ["OpenCodeClient"]
