"""Server-based invocation (HTTP mode)."""

from __future__ import annotations

import json
from typing import Any, cast

import httpx

from ..agent import agent_hash, build_agent_config
from ..events import StreamEvent, TimeInfo, parse_event
from ..helper import unix_ms
from ..request import Request
from ..response import Response
from ..usage import compute_usage
from .client import OpenCodeClient


async def async_invoke_via_server(
    request: Request,
    *,
    server_url: str = "http://127.0.0.1:4096",
    timeout: float | None = None,
) -> Response[Any]:
    """Run invocation via OpenCode HTTP server.

    Args:
        request: Request object
        server_url: OpenCode server URL
        timeout: Optional timeout (not yet implemented for server mode)

    Returns:
        Response object

    Raises:
        RuntimeError: If server communication fails
    """
    client = OpenCodeClient(server_url)

    # Build agent config and compute hash
    agent_cfg = build_agent_config(
        request.model,
        request.permission,
        mcp=cast(dict[str, Any] | None, request.mcp),
    )
    agent_name = agent_hash(agent_cfg)

    # Upsert agent to server
    client.upsert_agent(agent_name, agent_cfg)

    # Create or reuse session
    session_id = request.session
    if not session_id:
        session_id = client.create_session()

    # Build prompt with structured output instructions if needed
    prompt_text = request.prompt
    if request.response_format is not None:
        from slimschema import to_prompt  # type: ignore[import-untyped]

        from ..api import _render_permission_instructions
        from ..structured import _to_slimschema_yaml

        # Build permission text
        permission_text = _render_permission_instructions(request.permission)

        # Build schema instructions
        schema_yaml = _to_slimschema_yaml(request.response_format)
        schema_instructions = to_prompt(
            schema_yaml,
            errors=None,
            xml_tag="output",
            fence_language="json",
        )

        # Combine parts
        parts_list = [request.prompt]
        if permission_text:
            parts_list.append(permission_text)
        parts_list.append(schema_instructions)
        prompt_text = "\n\n---\n\n".join(parts_list)

    # Send message
    parts = [{"type": "text", "text": prompt_text}]

    events: list[StreamEvent] = []
    wall_start = unix_ms()

    try:
        async for part in client.send_message(
            session_id, parts, agent=agent_name
        ):
            # Server returns flat parts, wrap in event structure
            part_type = part.get("type", "").replace("-", "_")

            event_dict = {
                "type": part_type,
                "sessionID": part.get("sessionID", session_id),
                "part": part,
            }

            ev = parse_event(event_dict)
            ev.seq = len(events) + 1
            events.append(ev)
    except (httpx.HTTPError, json.JSONDecodeError) as e:
        raise RuntimeError(
            f"Server communication failed: {e.__class__.__name__}: {e}"
        ) from e

    wall_end = unix_ms()

    # Assemble output
    from ..output import assemble_output

    text = assemble_output(events, reset_on_tool=True, fallback_to_tool=True)

    resp: Response[Any] = Response(
        session_id=session_id or "",
        input=request.prompt,
        output=text,
        events=events,
        time=TimeInfo(start=wall_start, end=wall_end),
    )

    resp.usage = compute_usage(events)
    return resp


__all__ = ["async_invoke_via_server"]
