"""Agent configuration and hashing for OpenCode."""

from __future__ import annotations

import base64
import hashlib
import json
from typing import Any

from .permissions import Permission


def agent_hash(config: dict[str, Any]) -> str:
    """Generate deterministic agent name from config.

    Uses MD5 hash of canonical JSON (indent=2, sort_keys=True).
    Returns URL-safe base64 string with 'il-' prefix.

    Args:
        config: Agent configuration dictionary

    Returns:
        Agent name like 'il-a7f3b9c1d2e4f5'

    Example:
        >>> config = {"model": "anthropic/claude-sonnet-4-5", "tools": {"write": True}}
        >>> agent_hash(config)
        'il-a7f3b9c1d2e4f5a6'
    """
    canonical = json.dumps(config, indent=2, sort_keys=True)
    md5 = hashlib.md5(canonical.encode()).digest()
    b64 = base64.urlsafe_b64encode(md5).decode().rstrip("=")
    return f"il-{b64[:16]}"


def build_agent_config(
    model: str,
    permission: Permission,
    mcp: dict[str, Any] | None = None,
    providers: dict[str, Any] | None = None,
) -> dict[str, Any]:
    """Build OpenCode agent config from InnerLoop params.

    Args:
        model: Model string (e.g., "anthropic/claude-sonnet-4-5")
        permission: Permission object
        mcp: Optional MCP server definitions
        providers: Optional provider configurations

    Returns:
        OpenCode agent config dictionary with provider config if needed
    """
    # Build tools map from permissions
    tools: dict[str, bool] = {
        "write": permission.edit == Permission.ALLOW,
        "edit": permission.edit == Permission.ALLOW,
        "bash": (
            permission.bash != Permission.DENY
            if not isinstance(permission.bash, dict)
            else True
        ),
    }

    # Add MCP tools if provided
    if mcp:
        for server_name in mcp.keys():
            tools[f"{server_name}_*"] = True

    config: dict[str, Any] = {
        "model": model,
        "tools": tools,
        "permission": permission.to_dict(),
    }

    return config


def build_opencode_config(
    model: str,
    permission: Permission,
    mcp: dict[str, Any] | None = None,
    providers: dict[str, Any] | None = None,
) -> dict[str, Any]:
    """Build complete OpenCode config with agent + providers.

    This is what gets upserted to the server or written to opencode.json.

    Args:
        model: Model string
        permission: Permission object
        mcp: Optional MCP server definitions
        providers: Optional provider configurations

    Returns:
        Complete OpenCode config with schema, agent, and provider
    """
    agent_cfg = build_agent_config(model, permission, mcp, providers)
    agent_name = agent_hash(agent_cfg)

    config: dict[str, Any] = {
        "$schema": "https://opencode.ai/config.json",
        "agent": {agent_name: agent_cfg},
    }

    # Add providers if specified
    if providers:
        config["provider"] = providers

    # Add MCP if specified
    if mcp:
        config["mcp"] = mcp

    return config


__all__ = ["agent_hash", "build_agent_config"]
