"""Schema management commands."""

from __future__ import annotations

import typer

from ..schemas import delete_schema, list_schemas, load_schema

schema_app = typer.Typer(name="schema", help="Manage SlimSchema definitions")


@schema_app.command()
def list() -> None:
    """List all available schemas."""
    schemas = list_schemas()

    if not schemas:
        typer.echo("No schemas found")
        typer.echo(
            "Save a schema with: innerloop schema save <name> <module>:<class>"
        )
        return

    typer.echo(f"Found {len(schemas)} schema(s):")
    for name in schemas:
        typer.echo(f"  {name}")


@schema_app.command()
def show(name: str) -> None:
    """Show schema YAML."""
    try:
        content = load_schema(name)
        typer.echo(content)
    except FileNotFoundError:
        typer.echo(f"✗ Schema not found: {name}", err=True)
        raise typer.Exit(1) from None


@schema_app.command()
def delete(
    name: str,
    force: bool = typer.Option(
        False, "--force", "-f", help="Skip confirmation"
    ),
) -> None:
    """Delete a schema."""
    if not force:
        confirm = typer.confirm(f"Delete schema '{name}'?")
        if not confirm:
            typer.echo("Cancelled")
            raise typer.Exit(0)

    if delete_schema(name):
        typer.echo(f"✓ Deleted schema: {name}")
    else:
        typer.echo(f"✗ Schema not found: {name}", err=True)
        raise typer.Exit(1)


@schema_app.command()
def save(
    name: str,
    model_ref: str = typer.Argument(
        ..., help="Python reference: module.path:ClassName"
    ),
) -> None:
    """Save Pydantic model as SlimSchema YAML.

    Example:
        innerloop schema save person myapp.models:Person
    """
    # Parse module:class reference
    if ":" not in model_ref:
        typer.echo(
            "✗ Invalid reference format. Use: module.path:ClassName", err=True
        )
        raise typer.Exit(1) from None

    module_path, class_name = model_ref.split(":", 1)

    # Import the model
    try:
        import importlib

        module = importlib.import_module(module_path)
        model_class = getattr(module, class_name)
    except (ImportError, AttributeError) as e:
        typer.echo(f"✗ Failed to import {model_ref}: {e}", err=True)
        raise typer.Exit(1) from e

    # Save schema
    from ..schemas import save_schema as _save

    try:
        path = _save(name, model_class)
        typer.echo(f"✓ Saved schema: {name}")
        typer.echo(f"  File: {path}")
    except Exception as e:
        typer.echo(f"✗ Failed to save schema: {e}", err=True)
        raise typer.Exit(1) from e


__all__ = ["schema_app"]
