"""OpenCode server lifecycle management."""

from __future__ import annotations

import os
import signal
import subprocess
from typing import Any

from .client import OpenCodeClient


def start_server(
    port: int = 4096,
    hostname: str = "127.0.0.1",
    background: bool = True,
) -> subprocess.Popen[bytes] | None:
    """Start OpenCode server as background process.

    Args:
        port: Server port
        hostname: Server hostname
        background: Run in background (default: True)

    Returns:
        Process handle if background=True, None otherwise
    """
    from ..home import get_home

    # Set OPENCODE_CONFIG to BotAssembly OpenCode config
    # Auth comes from global ~/.local/share/opencode/auth.json
    home = get_home()
    config_file = home / "opencode" / "opencode.json"

    # Ensure config file exists (create minimal if missing)
    if not config_file.exists():
        config_file.parent.mkdir(parents=True, exist_ok=True)
        config_file.write_text(
            '{"$schema": "https://opencode.ai/config.json"}\n'
        )

    env = os.environ.copy()
    env["OPENCODE_CONFIG"] = str(config_file)

    cmd = [
        "opencode",
        "serve",
        "--port",
        str(port),
        "--hostname",
        hostname,
    ]

    if background:
        # Start in background, redirect output
        proc = subprocess.Popen(
            cmd,
            env=env,
            stdout=subprocess.DEVNULL,
            stderr=subprocess.DEVNULL,
            start_new_session=True,
        )
        return proc

    # Foreground mode
    subprocess.run(cmd, env=env, check=True)
    return None


def stop_server(url: str = "http://127.0.0.1:4096") -> bool:
    """Stop running OpenCode server.

    Args:
        url: Server URL

    Returns:
        True if server was stopped, False if not running
    """
    client = OpenCodeClient(url)

    if not client.is_available():
        return False

    # Find and kill the opencode serve process
    try:
        result = subprocess.run(
            ["pgrep", "-f", "opencode serve"],
            capture_output=True,
            text=True,
            check=False,
        )
        if result.returncode == 0 and result.stdout.strip():
            pid = int(result.stdout.strip().split()[0])
            os.kill(pid, signal.SIGTERM)
            return True
    except (ProcessLookupError, ValueError):
        # Process already dead or invalid PID
        pass

    return False


def get_status(url: str = "http://127.0.0.1:4096") -> dict[str, Any]:
    """Check if server is running.

    Args:
        url: Server URL to check

    Returns:
        Dict with: running (bool), url (str), agents (list | None)
    """
    client = OpenCodeClient(url)

    if not client.is_available():
        return {"running": False, "url": url, "agents": None}

    agents = client.list_agents()
    return {"running": True, "url": url, "agents": agents}


__all__ = ["start_server", "stop_server", "get_status"]
